<?php

declare(strict_types=1);

namespace Quote;

use Spatie\Snapshots\MatchesSnapshots;

use Mage;
use Varien_Object;

use PHPUnit\Framework\TestCase;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Config;
use Crossroads\Magento\Test\Integration\Request;

class AddressTest extends TestCase {
    use MatchesSnapshots;

    const EXCLUDE_ADDRESS_DATA = [
        "address_id" => true,
        "quote_id" => true,
        "created_at" => true,
        "updated_at" => true,
    ];

    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
        Config::unsetConfigPath("default/checkout/options/allow_placeholder_address");
    }

    public function onNotSuccessfulTest($e): void {
        MagentoManager::logQueries();
        Config::unsetConfigPath("default/checkout/options/allow_placeholder_address");

        throw $e;
    }

    public function testBillingMissingAddressParameter() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                setQuoteBillingAddress {
                    result
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(400, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testBillingEmptyAddressAndQuote() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                setQuoteBillingAddress(address: {}) {
                    result
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testBillingEmptyAddressAndQuoteWithValidation() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                setQuoteBillingAddress(address: {}) {
                    result
                    validationErrors
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testBillingEmptyQuoteWithValidation() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                setQuoteBillingAddress(address: {
                    firstname: "Test",
                    lastname: "Test",
                    street: ["Test"],
                    city: "Test",
                    postcode: "12345",
                    telephone: "1234567890",
                }) {
                    result
                    validationErrors
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testBilling() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                setQuoteBillingAddress(address: {
                    prefix: "Testprefix",
                    firstname: "Test",
                    middlename: "Testmiddlename",
                    lastname: "Testsson",
                    suffix: "Testsuffix",
                    company: "Testing",
                    street: ["TestStreet", "Street Row 2"],
                    city: "Test",
                    postcode: "12345",
                    countryCode: "NO",
                    telephone: "1234567890",
                }) {
                    result
                }
            }
        '));

        $quoteId = Mage::getSingleton("checkout/session")->getQuote()->getId();

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                addresses {
                    type
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    company
                    street
                    city
                    postcode
                    region {
                        code
                        name
                    }
                    country {
                        code
                    }
                    ... on QuoteAddressBilling {
                        isUsedAsShipping
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testShippingMissingAddressParameter() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                setQuoteShippingAddress {
                    result
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(400, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testShippingEmptyAddressAndQuote() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                setQuoteShippingAddress(address: {}) {
                    result
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testShippingEmptyAddressAndQuoteWithValidation() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                setQuoteShippingAddress(address: {}) {
                    result
                    validationErrors
                }
            }
        '));

        $quoteId = Mage::getSingleton("checkout/session")->getQuote()->getId();

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                addresses {
                    type
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    company
                    street
                    city
                    postcode
                    region {
                        code
                        name
                    }
                    country {
                        code
                    }
                    ... on QuoteAddressBilling {
                        isUsedAsShipping
                    }
                }
                items {
                    product {
                        sku
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testShippingEmptyQuoteWithValidation() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                setQuoteShippingAddress(address: {
                    firstname: "Test",
                    lastname: "Test",
                    street: ["Test"],
                    city: "Test",
                    postcode: "12345",
                    telephone: "1234567890",
                }) {
                    result
                    validationErrors
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuote()->getId();

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                addresses {
                    type
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    company
                    street
                    city
                    postcode
                    region {
                        code
                        name
                    }
                    country {
                        code
                    }
                    ... on QuoteAddressBilling {
                        isUsedAsShipping
                    }
                }
                items {
                    product {
                        sku
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testShipping() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                setQuoteShippingAddress(address: {
                    prefix: "Testprefix",
                    firstname: "Test",
                    middlename: "Testmiddlename",
                    lastname: "Testsson",
                    suffix: "Testsuffix",
                    company: "Testing",
                    street: ["TestStreet", "Street Row 2"],
                    city: "Test",
                    postcode: "12345",
                    countryCode: "NO",
                    telephone: "1234567890",
                }) {
                    result
                }
            }
        '));

        $quoteId = Mage::getSingleton("checkout/session")->getQuote()->getId();

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                addresses {
                    type
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    company
                    street
                    city
                    postcode
                    region {
                        code
                        name
                    }
                    country {
                        code
                    }
                    ... on QuoteAddressBilling {
                        isUsedAsShipping
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testWithProducts() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($product, $request);
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                setQuoteShippingAddress(address: {
                    prefix: "Testprefix",
                    firstname: "Test",
                    middlename: "Testmiddlename",
                    lastname: "Testsson",
                    suffix: "Testsuffix",
                    company: "Testing",
                    street: ["TestStreet", "Street Row 2"],
                    city: "Test",
                    postcode: "12345",
                    countryCode: "NO",
                    telephone: "1234567890",
                }) {
                    result
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                addresses {
                    type
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    company
                    street
                    city
                    postcode
                    region {
                        code
                        name
                    }
                    country {
                        code
                    }
                    ... on QuoteAddressBilling {
                        isUsedAsShipping
                    }
                }
                items {
                    qty
                    product {
                        sku
                    }
                }
            }
            shippingMethods {
                code
                price {
                    incVat
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testStreetTrim() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                setQuoteShippingAddress(address: {
                    street: ["TestStreet  "],
                }) {
                    result
                }
            }
        '));

        $quoteId = Mage::getSingleton("checkout/session")->getQuote()->getId();

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                addresses {
                    street
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                setQuoteShippingAddress(address: {
                    street: ["TestStreet", ""],
                }) {
                    result
                }
            }
        '));

        $quoteId = Mage::getSingleton("checkout/session")->getQuote()->getId();

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                addresses {
                    street
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                setQuoteShippingAddress(address: {
                    street: ["", "TestStreet"],
                }) {
                    result
                }
            }
        '));

        $quoteId = Mage::getSingleton("checkout/session")->getQuote()->getId();

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                addresses {
                    street
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetBillingAsShippingEmpty() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                setQuoteBillingAddressAsShippingAddress {
                    result
                    address {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        isUsedAsShipping
                    }
                }
            }
        '));

        $quoteId = Mage::getSingleton("checkout/session")->getQuote()->getId();

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                addresses {
                    type
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    company
                    street
                    city
                    postcode
                    region {
                        code
                        name
                    }
                    country {
                        code
                    }
                    ... on QuoteAddressBilling {
                        isUsedAsShipping
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetBillingAsShippingSimple() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($product, $request);
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                setQuoteBillingAddressAsShippingAddress {
                    result
                    address {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        isUsedAsShipping
                    }
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                addresses {
                    type
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    company
                    street
                    city
                    postcode
                    region {
                        code
                        name
                    }
                    country {
                        code
                    }
                    ... on QuoteAddressBilling {
                        isUsedAsShipping
                    }
                }
                items {
                    product {
                        sku
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetBillingAsShippingVirtual() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($product, $request);
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                setQuoteBillingAddressAsShippingAddress {
                    result
                    address {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        isUsedAsShipping
                    }
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                addresses {
                    type
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    company
                    street
                    city
                    postcode
                    region {
                        code
                        name
                    }
                    country {
                        code
                    }
                    ... on QuoteAddressBilling {
                        isUsedAsShipping
                    }
                }
                items {
                    product {
                        sku
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetBillingAsShippingWithData() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "prefix" => "The",
            "firstname" => "Test",
            "middlename" => "Middlename",
            "lastname" => "Testsson",
            "suffix" => "Sr",
            "company" => "The Company",
            "street" => "Street",
            "city" => "Testing",
            "postcode" => "1234567890",
            "country_id" => "SE",
            "telephone" => "0987654321",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
            "same_as_billing" => false,
        ]));
        $quote->addProduct($product, $request);
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                setQuoteBillingAddressAsShippingAddress {
                    result
                    address {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        isUsedAsShipping
                    }
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                addresses {
                    type
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    company
                    street
                    city
                    postcode
                    region {
                        code
                        name
                    }
                    country {
                        code
                    }
                    ... on QuoteAddressBilling {
                        isUsedAsShipping
                    }
                }
                items {
                    product {
                        sku
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quote = Mage::getModel("sales/quote")->load($quoteId);

        $this->assertMatchesJsonSnapshot(array_diff_key($quote->getShippingAddress()->getData(), self::EXCLUDE_ADDRESS_DATA));
    }

    public function testSetBillingAsShippingAndThenData() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                setQuoteBillingAddressAsShippingAddress {
                    result
                }
            }
        '));

        $quoteId = Mage::getSingleton("checkout/session")->getQuote()->getId();

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuoteBillingAddress(address: {
                prefix: "Testprefix",
                firstname: "Test",
                middlename: "Testmiddlename",
                lastname: "Testsson",
                suffix: "Testsuffix",
                company: "Testing",
                street: ["TestStreet", "Street Row 2"],
                city: "Test",
                postcode: "12345",
                countryCode: "NO",
                telephone: "1234567890",
            }) {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                addresses {
                    type
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    company
                    street
                    city
                    postcode
                    region {
                        code
                        name
                    }
                    country {
                        code
                    }
                    ... on QuoteAddressBilling {
                        isUsedAsShipping
                    }
                }
                items {
                    product {
                        sku
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quote = Mage::getModel("sales/quote")->load($quoteId);

        $this->assertMatchesJsonSnapshot(array_diff_key($quote->getShippingAddress()->getData(), self::EXCLUDE_ADDRESS_DATA));
    }

    public function testSetBillingAsShippingAndThenSetShippingAddress() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                setQuoteBillingAddressAsShippingAddress {
                    result
                }
            }
        '));

        $quoteId = Mage::getSingleton("checkout/session")->getQuote()->getId();

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuoteBillingAddress(address: {
                prefix: "Testprefix",
                firstname: "Test",
                middlename: "Testmiddlename",
                lastname: "Testsson",
                suffix: "Testsuffix",
                company: "Testing",
                street: ["TestStreet", "Street Row 2"],
                city: "Test",
                postcode: "12345",
                countryCode: "NO",
                telephone: "1234567890",
            }) {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quote = Mage::getModel("sales/quote")->load($quoteId);

        $this->assertMatchesJsonSnapshot(array_diff_key($quote->getBillingAddress()->getData(), self::EXCLUDE_ADDRESS_DATA));
        $this->assertMatchesJsonSnapshot(array_diff_key($quote->getShippingAddress()->getData(), self::EXCLUDE_ADDRESS_DATA));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuoteShippingAddress(address: {
                prefix: "Shipfix",
                firstname: "Ship",
                middlename: "Mc",
                lastname: "Shipface",
                suffix: "Jr",
                company: "Shipping CO",
                street: ["Ship here"],
                city: "Dock",
                postcode: "31523",
                countryCode: "FI",
                telephone: "09123745",
            }) {
                result
                address {
                    type
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    company
                    street
                    city
                    postcode
                    region {
                        code
                        name
                    }
                    country {
                        code
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quote = Mage::getModel("sales/quote")->load($quoteId);

        $this->assertMatchesJsonSnapshot(array_diff_key($quote->getBillingAddress()->getData(), self::EXCLUDE_ADDRESS_DATA));
        $this->assertMatchesJsonSnapshot(array_diff_key($quote->getShippingAddress()->getData(), self::EXCLUDE_ADDRESS_DATA));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                addresses {
                    type
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    company
                    street
                    city
                    postcode
                    region {
                        code
                        name
                    }
                    country {
                        code
                    }
                    ... on QuoteAddressBilling {
                        isUsedAsShipping
                    }
                }
                items {
                    product {
                        sku
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetQuoteBillingAddressPlaceholder() {
        Config::setConfigPath("default/checkout/options/allow_placeholder_address", "1");
        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($product, $request);
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuoteBillingAddressPlaceholder {
                result
                address {
                    type
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    company
                    street
                    city
                    postcode
                    telephone
                    region {
                        code
                        name
                    }
                    country {
                        code
                    }
                    ... on QuoteAddressBilling {
                        isUsedAsShipping
                    }
                }
                validationErrors
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuote()->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                addresses {
                    type
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    company
                    street
                    city
                    postcode
                    telephone
                    region {
                        code
                        name
                    }
                    country {
                        code
                    }
                    ... on QuoteAddressBilling {
                        isUsedAsShipping
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetQuoteBillingAddressPlaceholderEmpty() {
        Config::setConfigPath("default/checkout/options/allow_placeholder_address", "1");

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuoteBillingAddressPlaceholder {
                result
                address {
                    type
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    company
                    street
                    city
                    postcode
                    telephone
                    region {
                        code
                        name
                    }
                    country {
                        code
                    }
                    ... on QuoteAddressBilling {
                        isUsedAsShipping
                    }
                }
                validationErrors
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuote()->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                addresses {
                    type
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    company
                    street
                    city
                    postcode
                    region {
                        code
                        name
                    }
                    country {
                        code
                    }
                    ... on QuoteAddressBilling {
                        isUsedAsShipping
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetQuoteBillingAddressPlaceholderPhysical() {
        Config::setConfigPath("default/checkout/options/allow_placeholder_address", "1");
        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($product, $request);
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuoteBillingAddressPlaceholder {
                result
                address {
                    type
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    company
                    street
                    city
                    postcode
                    telephone
                    region {
                        code
                        name
                    }
                    country {
                        code
                    }
                    ... on QuoteAddressBilling {
                        isUsedAsShipping
                    }
                }
                validationErrors
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuote()->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                addresses {
                    type
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    company
                    street
                    city
                    postcode
                    telephone
                    region {
                        code
                        name
                    }
                    country {
                        code
                    }
                    ... on QuoteAddressBilling {
                        isUsedAsShipping
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetQuoteBillingAddressPlaceholderExisting() {
        Config::setConfigPath("default/checkout/options/allow_placeholder_address", "1");
        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
            "firstname" => "Test",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
        ]));
        $quote->addProduct($product, $request);
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuoteBillingAddressPlaceholder {
                result
                address {
                    type
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    company
                    street
                    city
                    postcode
                    telephone
                    region {
                        code
                        name
                    }
                    country {
                        code
                    }
                    ... on QuoteAddressBilling {
                        isUsedAsShipping
                    }
                }
                validationErrors
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuote()->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                addresses {
                    type
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    company
                    street
                    city
                    postcode
                    telephone
                    region {
                        code
                        name
                    }
                    country {
                        code
                    }
                    ... on QuoteAddressBilling {
                        isUsedAsShipping
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetQuoteBillingAddressPlaceholderDisabled() {
        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($product, $request);
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuoteBillingAddressPlaceholder {
                result
                address {
                    type
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    company
                    street
                    city
                    postcode
                    telephone
                    region {
                        code
                        name
                    }
                    country {
                        code
                    }
                    ... on QuoteAddressBilling {
                        isUsedAsShipping
                    }
                }
                validationErrors
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }
}
