<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

use function MageQL\snakeCaseToCamel;

class MageQL_Sales_Model_Quote_Item extends Mage_Core_Model_Abstract {
    const SUCCESS = "success";
    const ERROR_PRODUCT_NOT_FOUND = "errorProductNotFound";
    const ERROR_PRODUCT_NOT_IN_STOCK = "errorProductNotInStock";
    const ERROR_PRODUCT_QUANTITY_NOT_AVAILABLE = "errorProductQuantityNotAvailable";
    const ERROR_PRODUCT_MAX_QUANTITY = "errorProductMaxQuantity";
    const ERROR_PRODUCT_VARIANT_NOT_AVAILABLE = "errorProductVariantNotAvailable";
    const ERROR_PRODUCT_REQUIRES_BUNDLE_OPTIONS = "errorProductRequiresBundleOptions";
    const ERROR_BUNDLE_SINGLE_GOT_MULTI = "errorBundleSingleGotMulti";
    const ERROR_ITEM_NOT_FOUND = "errorQuoteItemNotFound";
    const ERROR_BUNDLE_EMPTY = "errorBundleEmpty";
    const ERROR_BUNDLE_SELECTION_QTY_IMMUTABLE = "errorBundleSelectionQtyImmutable";
    const ERROR_DECODE = "errorInvalidBuyRequest";

    public static function resolveBuyRequest(Mage_Sales_Model_Quote_Item $item): string {
        $request = [
            "i" => (int) $item->getId(),
            "p" => (int) $item->getProductId(),
        ];

        return json_encode($request);
    }

    public static function resolveCanOrder(Mage_Sales_Model_Quote_Item $item): bool {
        /**
         * @var ?Mage_CatalogInventory_Model_Stock_Item|Varien_Object
         */
        $stockItem = $item->getProduct()->getStockItem();
        $qty = $item->getQty();
        /**
         * @var ?Mage_CatalogInventory_Model_Stock_Item|Varien_Object
         */
        $parentStockItem = null;
        /**
         * @var ?Mage_Sales_Model_Quote_Item
         */
        $parentItem = null;

        if( ! $stockItem) {
            // Assume in stock
            return true;
        }

        if($item->getParentItem()) {
            $parentItem = $item->getParentItem();
            $parentStockItem = $parentItem->getProduct()->getStockItem();
        }

        if( ! $stockItem->getIsInStock() || $parentStockItem && ! $parentStockItem->getIsInStock()) {
            return false;
        }

        $options = $item->getQtyOptions();

        if($options) {
            foreach($options as $option) {
                $optionValue = $option->getValue();
                $optionQty = $qty * $optionValue;

                $optStockItem = $option->getProduct()->getStockItem();

                $optStockItem->setIsChildItem($parentStockItem ? true : false);
                $optStockItem->setSuppressCheckQtyIncrements(true);

                // FIXME: The second optionQty needs to the the cumulative sum
                // of the options of this type in the quote, see
                // Mage_CatalogInventory_Model_Observer::checkQuoteItemQty
                if($optStockItem->checkQuoteItemQty($optionQty, $optionQty, $optionValue)->getHasError()) {
                    return false;
                }

                $optStockItem->unsIsChildItem();
            }
        }
        else {
            $rowQty = $parentItem ? $parentItem->getQty() * $qty : $qty;

            // FIXME: The second rowQty needs to the the cumulative sum
            // of the items of this type in the quote, see
            // Mage_CatalogInventory_Model_Observer::checkQuoteItemQty
            if($stockItem->checkQuoteItemQty($rowQty, $rowQty, $qty)->getHasError()) {
                return false;
            }
        }

        return true;
    }

    /**
     * @param mixed $unusedSrc
     * @param array{buyRequest:string, qty:float, selectionId?: Array<array {
     *   optionId: string, qty?: float, selectionId?: string
     * }>} $args
     * @return Mage_Sales_Model_Quote_Item|string
     */
    public static function mutateAdd(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx
    ) {
        $model = Mage::getSingleton("mageql_sales/quote");
        $qty = (float) $args["qty"];
        $request = json_decode($args["buyRequest"], true, 3);

        if(json_last_error()) {
            return self::ERROR_DECODE;
        }

        $productId = (int)($request["p"] ?? 0);
        $product = Mage::getModel("catalog/product")
            ->setStoreId($ctx->getStore()->getId())
            ->load($productId);

        if( ! Mage::helper("mageql_catalog")->isProductVisible($product, $ctx->getStore())) {
            return self::ERROR_PRODUCT_NOT_FOUND;
        }

        // TODO: Deal with custom options (probably additional argument?)
        $buyRequest = [
            "qty" => self::clampProductQty($model->getQuote(), $product, $qty),
        ];

        switch($product->getTypeId()) {
        case "bundle":
            $bundleOptions = [];
            $bundleOptionQty = [];
            /**
             * @var Mage_Bundle_Model_Product_Type
             */
            $instance = $product->getTypeInstance(true);
            $options = $instance->getOptions($product);

            foreach($options as $opt) {
                $selections = MageQL_Sales_Model_Product_Buyrequest_Bundle::getOptionSelections(
                    $opt,
                    $ctx->getStore()
                );
                $qty = null;
                $selected = [];

                foreach($args["bundleOptions"] ?? [] as $userSelection) {
                    if($userSelection["optionId"] === $opt->getId()) {
                        if (empty($userSelection["selectionId"])) {
                            $qty = null;
                            $selected = null;
                        }
                        else {
                            foreach($selections as $s) {
                                if($s->getSelectionId() === $userSelection["selectionId"]) {
                                    $selected = $selected ?: [];
                                    $selected[] = $s->getSelectionId();
                                    $qty = $s->getSelectionQty();

                                    if( ! empty($userSelection["qty"])) {
                                        if( ! $s->getSelectionCanChangeQty() && $userSelection["qty"] != $qty) {
                                            return self::ERROR_BUNDLE_SELECTION_QTY_IMMUTABLE;
                                        }

                                        $qty = $userSelection["qty"];
                                    }

                                    continue 2;
                                }
                            }
                        }
                    }
                }

                // We fill in the defaults after if we have nothing
                if(empty($selected) && $selected !== null) {
                    foreach($selections as $sel) {
                        if($sel->getIsDefault()) {
                            $selected[] = $sel->getSelectionId();
                            $qty = $sel->getSelectionQty();
                        }
                    }
                }

                if( ! empty($selected) && ! empty($qty)) {
                    if($opt->isMultiSelection()) {
                        $bundleOptions[$opt->getId()] = $selected;
                    }
                    else {
                        if (count($selected) > 1) {
                            return self::ERROR_BUNDLE_SINGLE_GOT_MULTI;
                        }

                        $bundleOptions[$opt->getId()] = $selected[0];
                    }

                    $bundleOptionQty[$opt->getId()] = $qty;
                }
                else if($opt->getRequired()) {
                    return self::ERROR_PRODUCT_REQUIRES_BUNDLE_OPTIONS;
                }
            }

            if(empty($bundleOptions)) {
                return self::ERROR_BUNDLE_EMPTY;
            }

            $buyRequest["bundle_option"] = $bundleOptions;
            $buyRequest["bundle_option_qty"] = $bundleOptionQty;

            break;

        case "configurable":
            if(array_key_exists("a", $request)) {
                // TODO: Validate?
                $buyRequest["super_attribute"] = $request["a"];
            }

            break;
        case "simple":
        case "virtual":
            break;
        }

        try {
            return $model->addProduct($product, $buyRequest);
        }
        catch(Mage_Core_Exception $e) {
            return self::translateAddProductError($e->getMessage(), $product, $e);
        }
    }

    /**
     * @param mixed $unusedSrc
     * @param array{itemBuyRequest:string, qty:float} $args
     * @return Mage_Sales_Model_Quote_Item|string
     */
    public static function mutateUpdateQty(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx
    ) {
        $model = Mage::getSingleton("mageql_sales/quote");
        $qty = (float) $args["qty"];
        $request = json_decode($args["itemBuyRequest"], true, 3);

        if(json_last_error()) {
            return self::ERROR_DECODE;
        }

        $itemId = (int)($request["i"] ?? 0);
        $productId = (int)($request["p"] ?? 0);

        if( ! $itemId) {
            // TODO: Different error sicne this is a buyRequest
            return self::ERROR_DECODE;
        }

        $quote = $model->getQuote();
        $product = Mage::getModel("catalog/product")
            ->setStoreId($ctx->getStore()->getId())
            ->load($productId);

        if( ! Mage::helper("mageql_catalog")->isProductVisible($product, $ctx->getStore())) {
            return self::ERROR_PRODUCT_NOT_FOUND;
        }

        /**
         * @var ?Mage_Sales_Model_Quote_Item
         */
        $item = $quote->getItemById($request["i"]);

        if( ! $item) {
            return self::ERROR_ITEM_NOT_FOUND;
        }

        // TOOD: Custom options
        // Join with the existing buy-request to just update the qty
        $buyRequest = array_merge($item->getBuyRequest()->getData(), [
            "product" => $product->getId(),
            "qty" => self::clampProductQty($quote, $product, $qty),
            // Needed to prevent magento from attempting to fetch the original
            // qty + the new qty if we do not have id here the old quote item
            // is considered different and will be first summed with this
            // request and then deleted after the validation has been performed
            // It also needs to be a string to pass === in Mage_Sales_Model_Quote
            // If it is a different item,
            "id" => (string)$itemId,
        ]);

        try {
            return $model->updateItem($item, $product, $buyRequest);
        }
        catch(Mage_Core_Exception $e) {
            return self::translateAddProductError($e->getMessage(), $product, $e);
        }
    }

    /**
     * @param mixed $unusedSrc
     * @param array{itemBuyRequest:string} $args
     */
    public static function mutateRemove($unusedSrc, array $args): string {
        $model = Mage::getSingleton("mageql_sales/quote");
        $request = json_decode($args["itemBuyRequest"], true, 3);

        if(json_last_error() || ! ($request["i"] ?? 0)) {
            return self::ERROR_DECODE;
        }

        $quote = $model->getQuote();
        /**
         * @var ?Mage_Sales_Model_Quote_Item
         */
        $item = $quote->getItemById($request["i"] ?? 0);

        if( ! $item || ! $item->getId()) {
            return self::ERROR_ITEM_NOT_FOUND;
        }

        $quote->removeItem($item->getId());

        // Recollect shipping rates since we have removed an item
        $model->saveSessionQuoteWithShippingRates();

        return self::SUCCESS;
    }

    /**
     * Clamps the product quantity if it is not a configurable product.
     *
     * @param float $qty
     */
    public static function clampProductQty(
        Mage_Sales_Model_Quote $quote,
        Mage_Catalog_Model_Product $product,
        $qty
    ): float {
        if($product->isConfigurable() ||
           !$product->getStockItem() ||
           $quote->hasProductId($product->getId())) {
            return $qty;
        }

        return max($product->getStockItem()->getMinSaleQty(), $qty);
    }

    public static function translateAddProductError(
        string $errorMsg,
        Mage_Catalog_Model_Product $product,
        Exception $prev = null
    ): string {
        $error = explode("\n", $errorMsg)[0];

        // TODO: Add option errors if bundle-products need it
        // Option errors cannot happen since we construct the attributes on the server

        if($error === Mage::helper("checkout")->__("The product could not be found.")) {
            return self::ERROR_PRODUCT_NOT_FOUND;
        }

        if($error === Mage::helper('cataloginventory')->__('This product is currently out of stock.')) {
            return self::ERROR_PRODUCT_NOT_IN_STOCK;
        }

        if(stripos($error, Mage::helper('sales')->__('Item qty declaration error.')) !== false ||
           stripos($error, Mage::helper('cataloginventory')->__('Item qty declaration error.\nThe requested quantity for \"%s\" is not available.', $product->getName())) !== false ||
           stripos($error, Mage::helper('cataloginventory')->__('The requested quantity for "%s" is not available.', $product->getName())) !== false) {
           return self::ERROR_PRODUCT_QUANTITY_NOT_AVAILABLE;
        }

        if(stripos($error, strstr(Mage::helper('cataloginventory')->__('The maximum quantity allowed for purchase is %s.', "DEADBEEF"), "DEADBEEF", true) ?: "DEADBEEF") !== false) {
           return self::ERROR_PRODUCT_MAX_QUANTITY;
        }

        if($error === Mage::helper('cataloginventory')->__('This product with current option is not available')) {
            return self::ERROR_PRODUCT_VARIANT_NOT_AVAILABLE;
        }

        throw new Exception($errorMsg, 0, $prev);
    }
}
