<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

use MageQL\Registry;
use MageQL\Type\AbstractBuilder;
use MageQL\Type\FieldBuilder;

abstract class MageQL_Sales_Model_Payment_Abstract extends MageQL_Core_Model_Schema_Abstract {
    /**
     * Payment method was set successfully.
     */
    const SUCCESS = "success";
    const ERROR_NOT_AVAILABLE = "errorNotAvailable";
    const ERROR_NOT_ALLOWED_FOR_BILLING_COUNTRY = "errorCountryNotAllowed";

    /**
     * Returns the payment method code.
     */
    abstract public function getCode(): string;

    /**
     * Returns the type name for the payment type.
     */
    abstract public function getMethodTypeName(): string;

    /**
     * Lists the extra fields available on the enabled payment method data.
     *
     * @return Array<FieldBuilder>
     */
    abstract public function getPaymentMethodFields(Registry $registry): array;

    /**
     * Lists the extra fields available on the selected quote payment method data.
     *
     * @return Array<FieldBuilder>
     */
    abstract public function getQuotePaymentFields(Registry $registry): array;

    /**
     * @return Array<FieldBuilder>
     */
    abstract public function getMutateSetQuotePaymentFields(Registry $registry): array;

    /**
     * @return string
     */
    abstract public function mutateSetPaymentMethod(Mage_Sales_Model_Quote $quote, array $args, MageQL_Core_Model_Context $ctx, ResolveInfo $info);

    /**
     * Returns a list of errors which can be caught when placing orders,
     * override in child class to provide abaility for payment to catch
     * erorrs.
     *
     * The key is the result-enum-value and `apiErrorCode` is the exception
     * error code for `Crossroads_API_ResponseException` if any.
     *
     * @return Array<string, array{apiErrorCode?:int, description:string}>
     */
    public function getPlaceOrderErrors(): array {
        return [];
    }

    /**
     * Translates a given magento payment exception into an error code,
     * extend to handle payment-specific errors.
     *
     * Should return one of the error codes returned from `getMutationResultErrors`.
     */
    public function translatePaymentError(Mage_Core_Exception $e): string {
        if($e->getMessage() === Mage::helper("sales")->__("The requested Payment Method is not available.")) {
            return self::ERROR_NOT_AVAILABLE;
        }

        if($e->getMessage() === Mage::helper("payment")->__("Selected payment type is not allowed for billing country.")) {
            return self::ERROR_NOT_ALLOWED_FOR_BILLING_COUNTRY;
        }

        throw $e;
    }

    /**
     * Returns a map of enum-value => enum-config for all errors the setQuotePaymentMethod mutation can result in.
     *
     * Extend and add additional variants supported by translatePaymentError.
     *
     * @return Array<string, array{description:string}>
     */
    public function getMutationResultErrors(): array {
        return [
            self::SUCCESS => [
                "description" => "Successfully set the payment method",
            ],
            self::ERROR_NOT_AVAILABLE => [
                "description" => "This payment method is not available for the supplied quote",
            ],
            self::ERROR_NOT_ALLOWED_FOR_BILLING_COUNTRY => [
                "description" => "This payment method is not allowed for the billing address country",
            ],
        ];
    }

    /**
     * Type name for object containing information about the available payment method.
     */
    public function getPaymentMethodType(): string {
        return "PaymentMethod".$this->getMethodTypeName();
    }

    /**
     * Type name for object containing information about the selected payment method.
     */
    public function getQuotePaymentType(): string {
        return "QuotePayment".$this->getMethodTypeName();
    }

    /**
     * Type name for the result of the quote mutation to set this payment method.
     */
    public function getMutationType(): string {
        return "MutationSetQuotePaymentMethod".$this->getMethodTypeName();
    }

    /**
     * Type name for the result type for the setQuotePaymentMethod mutation.
     */
    public function getMutationResultType(): string {
        return "MutationSetQuotePaymentMethod".$this->getMethodTypeName()."Result";
    }

    /**
     * Creates the FieldBuilder for the payment method, override and modify the
     * return value with extra arguments if necessary.
     *
     * Example:
     *
     *   public function getQuotePaymentMutationFieldBuilder(Registry $registry): AbstractBuilder {
     *       $builder = parent::getQuotePaymentMutationFieldBuilder($registry);
     *
     *       $builder->addArgument($this->argument("Int!", "The cool number required to use this payment method"));
     *
     *       return $builder;
     *   }
     *
     */
    public function getQuotePaymentMutationFieldBuilder(Registry $unusedRegistry): AbstractBuilder {
        $code = $this->getCode();

        return $this->field($this->getMutationType(), "Updates the quote to use the payment method $code.")
            ->setResolver([$this, "mutateSetPaymentMethod"]);
    }

    /**
     * Creates the type builder for the active quote payment information when using this payment method.
     */
    public function getQuotePaymentTypeBuilder(): AbstractBuilder {
        $methodName = $this->getMethodTypeName();

        return $this->object("Selected $methodName payment data")
            ->setInterfaces(["QuotePayment"])
            ->setResolveField("MageQL\\defaultVarienObjectResolver");
    }

    /**
     * Creates the type builder for the payment method information.
     */
    public function getPaymentMethodTypeBuilder(): AbstractBuilder {
        $methodName = $this->getMethodTypeName();

        return $this->object("$methodName payment method data")
            ->setInterfaces(["PaymentMethod"])
            ->setResolveField("MageQL\\defaultVarienObjectResolver");
    }

    public function getMutationBuilder(): AbstractBuilder {
        $methodName = $this->getMethodTypeName();

        return $this->object("The result of attempting to set the quote payment method to $methodName");
    }

    public function getMutationResultBuilder(): AbstractBuilder {
        $methodName = lcfirst($this->getMethodTypeName());

        return $this->enum("Result of a SetQuotePaymentMethod.$methodName mutation", $this->getMutationResultErrors());
    }

    public function getTypeBuilder(string $typeName, Registry $unusedRegistry): ?AbstractBuilder {
        switch($typeName) {
        case $this->getPaymentMethodType():
            return $this->getPaymentMethodTypeBuilder();

        case $this->getQuotePaymentType():
            return $this->getQuotePaymentTypeBuilder();

        case $this->getMutationType():
            return $this->getMutationBuilder();

        case $this->getMutationResultType():
            return $this->getMutationResultBuilder();

        default:
            return null;
        }
    }

    public function getTypeFields(string $typeName, Registry $registry): array {
        switch($typeName) {
        case "MutationSetQuotePaymentMethod":
            return [
                lcfirst($this->getMethodTypeName()) => $this->getQuotePaymentMutationFieldBuilder($registry),
            ];

        case $this->getPaymentMethodType():
            return array_merge(
                $registry->getFieldBuilders("PaymentMethod"),
                $this->getPaymentMethodFields($registry)
            );

        case $this->getQuotePaymentType():
            return array_merge(
                $registry->getFieldBuilders("QuotePayment"),
                $this->getQuotePaymentFields($registry)
            );

        case $this->getMutationType():
            return $this->getMutateSetQuotePaymentFields($registry);

        default:
            return [];
        }
    }

    public function getUnreachableTypes(): array {
        return [
            $this->getQuotePaymentType(),
            $this->getPaymentMethodType(),
        ];
    }
}
