<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

class MageQL_Sales_Model_Order extends Mage_Core_Model_Abstract {
    public static function resolveEmail(Mage_Sales_Model_Order $src): string {
        return $src->getCustomerEmail();
    }

    public static function resolveLast(): ?Mage_Sales_Model_Order {
        $session = Mage::getSingleton("checkout/session");
        $orderId = $session->getLastOrderId();

        if( ! $orderId) {
            return null;
        }

        $order = Mage::getModel("sales/order");
        $order->load($orderId);

        if( ! $order->getId()) {
            return null;
        }

        return $order;
    }

    /**
     * @return Array<Mage_Sales_Model_Order_Address>
     */
    public static function resolveAddresses(
        Mage_Sales_Model_Order $src,
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx,
        ResolveInfo $info
    ): array {
        $addresses = [];

        foreach($src->getAddressesCollection()->getItems() as $addr) {
            if($addr->getAddressType() !== Mage_Customer_Model_Address_Abstract::TYPE_SHIPPING ||
                ! $addr->getSameAsBilling()) {
                $addresses[] = $addr;
            }
        }

        return $addresses;
    }

    public static function resolveId(Mage_Sales_Model_Order $order): string {
        return $order->getIncrementId();
    }

    /**
     * @return Array<Mage_Sales_Model_Order_Item>
     */
    public static function resolveItems(
        Mage_Sales_Model_Order $src,
        array $unusedArgs,
        MageQL_Core_Model_Context $unusedCtx,
        ResolveInfo $info
    ) {
        $config = Mage::getSingleton("mageql_catalog/attributes_product");
        /**
         * 2 levels deep to get attributes (1 item, 2 product, 3 attributes)
         *
         * @var array{product:?array{attributes:Array<string, bool>}}
         */
        $fields = $info->getFieldSelection(3);
        // Merge attributes
        $fields = array_merge($fields["product"] ?? [], $fields["product"]["attributes"] ?? []);

        // We use all attribute sets since we cannot be certain about which sets we will get
        $toSelect = $config->getUsedAttributes(
            $config->getAttributesByArea(MageQL_Catalog_Model_Attributes_Abstract::AREA_LIST),
            $fields
        );

        // Set the attributes so they are loaded when we get the quote items
        Mage::getSingleton("mageql_sales/attributes_product")->setQuoteAttributes($toSelect);

        // Do not use a cache since we want to use the previously set attributes
        // Note that the $useCache flag does not work on quote getItemsCollection
        $collection = Mage::getModel("sales/order_item")->getCollection()->setOrderFilter($src);

        $items = $collection->getItems();

        return array_values(array_filter($items, function(Mage_Sales_Model_Order_Item $item) {
            return ! $item->isDeleted() && ! $item->getParentItemId();
        }));
    }

    public static function resolveCreatedAt(Mage_Sales_Model_Order $order): string {
        return gmdate("Y-m-d\TH:i:s\Z", strtotime($order->getCreatedAt()));
    }

    public static function resolveItemProduct(
        Mage_Sales_Model_Order_Item $item
    ): Mage_Catalog_Model_Product {
        return $item->getProduct();
    }
}
