<?php

declare(strict_types=1);

use Spatie\Snapshots\MatchesSnapshots;

use PHPUnit\Framework\TestCase;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;

class OrderTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        $this->reset();
    }

    public function onNotSuccessfulTest($e): void {
        $this->reset();

        throw $e;
    }

    public function reset(): void {
        MagentoManager::logQueries();
        MagentoManager::reset();
        MagentoManager::initAdmin();

        // Got to reset these
        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");

        $setupModel->setConfigData("carriers/freeshipping/active", 1);
        $setupModel->setConfigData("carriers/freeshipping/free_shipping_subtotal", 100);

        // Reset stock on the product, modified by testPlaceOrderOutOfStock
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        Mage::getModel("cataloginventory/stock_item")
            ->loadByProduct($product)
            ->addData([
                "is_in_stock" => 1,
                "qty" => 999,
            ])
            ->save();

        // we have to reindex the stock to not affect the other tests
        MagentoManager::reset();
        MagentoManager::reindex();
    }

    public function testNoLastOrder() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            lastOrder {
                id
                items {
                    qty
                    rowTotal {
                        incVat
                    }
                    product {
                        sku
                        attributes {
                            shortDescription
                            smallImage
                        }
                    }
                }
                addresses {
                    type
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    company
                    street
                    city
                    postcode
                    region {
                        code
                        name
                    }
                    country {
                        code
                    }
                }
                email
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testLastOrder() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $order = Mage::getModel("sales/order")->loadByIncrementId("TEST-1");
        $orderId = $order->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setLastOrderId($orderId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            lastOrder {
                id
                items {
                    qty
                    rowTotal {
                        incVat
                    }
                    product {
                        sku
                        name
                        attributes {
                            shortDescription
                            smallImage
                        }
                    }
                    ... on OrderItemConfigurable {
                        configOption {
                            attributes {
                                attribute
                                label
                                value
                            }
                            product {
                                sku
                                name
                                attributes {
                                    shortDescription
                                    smallImage
                                }
                            }
                        }
                    }
                    ... on OrderItemBundle {
                        bundleOptions {
                            type
                            title
                            products {
                                qty
                                product {
                                    sku
                                    name
                                    attributes {
                                        shortDescription
                                        smallImage
                                    }
                                }
                            }
                        }
                    }
                }
                addresses {
                    type
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    company
                    street
                    city
                    postcode
                    region {
                        code
                        name
                    }
                    country {
                        code
                    }
                }
                email
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testLastOrderInvalidId() {
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setLastOrderId(1234567890);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            lastOrder {
                id
                items {
                    qty
                    rowTotal {
                        incVat
                    }
                    product {
                        sku
                        attributes {
                            shortDescription
                            smallImage
                        }
                    }
                }
                addresses {
                    type
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    company
                    street
                    city
                    postcode
                    region {
                        code
                        name
                    }
                    country {
                        code
                    }
                }
                email
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testPlaceOrder() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // Decrease so we get a zero subtotal checkout
        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");

        $setupModel->setConfigData("carriers/freeshipping/free_shipping_subtotal", 0);

        MagentoManager::reset();
        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        // Add the product and then modify the item to zero the price
        $quote->addProduct($product, $request)
            ->setOriginalCustomPrice(0)
            ->setCustomPrice(0)
            ->setIsSuperMode(true);

        // It is not a virtual quote, use shipping address for payment
        $quote->getShippingAddress()->setPaymentMethod("free");
        $quote->getPayment()->importData([
            "method" => "free",
        ]);
        $quote->getShippingAddress()->setShippingMethod("freeshipping_freeshipping");
        $quote->setCustomerEmail("test@example.com");

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            placeOrder {
                result
                ...on PlaceOrderResultSuccess {
                    paymentRedirectUrl
                    order {
                        items {
                            qty
                            rowTotal {
                                incVat
                            }
                            product {
                                sku
                                attributes {
                                    shortDescription
                                    smallImage
                                }
                            }
                        }
                        addresses {
                            type
                            prefix
                            firstname
                            middlename
                            lastname
                            suffix
                            company
                            street
                            city
                            postcode
                            region {
                                code
                                name
                            }
                            country {
                                code
                            }
                        }
                        email
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testPlaceOrderInvalidPayment() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        // Add the product and then modify the item to zero the price
        $quote->addProduct($product, $request)
            ->setOriginalCustomPrice(0)
            ->setCustomPrice(0)
            ->setIsSuperMode(true);

        // It is not a virtual quote, use shipping address for payment
        $quote->getShippingAddress()->setPaymentMethod("free");
        $quote->getPayment()->importData([
            "method" => "free",
        ]);
        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");
        $quote->setCustomerEmail("test@example.com");

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            placeOrder {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testPlaceOrderNoItems() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));

        // It is not a virtual quote, use shipping address for payment
        $quote->getShippingAddress()->setPaymentMethod("free");
        $quote->getPayment()->importData([
            "method" => "free",
        ]);
        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");
        $quote->setCustomerEmail("test@example.com");

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            placeOrder {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testPlaceOrderFaultyShipping() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        // Add the product and then modify the item to zero the price
        $quote->addProduct($product, $request)
            ->setOriginalCustomPrice(0)
            ->setCustomPrice(0)
            ->setIsSuperMode(true);

        // It is not a virtual quote, use shipping address for payment
        $quote->getShippingAddress()->setPaymentMethod("free");
        $quote->getPayment()->importData([
            "method" => "free",
        ]);
        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");
        $quote->setCustomerEmail("test@example.com");

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            placeOrder {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testPlaceOrderMissingShippingAddress() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        // Add the product and then modify the item to zero the price
        $quote->addProduct($product, $request)
            ->setOriginalCustomPrice(0)
            ->setCustomPrice(0)
            ->setIsSuperMode(true);

        // It is not a virtual quote, use shipping address for payment
        $quote->getShippingAddress()->setPaymentMethod("free");
        $quote->getPayment()->importData([
            "method" => "free",
        ]);
        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");
        $quote->setCustomerEmail("test@example.com");

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            placeOrder {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testPlaceOrderMissingShipping() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        // Add the product and then modify the item to zero the price
        $quote->addProduct($product, $request)
            ->setOriginalCustomPrice(0)
            ->setCustomPrice(0)
            ->setIsSuperMode(true);

        // It is not a virtual quote, use shipping address for payment
        $quote->getShippingAddress()->setPaymentMethod("free");
        $quote->getPayment()->importData([
            "method" => "free",
        ]);
        $quote->setCustomerEmail("test@example.com");

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            placeOrder {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testPlaceOrderMissingPayment() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        // Add the product and then modify the item to zero the price
        $quote->addProduct($product, $request)
            ->setOriginalCustomPrice(0)
            ->setCustomPrice(0)
            ->setIsSuperMode(true);

        // It is not a virtual quote, use shipping address for payment
        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");
        $quote->setCustomerEmail("test@example.com");

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            placeOrder {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testPlaceOrderInvalidBilling() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        // Add the product and then modify the item to zero the price
        $quote->addProduct($product, $request)
            ->setOriginalCustomPrice(0)
            ->setCustomPrice(0)
            ->setIsSuperMode(true);

        // It is not a virtual quote, use shipping address for payment
        $quote->getShippingAddress()->setPaymentMethod("free");
        $quote->getPayment()->importData([
            "method" => "free",
        ]);
        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");
        $quote->setCustomerEmail("test@example.com");

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            placeOrder {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testPlaceOrderInvalidEmail() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // Decrease so we get a zero subtotal checkout
        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");

        $setupModel->setConfigData("carriers/freeshipping/free_shipping_subtotal", 0);

        MagentoManager::reset();
        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        // Add the product and then modify the item to zero the price
        $quote->addProduct($product, $request)
            ->setOriginalCustomPrice(0)
            ->setCustomPrice(0)
            ->setIsSuperMode(true);

        // It is not a virtual quote, use shipping address for payment
        $quote->getShippingAddress()->setPaymentMethod("free");
        $quote->getPayment()->importData([
            "method" => "free",
        ]);
        $quote->getShippingAddress()->setShippingMethod("freeshipping_freeshipping");
        $quote->setCustomerEmail("testfail");

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            placeOrder {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testPlaceOrderOutOfStock() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // Decrease so we get a zero subtotal checkout
        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");

        $setupModel->setConfigData("carriers/freeshipping/free_shipping_subtotal", 0);

        MagentoManager::reset();
        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        // Add the product and then modify the item to zero the price
        $quote->addProduct($product, $request)
            ->setOriginalCustomPrice(0)
            ->setCustomPrice(0)
            ->setIsSuperMode(true);

        // It is not a virtual quote, use shipping address for payment
        $quote->getShippingAddress()->setPaymentMethod("free");
        $quote->getPayment()->importData([
            "method" => "free",
        ]);
        $quote->getShippingAddress()->setShippingMethod("freeshipping_freeshipping");
        $quote->setCustomerEmail("test@example.com");

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        // Update stock on the product
        Mage::getModel("cataloginventory/stock_item")
            ->loadByProduct($product)
            ->addData([
                "is_in_stock" => 0,
                "qty" => 0,
            ])
            ->save();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            placeOrder {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }
}
