<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

/**
 * @psalm-type Result array{0:string, 1:Mage_Sales_Model_Quote_Address}
 */
class MageQL_Sales_Model_Quote_Address extends Mage_Core_Model_Abstract {
    const NOT_MODIFIED = "notModified";
    const SUCCESS = "success";
    // TODO: define the fields in XML like Magento does?
    const FIELDS = [
        "prefix" => "Prefix",
        "firstname" => "Firstname",
        "middlename" => "Middlename",
        "lastname" => "Lastname",
        "suffix" => "Suffix",
        "company" => "Company",
        "street" => "Street",
        "city" => "City",
        "postcode" => "Postcode",
        "regionCode" => "RegionCode",
        "countryCode" => "CountryId",
        "telephone" => "Telephone",
    ];

    const COPY_FIELDS = [
        "prefix",
        "firstname",
        "middlename",
        "lastname",
        "suffix",
        "company",
        "street",
        "city",
        "postcode",
        "region",
        "region_id",
        "country_id",
        "telephone",
    ];

    /**
     * Fields which are part of an address.
     */
    const ADDRESS_DATA_FIELDS = [
        "prefix",
        "firstname",
        "middlename",
        "lastname",
        "suffix",
        "company",
        "street",
        "city",
        "postcode",
        "region",
        "region_id",
        "telephone",
        "country_id",
    ];

    public static function resolveIsUsedAsShipping(Mage_Customer_Model_Address_Abstract $src): bool {
        if($src->getAddressType() === Mage_Sales_Model_Quote_Address::TYPE_SHIPPING) {
            return true;
        }

        // TODO: Use the quote from $src?
        $shipping = Mage::getSingleton("mageql_sales/quote")->getShippingAddress();

        return $shipping ? (bool)$shipping->getSameAsBilling() : false;
    }

    public static function resolveType(Mage_Customer_Model_Address_Abstract $src): string {
        return $src->getAddressType();
    }

    /**
     * @param Result $src
     */
    public static function resolveSetAddressResultType($src): string {
        // Result enum value is the first parameter
        return $src[0];
    }

    /**
     * @param Result $src
     * @return Array<string>
     */
    public static function resolveSetAddressResultValidationErrors($src): array {
        // The modified address is the second parameter
        return MageQL_Core_Model_Address::resolveValidationErrors($src[1]);
    }

    /**
     * @param Result $src
     */
    public static function resolveSetAddressResultAddress($src): Mage_Sales_Model_Quote_Address {
        // The modified address is the second parameter
        return $src[1];
    }

    /**
     * @param mixed $unusedSrc
     * @return Result
     */
    public static function mutateSetBilling($unusedSrc, array $args) {
        $model = Mage::getSingleton("mageql_sales/quote");
        // We actually want to create it if it does not yet exist
        $billingAddress = $model->makeBillingAddress();
        $shippingAddress = $model->getShippingAddress();
        $addressData = $args["address"];

        // TODO: Load customer-address data here? Or do we make that a separate
        //       mutation, to reset to customer data? Or do we have a parameter
        //       which tells us to merge?

        if(self::setCommonFields($billingAddress, $addressData)) {
            // If we have modified it, also propagate to shipping if it is marked as same
            if($shippingAddress && $shippingAddress->getSameAsBilling()) {
                // Copy the info
                foreach(self::COPY_FIELDS as $field) {
                    $shippingAddress->setData($field, $billingAddress->getData($field));
                }

                // $shippingAddress->setSameAsBilling(0);
            }

            $model->saveSessionQuote();

            return [self::SUCCESS, $billingAddress];
        }

        return [self::NOT_MODIFIED, $billingAddress];
    }

    /**
     * @param mixed $unusedSrc
     * @return Result
     */
    public static function mutateSetShipping($unusedSrc, array $args) {
        $model = Mage::getSingleton("mageql_sales/quote");
        // We actually want to create it if it does not yet exist
        $shippingAddress = $model->makeShippingAddress();
        $addressData = $args["address"];

        // TODO: Load customer-address data here? Or do we make that a separate
        //       mutation, to reset to customer data? Or do we have a parameter
        //       which tells us to merge?

        if(self::setCommonFields($shippingAddress, $addressData)) {
            $shippingAddress->setSameAsBilling(false);

            $model->saveSessionQuoteWithShippingRates();

            return [self::SUCCESS, $shippingAddress];
        }

        return [self::NOT_MODIFIED, $shippingAddress];
    }

    /**
     * @return Result
     */
    public static function mutateUseBillingAsShipping() {
        $model = Mage::getSingleton("mageql_sales/quote");
        $quote = $model->getQuote();
        $billingAddress = $model->makeBillingAddress();
        $shippingAddress = $model->getShippingAddress();
        $shippingMethod = $shippingAddress ? $shippingAddress->getShippingMethod() : null;

        // Create a new address object
        $newAddress = Mage::getModel("sales/quote_address");

        foreach($billingAddress->getData() as $key => $value) {
            if($value !== null && in_array($key, self::COPY_FIELDS)) {
                $newAddress->setData($key, $value);
            }
        }

        $newAddress->setSameAsBilling(true)
            ->setSaveInAddressBook(false)
            ->setCollectShippingRates(true);

        if($shippingMethod) {
            // TODO: Validate?
            $newAddress->setShippingMethod($shippingMethod);
        }

        if($shippingAddress) {
            $quote->removeAddress($shippingAddress);
        }

        $quote->setShippingAddress($newAddress);

        $model->saveSessionQuoteWithShippingRates();

        return [self::SUCCESS, $billingAddress];
    }

    public static function setCommonFields(Mage_Sales_Model_Quote_Address $address, array $data): bool {
        $modified = false;

        foreach(self::FIELDS as $field => $method) {
            $read = "get$method";
            $write = "set$method";

            switch($field) {
            case "street":
                if(array_key_exists($field, $data)) {
                    $newStreet = trim(implode("\n", $data[$field]));
                    $oldStreet = trim(implode("\n", (array)$address->getStreet()));

                    if($newStreet !== $oldStreet) {
                        // Since we have already trimmed it and imploded it we
                        // do no have to call implodeStreetAddress
                        $address->setStreet($newStreet);

                        $modified = true;
                    }
                }

                break;

            // TODO: Validate countryCode
            case "regionCode":
                if(array_key_exists($field, $data) && $data[$field] !== $address->getRegionCode()) {
                    // TODO: Validate
                    $address->setRegionCode($data[$field]);
                    // Reset
                    $address->unsRegionId();
                    $address->unsRegion();

                    $modified = true;
                }

                break;

            default:
                if(array_key_exists($field, $data) && $data[$field] !== $address->$read()) {
                    $address->$write($data[$field]);
                    $modified = true;
                }
            }
        }

        if($modified) {
            // Make sure we do not tie it to a customer address
            $address->setCustomerAddressId(null);
            $address->setSaveInAddressBook(false);
        }

        return $modified;
    }

    /**
     * Checks if the address is empty with respect to store defaults.
     */
    public static function isAddressEmpty(
        Mage_Customer_Model_Address_Abstract $address,
        Mage_Core_Model_Store $store
    ): bool {
        $defaultAddress = [
            "country_id" => $store->getConfig(Mage_Core_Helper_Data::XML_PATH_DEFAULT_COUNTRY),
        ];
        $addrFields = array_flip(self::ADDRESS_DATA_FIELDS);
        $data = array_filter(array_diff(array_intersect_key($address->getData(), $addrFields), $defaultAddress));

        return count($data) === 0;
    }
}
