<?php

declare(strict_types=1);

use Spatie\Snapshots\MatchesSnapshots;

use PHPUnit\Framework\TestCase;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;

class QuoteTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
    }

    public function onNotSuccessfulTest($e): void {
        MagentoManager::logQueries();

        throw $e;
    }

    protected function customerFixture() {
        MagentoManager::init();

        $store = Mage::app()->getStore();

        $customer = Mage::getModel("customer/customer")
            ->setStore($store)
            ->loadByEmail("test-customer@example.com")
            ->addData([
                "website_id" => $store->getWebsiteId(),
                "email" => "test-customer@example.com",
                "firstname" => "Test",
                "lastname" => "Customer",
            ])
            ->setStore($store)
            ->setPassword("test-customer")
            ->save();

        return $customer;
    }

    public function testRemoveQuoteEmpty() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], json_encode([
            [ "query" => 'query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    email
                    grandTotal {
                        incVat
                    }
                    items {
                        product {
                            sku
                            attributes {
                                shortDescription
                                smallImage
                            }
                        }
                    }
                    shipping {
                        method {
                            description
                        }
                    }
                }
            }'],
            [ "query" => 'mutation {
                removeQuote
            }'],
            [ "query" => 'query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    email
                    grandTotal {
                        incVat
                    }
                    items {
                        product {
                            sku
                            attributes {
                                shortDescription
                                smallImage
                            }
                        }
                    }
                    shipping {
                        method {
                            description
                        }
                    }
                }
            }'],
        ])));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testRemoveQuote() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($product, $request);
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], json_encode([
            [ "query" => 'query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    email
                    grandTotal {
                        incVat
                    }
                    items {
                        product {
                            sku
                            attributes {
                                shortDescription
                                smallImage
                            }
                        }
                    }
                    shipping {
                        method {
                            description
                        }
                    }
                }
            }'],
            [ "query" => 'mutation {
                removeQuote
            }'],
            [ "query" => 'query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    email
                    grandTotal {
                        incVat
                    }
                    items {
                        product {
                            sku
                            attributes {
                                shortDescription
                                smallImage
                            }
                        }
                    }
                    shipping {
                        method {
                            description
                        }
                    }
                }
            }'],
        ])));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testQuoteTaxRates() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($product, $request);
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                taxRates {
                    percent
                    amount
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testEmail() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $quote = Mage::getSingleton("checkout/session")->getQuote();

        $quote->setCustomerEmail("test@example.com");

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                email
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetEmail() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuoteEmail(email: "test@example.com")  {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuote()->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                email
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuote()->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuoteEmail(email: "another@example.com")  {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuote()->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                email
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetEmailInvalid() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuoteEmail(email: "invalid")  {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetEmailLoggedIn() {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuoteEmail(email: "invalid")  {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }
}
