<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

/**
 * @psalm-type BundleOptions Array<string, Array<string> | string>
 */
class MageQL_Sales_Model_Product_Buyrequest_Bundle extends MageQL_Sales_Model_Product_Buyrequest {
    /**
     * @var ?BundleOptions
     */
    protected $bundleOptions;

    /**
     * @param ?BundleOptions $bundleOptions
     */
    protected function __construct(
        Mage_Catalog_Model_Product $product,
        array $bundleOptions = null
    ) {
        $this->product = $product;
        $this->bundleOptions = $bundleOptions;
    }

    /**
     * Attempts to generate a valid buy-request for a bundle using default
     * selections, will return null if there are required options without
     * default selection(s).
     */
    public static function fromProductDefaults(
        Mage_Catalog_Model_Product $product,
        Mage_Core_Model_Store $store
    ): ?self {
        if (self::requiresSelection($product, $store)) {
            return null;
        }

        return new self($product, null);
    }

    public static function getOptionSelections(
        Mage_Bundle_Model_Option $option,
        Mage_Core_Model_Store $store
    ): Mage_Bundle_Model_Resource_Selection_Collection {
        /**
         * @var Mage_Bundle_Model_Resource_Selection_Collection
         */
        $selections = Mage::getResourceModel("bundle/selection_collection");

        $selections->setOptionIdsFilter([$option->getId()]);

        // No visibility filter here since bundle option selections can be invisible
        $selections->addStoreFilter($store);
        $selections->addMinimalPrice();
        $selections->joinPrices((int)$store->getWebsiteId());

        return $selections;
    }

    public static function requiresSelection(
        Mage_Catalog_Model_Product $product,
        Mage_Core_Model_Store $store
    ): bool {
        /**
         * @var Mage_Bundle_Model_Product_Type
         */
        $instance = $product->getTypeInstance(true);
        $options = $instance->getOptions($product);

        foreach($options as $opt) {
            if($opt->getType() === "hidden" || !$opt->getRequired()) {
                continue;
            }

            $hasSelection = false;

            foreach(self::getOptionSelections($opt, $store) as $sel) {
                if($sel->getIsDefault() && $sel->getSelectionQty() > 0) {
                    $hasSelection = true;
                }
            }

            if( ! $hasSelection) {
                return true;
            }
        }

        return false;
    }

    public function toRequest(): array {
        $req = parent::toRequest();

        if ($this->bundleOptions) {
            $req["bo"] = $this->bundleOptions;
        }

        return $req;
    }
}
