<?php

declare(strict_types=1);

namespace Quote;

use Spatie\Snapshots\MatchesSnapshots;

use Mage;
use Varien_Object;

use PHPUnit\Framework\TestCase;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;

class ShippingTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
    }

    public function onNotSuccessfulTest($e): void {
        MagentoManager::logQueries();

        throw $e;
    }

    public function testShippingMethodsEmptyQuote() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            shippingMethods {
                code
                carrierTitle
                methodTitle
                price {
                    incVat
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testEmptyAddressSingleProduct() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($product, $request);
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            shippingMethods {
                code
                carrierTitle
                methodTitle
                methodDescription
                price {
                    incVat
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testEmptyAddressVirtualProduct() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($product, $request);
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            shippingMethods {
                code
                carrierTitle
                methodTitle
                methodDescription
                price {
                    incVat
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetMethodToCheapest() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($product, $request);
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                shipping {
                    method {
                        code
                        description
                    }
                    total {
                        incVat
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuoteShippingMethodToCheapest {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                shipping {
                    method {
                        code
                        description
                    }
                    total {
                        incVat
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetMethodSinple() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($product, $request);
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                shipping {
                    method {
                        code
                        description
                    }
                    total {
                        incVat
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuoteShippingMethod(shippingMethodCode: "flatrate_flatrate") {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                shipping {
                    method {
                        code
                        description
                    }
                    total {
                        incVat
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetMethodSinpleMultiple() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($product, $request);
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                shipping {
                    method {
                        code
                        description
                    }
                    total {
                        incVat
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuoteShippingMethod(shippingMethodCode: "flatrate_flatrate") {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        // Ensure that we do not set it again
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuoteShippingMethod(shippingMethodCode: "flatrate_flatrate") {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                shipping {
                    method {
                        code
                        description
                    }
                    total {
                        incVat
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }
}
