<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

class MageQL_Sales_Model_Quote extends Mage_Core_Model_Abstract {
    const SUCCESS = "success";
    const ERROR_INVALID_EMAIL_ADDRESS = "errorInvalidEmailAddress";

    /**
     * Event issued to collect potential result-types for placeOrder mutation.
     *
     * Params:
     *
     *  * errors: Varien_Object of Array<resultEnumValue, { apiErrorCode?:int, description: string }>
     *
     *    The key is the result-enum-value and `apiErrorCode` is the exception
     *    error code for `Crossroads_API_ResponseException` for backwards compat,
     *    if any.
     */
    const EVENT_PLACE_ORDER_RESULT_TYPE = "mageql_sales_place_order_result_errors";

    /**
     * The checks which the payment method should do when validating.
     */
    const PAYMENT_CHECKS = Mage_Payment_Model_Method_Abstract::CHECK_USE_CHECKOUT
        | Mage_Payment_Model_Method_Abstract::CHECK_USE_FOR_COUNTRY
        | Mage_Payment_Model_Method_Abstract::CHECK_USE_FOR_CURRENCY
        | Mage_Payment_Model_Method_Abstract::CHECK_ORDER_TOTAL_MIN_MAX
        | Mage_Payment_Model_Method_Abstract::CHECK_ZERO_TOTAL;

    public static function resolve(): Mage_Sales_Model_Quote {
        return Mage::getSingleton("mageql_sales/quote")->getQuote();
    }

    /**
     * @return Array<Mage_Sales_Model_Quote_Address>
     */
    public static function resolveAddresses(
        Mage_Sales_Model_Quote $src,
        array $args,
        $ctx,
        ResolveInfo $info
    ): array {
        $addresses = [];

        foreach($src->getAddressesCollection()->getItems() as $addr) {
            if($addr->getAddressType() !== Mage_Customer_Model_Address_Abstract::TYPE_SHIPPING ||
                ! $addr->getSameAsBilling()) {
                $addresses[] = $addr;
            }
        }

        return $addresses;
    }

    public static function resolveEmail(Mage_Sales_Model_Quote $src): ?string {
        $email = $src->getCustomerEmail();

        if( ! $email) {
            return null;
        }

        return $email;
    }

    public static function resolveGrandTotal(Mage_Sales_Model_Quote $src, array $args, $ctx) {
        $store = $ctx->getStore();
        $tax = 0.0;

        foreach($src->getAllAddresses() as $addr) {
            $tax += (float)$addr->getTaxAmount();
        }

        return new Varien_Object([
            "inc_vat" => (float) $src->getGrandTotal(),
            "ex_vat" => (float) $store->roundPrice($src->getGrandTotal() - $tax),
            "vat" => $tax,
        ]);
    }

    public static function resolveIsVirtual(Mage_Sales_Model_Quote $src) {
        return $src->isVirtual();
    }

    public static function resolveTaxRatePercent($src) {
        return $src["percent"];
    }

    public static function resolveTaxRateAmount($src, array $args, $ctx) {
        $store = $ctx->getStore();

        return $store->roundPrice($src["amount"]);
    }

    public static function resolveTaxRates(Mage_Sales_Model_Quote $src) {
        // No idea why it is on shipping address sometimes and other times on billing,
        // something related to tax-settings.
        $rates = [];

        foreach($src->getAllAddresses() as $addr) {
            $appliedRates = $addr->getAppliedTaxes();

            if( ! empty($appliedRates)) {
                $rates = array_merge($rates, $appliedRates);
            }
        }

        if(empty($rates)) {
            $rates = method_exists($src, "getFullTaxInfo") ?  $src->getFullTaxInfo() : [];
        }

        return array_reduce($rates, function($taxes, $tax) {
            for($i = 0, $c = count($taxes); $i < $c; $i++) {
                if($taxes[$i]["percent"] === (float) $tax["percent"]) {
                    $taxes[$i]["amount"] += $tax["amount"];

                    return $taxes;
                }
            }

            $taxes[] = [
                "percent" => (float) $tax["percent"],
                "amount" => (float) $tax["amount"]
            ];

            return $taxes;
        }, []);
    }

    public static function resolveShipping(Mage_Sales_Model_Quote $src) {
        if( ! $src->isVirtual()) {
            $model = Mage::getSingleton("mageql_sales/quote");
            $shipping = $model->getShippingAddress();

            if($shipping && $shipping->getShippingMethod()) {
                return $shipping;
            }
        }

        return  null;
    }

    public static function resolveSubTotal(Mage_Sales_Model_Quote $src, array $args, $ctx) {
        $totals = $src->getTotals();
        $store = $ctx->getStore();
        $subTotal = 0;
        $exclTax = 0;

        if(array_key_exists("subtotal", $totals)) {
            $subTotal = (float) $totals["subtotal"]->getValueInclTax();
            $exclTax = (float) $totals["subtotal"]->getValueExclTax();
        }

        return new Varien_Object([
            "ex_vat" => $exclTax,
            "inc_vat" => $subTotal,
            "vat" => (float) $store->roundPrice($subTotal - $exclTax),
        ]);
    }

    public static function resolveItems(
        Mage_Sales_Model_Quote $src,
        array $args,
        $ctx,
        ResolveInfo $info
    ) {
        $config = Mage::getSingleton("mageql_catalog/attributes_product");
        // 2 levels deep to get attributes (1 item, 2 product, 3 attributes)
        $fields = $info->getFieldSelection(3);
        // Merge attributes
        $fields = array_merge($fields["product"] ?? [], $fields["product"]["attributes"] ?? []);

        // We use all attribute sets since we cannot be certain about which sets we will get
        $toSelect = $config->getUsedAttributes(
            $config->getAttributesByArea(MageQL_Catalog_Model_Attributes_Abstract::AREA_LIST),
            $fields
        );

        // Set the attributes so they are loaded when we get the quote items
        Mage::getSingleton("mageql_sales/attributes_product")->setQuoteAttributes($toSelect);

        // Do not use a cache since we want to use the previously set attributes
        // Note that the $useCache flag does not work on quote getItemsCollection
        $collection = Mage::getModel("sales/quote_item")->getCollection()->setQuote($src);

        $items = $collection->getItems();

        return array_values(array_filter($items, function($item) {
            return ! $item->isDeleted() && ! $item->getParentItemId();
        }));
    }

    public static function mutateRemove(): bool {
        $model = Mage::getSingleton("mageql_sales/quote");

        $model->removeQuote();

        return true;
    }

    public static function mutateSetEmail($src, array $args): string {
        $model = Mage::getSingleton("mageql_sales/quote");
        $email = $args["email"];
        $quote = $model->getQuote();

        if( ! Zend_Validate::is($email, "EmailAddress")) {
            return self::ERROR_INVALID_EMAIL_ADDRESS;
        }

        $quote->setCustomerEmail($email);

        foreach($quote->getAllAddresses() as $addr) {
            $addr->setEmail($email);
        }

        $model->saveSessionQuote();

        return self::SUCCESS;
    }

    public static function mutatePlaceOrder(): MageQL_Sales_Model_Order_AbstractResult {
        $model = Mage::getSingleton("mageql_sales/quote");
        $session = Mage::getSingleton("checkout/session");
        $quote = $model->getQuote();
        $customer = Mage::getSingleton("customer/session")->getCustomer();

        if($customer) {
            // Store email since setCustomer overwrites it
            $email = $quote->getCustomerEmail();

            $quote->setCustomer($customer);

            // Restore if we had something
            if($email) {
                $quote->setCustomerEmail($email);
            }
        }
        else {
            $quote->setCustomerId(null);
            $quote->setCustomerIsGuest(true);
            $quote->setCustomerGroupId(Mage_Customer_Model_Group::NOT_LOGGED_IN_ID);
        }

        foreach([$quote->getBillingAddress(), $quote->getShippingAddress()] as $address) {
            $address->setEmail($quote->getCustomerEmail());
        }

        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->setTotalsCollectedFlag(false);
        $quote->collectTotals();
        $quote->save();

        // TODO: Validate agreements

        $err = $model->validateQuote($quote);

        if($err) {
            return $err;
        }

        try {
            $service = Mage::getModel("sales/service_quote", $quote);
            $service->submitAll();

            $order = $service->getOrder();
            $payment = $quote->getPayment();

            $result = new MageQL_Sales_Model_Order_Result($order);

            $redirect = $payment->getOrderPlaceRedirectUrl();
            $agreement = $payment->getBillingAgreement();

            if($redirect) {
                $result->setPaymentRedirectUrl($redirect);
            }

            if( ! $redirect && $order->getCanSendNewEmailFlag()) {
                try {
                    $order->queueNewOrderEmail();
                }
                catch(Exception $e) {
                    Mage::logException($e);
                }
            }

            $session->unsetAll();
            $session->clearHelperData();
            $session->setLastQuoteId($quote->getId());
            $session->setLastSuccessQuoteId($quote->getId());
            $session->setLastOrderId($order->getId());
            $session->setLastRealOrderId($order->getIncrementId());

            if($agreement) {
                $sesson->setLastBillingAgreementId($agreement->getId());
            }

            $quote->setIsActive(false);
            $quote->save();

            return $result;
        }
        catch(MageQL_Sales_Model_ErrorInstance $e) {
            $errorCode = $e->getErrorCode();
            $possibleErrors = new Varien_Object([]);

            Mage::dispatchEvent(self::EVENT_PLACE_ORDER_RESULT_TYPE, [
                "errors" => $possibleErrors,
            ]);

            if( ! $possibleErrors->hasData($errorCode)) {
                Mage::log(sprintf(
                    "%s: Error code '%s' unrecognized for mutation placeOrder",
                    __METHOD__,
                    $errorCode
                ));

                throw $e;
            }

            return new MageQL_Sales_Model_Order_Error($errorCode);
        }
        // Backwards compatibility block, attempt to convert any codes to responses
        catch(Crossroads_API_ResponseException $e) {
            $possibleErrors = new Varien_Object([]);

            Mage::dispatchEvent(self::EVENT_PLACE_ORDER_RESULT_TYPE, [
                "errors" => $possibleErrors,
            ]);

            foreach($possibleErrors->getData() as $enumValue => $error) {
                if(array_key_exists("apiErrorCode", $error) && $error["apiErrorCode"] === $e->getCode()) {
                    return new MageQL_Sales_Model_Order_Error($error);
                }
            }

            // We tried our best
            throw $e;
        }
    }

    public function validateQuote(Mage_Sales_Model_Quote $quote): ?MageQL_Sales_Model_Order_Error {
        if( ! $quote->hasItems()) {
            return new MageQL_Sales_Model_Order_Error(MageQL_Sales_Model_Order_Error::NO_ITEMS);
        }

        if( ! $quote->isVirtual()) {
            $shipping = $quote->getShippingAddress();
            $validation = $shipping->validate();

            if($validation !== true) {
                // TODO: Propagate shipping address validation errors
                return new MageQL_Sales_Model_Order_Error(MageQL_Sales_Model_Order_Error::SHIPPING_ADDRESS_VALIDATION_FAILED);
            }

            $method = $shipping->getShippingMethod();
            $rate = $shipping->getShippingRateByCode($method);

            if( ! $method || ! $rate) {
                return new MageQL_Sales_Model_Order_Error(MageQL_Sales_Model_Order_Error::NO_SHIPPING_METHOD);
            }
        }

        $validation = $quote->getBillingAddress()->validate();

        if($validation !== true) {
            // TODO: Propagate billing address validation errors
            return new MageQL_Sales_Model_Order_Error(MageQL_Sales_Model_Order_Error::BILLING_ADDRESS_VALIDATION_FAILED);
        }

        $payment = $quote->getPayment();

        if( ! $payment->getMethod()) {
            return new MageQL_Sales_Model_Order_Error(MageQL_Sales_Model_Order_Error::NO_PAYMENT_METHOD);
        }

        $paymentSchema = Mage::getSingleton("mageql_sales/schema_default_payment");

        if( ! $paymentSchema->hasPaymentMethod($payment->getMethod())) {
            // TODO: Log this as an exception?
            return new MageQL_Sales_Model_Order_Error(MageQL_Sales_Model_Order_Error::INVALID_PAYMENT_METHOD);
        }

        // We have to ensure the payment method is still available before placing the order
        $paymentMethod = $payment->getMethodInstance();

        if( ! $paymentMethod->isAvailable($quote) || ! $paymentMethod->isApplicableToQuote($quote, self::PAYMENT_CHECKS)) {
            return new MageQL_Sales_Model_Order_Error(MageQL_Sales_Model_Order_Error::PAYMENT_NOT_AVAILABLE);
        }

        if( ! Zend_Validate::is($quote->getCustomerEmail(), "EmailAddress")) {
            return new MageQL_Sales_Model_Order_Error(MageQL_Sales_Model_Order_Error::INVALID_EMAIL_ADDRESS);
        }

        return null;
    }

    public function getQuote() {
        $session = Mage::getSingleton("checkout/session");
        $quote = $session->getQuote();

        // TODO: Check if it is a new session and then apply defaults?

        // Ensure we have defaults for addresses
        $this->setQuoteDefaults($quote);

        return $quote;
    }

    public function removeQuote() {
        $session = Mage::getSingleton("checkout/session");
        $quote = $session->getQuote();

        // Only save it if we need to
        if($quote->getId()) {
            $quote->setIsActive(false);
            $quote->save();
        }

        $session->unsetAll();
    }

    /**
     * Variant of Mage_Sales_Model_Quote::getBillingAddress() which will NOT
     * automatically create an address if none exists.
     */
    public function getBillingAddress(): ?Mage_Sales_Model_Quote_Address {
        $quote = $this->getQuote();

        foreach($quote->getAllAddresses() as $addr) {
            if($addr->getAddressType() === Mage_Customer_Model_Address_Abstract::TYPE_BILLING) {
                return $addr;
            }
        }

        return null;
    }

    /**
     * Variant of Mage_Sales_Model_Quote::getShippingAddress() which will NOT
     * automatically create an address if none exists.
     */
    public function getShippingAddress(): ?Mage_Sales_Model_Quote_Address {
        $quote = $this->getQuote();

        foreach($quote->getAllAddresses() as $addr) {
            if($addr->getAddressType() === Mage_Customer_Model_Address_Abstract::TYPE_SHIPPING) {
                return $addr;
            }
        }

        return null;
    }

    /**
     * Gets billing address, if it does not already exists it will be created, with defaults.
     */
    public function makeBillingAddress(): Mage_Sales_Model_Quote_Address {
        $quote = $this->getQuote();
        $billing = $quote->getBillingAddress();

        $this->setAddressDefaults($quote, $billing);

        return $billing;
    }

    /**
     * Gets shipping address, if it does not already exists it will be created, with defaults.
     */
    public function makeShippingAddress(): Mage_Sales_Model_Quote_Address {
        $quote = $this->getQuote();
        $shipping = $quote->getShippingAddress();

        $this->setAddressDefaults($quote, $shipping);

        return $shipping;
    }

    /**
     * Fills default values in a quote, will not instantiate new addresses.
     */
    public function setQuoteDefaults(Mage_Sales_Model_Quote $quote) {
        foreach($quote->getAllAddresses() as $addr) {
            $this->setAddressDefaults($quote, $addr);
        }
    }

    /**
     * Attempts to set the payment method to free if available, will remove it if not available.
     */
    public function trySetFreeIfAvailable(Mage_Sales_Model_Quote $quote) {
        $store = $quote->getStore();
        $paymentSchema = Mage::getSingleton("mageql_sales/schema_default_payment");

        if( ! $paymentSchema->hasPaymentMethod("free")) {
            return;
        }

        $free = Mage::getModel($store->getConfig("payment/free/model"));

        if( ! $free) {
            // Should not be free now
            if($quote->getPayment()->getMethod() === "free") {
                $quote->removePayment();
            }

            return;
        }

        $free->setStore($store);

        if( ! $free->isAvailable($quote) || ! $free->isApplicableToQuote($quote, self::PAYMENT_CHECKS)) {
            // Should not be free now
            if($quote->getPayment()->getMethod() === "free") {
                $quote->removePayment();
            }

            return;
        }

        $payment = $quote->getPayment();

        if($payment->getMethod() === "free") {
            return;
        }

        $this->importPaymentData($quote, "free", []);
    }

    public function setAddressDefaults(Mage_Sales_Model_Quote $quote, Mage_Sales_Model_Quote_Address $address) {
        $store = $quote->getStore();

        // Ensure that we always have a country
        if( ! $address->getCountryId()) {
            $address->setCountryId($store->getConfig(Mage_Core_Model_Locale::XML_PATH_DEFAULT_COUNTRY));
        }

        // Adapted from Mage_Checkout_Model_Cart::init(), just make sure to not create
        // addresses early
        if(
            $address->getAddressType() === Mage_Customer_Model_Address_Abstract::TYPE_SHIPPING &&
            ! $quote->hasItems()
        ) {
            $address->setCollectShippingRates(false)->removeAllShippingRates();
        }
    }

    public function saveSessionQuoteWithShippingRates() {
        return $this->saveSessionQuote(true);
    }

    public function saveSessionQuote($forceCollect = null) {
        // TODO: Break out to make a separate save for quotes in general (unit testing)
        $quote = $this->getQuote();

        // Ensure that we have both a billing and a shipping address, and that
        // we collect rates since they are both new
        $quote->getBillingAddress();
        // Recollect shipping rates since we have a new product
        $shipping = $quote->getShippingAddress();

        if($forceCollect !== null) {
            $shipping->setCollectShippingRates($forceCollect);
        }

        $this->setQuoteDefaults($quote);

        $quote->collectTotals();

        // Only perform this when we save
        if($quote->hasItems()) {
            $this->trySetFreeIfAvailable($quote);
        }

        $quote->save();

        Mage::getSingleton("checkout/session")->setQuoteId($quote->getId());
    }

    /**
     * Adds a product to the current quote.
     */
    public function addProduct(
        Mage_Catalog_Model_Product $product,
        array $buyRequest
    ): Mage_Sales_Model_Quote_Item {
        $session = Mage::getSingleton("checkout/session");
        $quote = $this->getQuote();

        $result = $quote->addProduct($product, new Varien_Object($buyRequest));

        if(is_string($result)) {
            // Throw to let the mutation handle it
            Mage::throwException($result);
        }

        // Added quote item, affects shipping
        $this->saveSessionQuoteWithShippingRates();

        $session->setLastAddedProductId($product->getId());

        return $result;
    }

    public function updateItem(
        Mage_Sales_Model_Quote_Item $item,
        Mage_Catalog_Model_Product $product,
        array $buyRequest
    ): Mage_Sales_Model_Quote_Item {
        $session = Mage::getSingleton("checkout/session");
        $quote = $this->getQuote();
        $result = $quote->updateItem($item->getId(), new Varien_Object($buyRequest));

        if(is_string($result)) {
            // Throw to let the mutation handle it
            Mage::throwException($result);
        }

        // Recollect shipping rates since we have modified a quantity
        $this->saveSessionQuoteWithShippingRates();

        // Wishlist listens for this
        Mage::dispatchEvent("checkout_cart_update_item_complete", [
            "product"  => $product,
            "request"  => Mage::app()->getRequest(),
            "response" => Mage::app()->getResponse(),
        ]);

        return $result;
    }

    // TODO: Handle throws
    public function importPaymentData(
        Mage_Sales_Model_Quote $quote,
        string $method,
        array $data
    ): Mage_Sales_Model_Quote_Payment {
        if($quote->isVirtual()) {
            $quote->getBillingAddress()->setPaymentMethod($method);
        }
        else {
            // Shipping totals may be affected by payment method
            $quote->getShippingAddress()
                  ->setPaymentMethod($method)
                  ->setCollectShippingRates(true);
        }

        $data["method"] = $method;
        $data["checks"] = self::PAYMENT_CHECKS;

        $payment = $quote->getPayment();

        // Make sure we reset the payment method instance when updating
        if($payment->getMethod() !== $method) {
            $payment->unsMethodInstance();
        }

        $payment->importData($data);

        return $quote->getPayment();
    }
}
