<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

class MageQL_Sales_Model_Product extends Mage_Core_Model_Abstract {
    /**
     * Generates a buy request for a simple/virtual product.
     *
     * @param Mage_Catalog_Model_Product
     */
    public static function resolveBuyRequestSimple($src) {
        switch($src->getTypeId()) {
        case "bundle":
        case "configurable":
            // Bundles and configs require a buyRequest from a variant
            return null;
        case "simple":
        case "virtual":
            if( ! $src->isSaleable()) {
                return null;
            }

            if($src->getParentId()) {
                return null;
            }

            return json_encode([
                "p" => (int)$src->getId(),
            ]);
        default:
            throw new Exception(sprintf("%s: Unknown product type %s", __METHOD__, $src->getTypeId()));
        }
    }

    /**
     * Generates a buy-request string for a configurable item which can be used
     * to add a product to a quote.
     */
    public static function resolveBuyRequestConfigItem($src) {
        return self::createConfigurableChildBuyRequest($src[0], $src[1]);
    }

    /**
     * @param Mage_Catalog_Model_Product
     * @param array Array of configurable attributes
     */
    public static function createConfigurableChildBuyRequest($child, $configAttrs) {
        $requestAttrs = [];

        if( ! in_array($child->getTypeId(), ["simple", "virtual"])) {
            throw new Exception(sprintf("%s: Invalid product type %s for config child", __METHOD__, $child->getTypeId()));
        }

        if( ! $child->getParentId()) {
            throw new Exception(sprintf("%s: Invalid parent product id on %s (%d) for config child", __METHOD__, $child->getSku(), $child->getId()));
        }

        if( ! $child->isSaleable()) {
            return null;
        }

        foreach($configAttrs as $attr) {
            $prodAttr = $attr->getProductAttribute();

            $requestAttrs[$prodAttr->getAttributeId()] = (int)$child->getData($prodAttr->getAttributeCode());
        }

        return json_encode([
            "p" => (int)$child->getParentId(),
            "a" => $requestAttrs,
        ]);
    }
}
