<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

use function MageQL\snakeCaseToCamel;

use MageQL\Registry;
use MageQL\Type\AbstractBuilder;

class MageQL_Sales_Model_Schema_Default_Payment extends MageQL_Core_Model_Schema_Abstract {
    protected $models = null;

    /**
     * Returns a list of enabled payment methods for the current store.
     *
     * @return array of MageQL_Sales_Model_Payment_Abstract
     */
    public function getModels(): array {
        if($this->models === null) {
            $this->models = [];

            $store = Mage::app()->getStore();
            $payments = $store->getConfig("mageql/sales/payment");

            foreach($payments as $code => $data) {
                $model = Mage::getSingleton($data["model"]);
                $code = $model->getCode();

                // We do not want to include inactives since that will give the
                // wrong impression to clients if types for unsupported payment
                // types exist
                if($store->getConfig("payment/$code/active")) {
                    $this->models[$code] = $model;
                }
            }
        }

        return $this->models;
    }

    /**
     * Returns true if this schema has support for the supplied payment method.
     */
    public function hasPaymentMethod(string $code): bool {
        foreach($this->getModels() as $m) {
            if($m->getCode() === $code) {
                return true;
            }
        }

        return false;
    }

    public function typePaymentMethod(Mage_Payment_Model_Method_Abstract $src): string {
        $match = $this->getModels()[$src->getCode()] ?? null;

        if( ! $match) {
            // This should not happen unless paymentMethods resolver is broken
            throw new Exception(sprintf("%s: Unsupported payment method '%s'", __METHOD__, $src->getCode()));
        }

        return $match->getPaymentMethodType();
    }

    public function typeQuotePayment(Mage_Sales_Model_Quote_Payment $src): string {
        $match = $this->getModels()[$src->getMethod()] ?? null;

        if( ! $match) {
            // This should not happen unless it is a broken quote, or the quote payment resolver is broken
            throw new Exception(sprintf("%s: Unsupported payment method '%s'", __METHOD__, $src->getMethod()));
        }

        return $match->getQuotePaymentType();
    }

    public function resolvePaymentMethods(): array {
        $methods = [];
        $paymentHelper = Mage::helper("payment");
        $quote = Mage::getSingleton("mageql_sales/quote")->getQuote();
        $payment = $quote->getPayment();

        // This function uses isAvailable on the instance
        foreach($paymentHelper->getStoreMethods($quote->getStore(), $quote) as $method) {
            if($method->canUseCheckout()) {
                foreach($this->getModels() as $m) {
                    if($m->getCode() === $method->getCode()) {
                        $method->setInfoInstance($payment);

                        $methods[] = $method;
                    }
                }
            }
        }

        return $methods;
    }

    public function getTypeBuilder(string $typeName, Registry $registry): ?AbstractBuilder {
        switch($typeName) {
        case "MutationSetQuotePaymentMethod":
            return $this->object("Available payment methods to set");

        case "PaymentMethod":
            return $this->interface("Object representing a payment method")
                ->setResolveType([$this, "typePaymentMethod"]);

        case "QuotePayment":
            return $this->interface("Object containing data for the selected payment method")
                ->setResolveType([$this, "typeQuotePayment"]);
        }

        foreach($this->getModels() as $model) {
            $builder = $model->getTypeBuilder($typeName, $registry);

            if($builder) {
                return $builder;
            }
        }

        return null;
    }

    public function getTypeFields(string $typeName, Registry $registry): array {
        switch($typeName) {
        case "PaymentMethod":
            return [
                // TODO: Is this actually useful?
                "code" => $this->field("ID!", "Payment method code"),
                "title" => $this->field("String!", "Payment method title"),
            ];

        case "Query":
            return [
                "paymentMethods" => $this->field(
                    "[PaymentMethod!]!",
                    "List of enabled payment methods"
                )
                    ->setResolver([$this, "resolvePaymentMethods"]),
            ];

        case "QuotePayment":
            return [
                "code" => $this->field("ID!", "Payment code for the selected payment type")
                    ->setResolver("MageQL_Sales_Model_Quote_Payment::resolvePaymentCode"),
            ];
        }

        $fields = [];

        foreach($this->getModels() as $model) {
            $fields = array_merge($fields, $model->getTypeFields($typeName, $registry));
        }

        return $fields;
    }

    public function getUnreachableTypes(): array {
        $types = [];

        foreach($this->getModels() as $model) {
            $types = array_merge($types, $model->getUnreachableTypes());
        }

        return $types;
    }
}
