<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

use function MageQL\snakeCaseToCamel;

class MageQL_Sales_Model_Item extends Mage_Core_Model_Abstract {
    public static function resolveProduct(Mage_Core_Model_Abstract $item): Mage_Catalog_Model_Product {
        return $item->getProduct();
    }

    /**
     * Resolves the configuration option into a [parent, child] tuple.
     */
    public static function resolveQuoteConfigOption(Mage_Sales_Model_Quote_Item_Abstract $item): array {
        $children = $item->getChildren();
        $parent = $item->getProduct();
        $child = $children[0]->getProduct();

        // Add the parent id so we skip buyRequest on it
        $child->addData([
            "parent_id" => $parent->getId(),
        ]);

        return [$parent, $child];
    }

    /**
     * Resolves the configuration option into a [parent, child] tuple.
     */
    public static function resolveOrderConfigOption(Mage_Sales_Model_Order_Item $item): array {
        // The only differing line compared to quote:
        $children = $item->getChildrenItems();
        $parent = $item->getProduct();
        $child = $children[0]->getProduct();

        // Add the parent id so we skip buyRequest on it
        $child->addData([
            "parent_id" => $parent->getId(),
        ]);

        return [$parent, $child];
    }

    public static function resolveConfigurableAttributeCode($attrProductTuple): string {
        [$attr, $product] = $attrProductTuple;

        // TODO: Should be a bit more reusable
        return snakeCaseToCamel($attr->getProductAttribute()->getAttributeCode());
    }

    public static function resolveConfigurableAttributeLabel($attrProductTuple): string {
        [$attr, $product] = $attrProductTuple;

        // TODO: Should be a bit more reusable
        $prodAttr = $attr->getProductAttribute();

        return $prodAttr->getStoreLabel() ?: $prodAttr->getFrontend()->getLabel() ?: $prodAttr->getLabel();
    }

    public static function resolveConfigurableAttributeValue($attrProductTuple, array $args, $ctx): string {
        [$attr, $product] = $attrProductTuple;

        // TODO: Can we get this part to actually join the possible attributes when obtaining the collection?
        // We need to load the raw attribute in case it is not set since it
        // might not be joined on the collection because the attribute does
        // not have to be included when querying
        $resource = $product->getResource();
        $prodAttr = $attr->getProductAttribute();
        $model = $prodAttr->getSource();
        // Attempt to load it from the product first, and if it fails then
        // read it from DB
        $value = $product->getData($prodAttr->getAttributeCode()) ?:
            $resource->getAttributeRawValue($product->getId(), $prodAttr->getAttributeCode(), $ctx->getStore());
        $option = $model->getOptionText($value);

        return $option;
    }

    /**
     * Uses a [parent, child] tuple as source.
     */
    public static function resolveConfigurableOptionAttributes($tuple): array {
        [$parent, $item] = $tuple;
        $instance = $parent->getTypeInstance(true);
        $attrs = [];

        foreach($instance->getConfigurableAttributes($parent) as $attr) {
            $attrs[] = [$attr, $item];
        }

        return $attrs;
    }

    public static function resolveConfigurableOptionProduct($tuple): Mage_Catalog_Model_Product {
        return $tuple[1];
    }

    public static function resolveRowTotalExVat($src, array $args, $ctx): float {
        $store = $ctx->getStore();

        return (float)$store->roundPrice($src->getRowTotal() - $src->getDiscountAmount());
    }

    public static function resolveRowTotalIncVat($src, array $args, $ctx): float {
        $store = $ctx->getStore();

        return (float)$store->roundPrice($src->getRowTotal()
            + $src->getTaxAmount()
            + $src->getHiddenTaxAmount()
            - $src->getDiscountAmount());
    }

    public static function resolveRowTotalVat($src): float {
        return (float)$src->getTaxAmount();
    }
}
