<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

use function MageQL\snakeCaseToCamel;

use MageQL\Registry;
use MageQL\Type\AbstractBuilder;
use MageQL\Type\InputFieldBuilder;

class MageQL_Sales_Model_Schema_Default_Quote extends MageQL_Core_Model_Schema_Abstract {
    public function getAddQuoteItemResultTypeOptions(): array {
        return [
            MageQL_Sales_Model_Quote_Item::SUCCESS => [
                "description" => "Product was added to the quote successfully",
            ],
            MageQL_Sales_Model_Quote_Item::ERROR_DECODE => [
                "description" => "The supplied buyRequest was invalid",
            ],
            MageQL_Sales_Model_Quote_Item::ERROR_PRODUCT_NOT_FOUND => [
                "description" => "The supplied buyRequest refers to a product which cannot be found",
            ],
            MageQL_Sales_Model_Quote_Item::ERROR_PRODUCT_NOT_IN_STOCK => [
                "description" => "The product is not in stock",
            ],
            MageQL_Sales_Model_Quote_Item::ERROR_PRODUCT_QUANTITY_NOT_AVAILABLE => [
                "description" => "The requested product quantity is not available",
            ],
            MageQL_Sales_Model_Quote_Item::ERROR_PRODUCT_REQUIRES_BUNDLE_OPTIONS => [
                "description" => "The supplied product requires bundle options configuration.",
            ],
            MageQL_Sales_Model_Quote_Item::ERROR_PRODUCT_MAX_QUANTITY => [
                "description" => "The requested quantity exceeds the maximum allowed quantity",
            ],
            MageQL_Sales_Model_Quote_Item::ERROR_PRODUCT_VARIANT_NOT_AVAILABLE => [
                "description" => "The requested product variant is not available",
            ],
            MageQL_Sales_Model_Quote_Item::ERROR_BUNDLE_SINGLE_GOT_MULTI => [
                "description" => "Got multiple options for a single-select bundle option",
            ],
            MageQL_Sales_Model_Quote_Item::ERROR_BUNDLE_EMPTY => [
                "description" => "The bundle is empty and cannot be added to cart",
            ],
            MageQL_Sales_Model_Quote_Item::ERROR_BUNDLE_SELECTION_QTY_IMMUTABLE => [
                "description" => "The bundle selection qty is not mutable",
            ],
        ];
    }

    public function getTypeBuilder(string $typeName, Registry $registry): ?AbstractBuilder {
        switch($typeName) {
        case "AddQuoteItemResult":
            return $this->object("Object containing the result of adding a product to the quote");

        case "AddQuoteItemResultType":
            // TODO: Make this enum extendable
            return $this->enum(
                "Enumeration indicating if addQuoteItem succeeded or what kind of error occurred",
                $this->getAddQuoteItemResultTypeOptions()
            );

        case "AddQuoteItemBundleOption":
            return $this->inputObject("An object describing a selection for a bundle option");

        case "AddQuoteItemCustomOption":
            return $this->inputObject("An object describing for setting a custom option");

        case "AddQuoteItemCustomOptionValue":
            return $this->inputObject("An object describing the value of a custom option");

        case "ItemBundleOption":
            return $this->object("Object containing a bundle option and its selection")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "ItemBundleOptionSelection":
            return $this->object("A selected product and quantity for a bundle option");

        case "ItemConfigurableAttribute":
            return $this->object("An attribute which can be configured on configurable products");

        case "ItemConfigurableOption":
            return $this->object("A selected configuration option");

        case "ItemCustomOption":
            return $this->object("A selected custom option");

        case "ItemCustomOptionValue":
            return $this->object("A selected custom option value");

        case "ItemRowTotal":
            return $this->object("Contains information about the row price")
                ->setInterfaces(["Price"]);

        case "PlaceOrderResult":
            return $this->interface("Result of an attempt to submit an order")
                ->setTypeResolver(function(MageQL_Sales_Model_Order_AbstractResult $r) {
                    if($r instanceof MageQL_Sales_Model_Order_Result) {
                        return "PlaceOrderResultSuccess";
                    }

                    if($r instanceof MageQL_Sales_Model_Order_Error) {
                        return "PlaceOrderResultError";
                    }

                    throw new Exception(sprintf("Unknown order result type '%s'.", get_class($r)));
                });

        case "PlaceOrderResultError":
            return $this->object("Failure result of an attempt to submit an order")
                ->setInterfaces(["PlaceOrderResult"])
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PlaceOrderResultSuccess":
            return $this->object("Failure result of an attempt to submit an order")
                ->setInterfaces(["PlaceOrderResult"])
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PlaceOrderResultType":
            // Dispatch event to allow extra result types
            // TODO: Remove this whole type and just throw the exceptions
            $paymentManager = MageQL_Sales_Model_Payment::instance(Mage::app()->getStore());
            $placeOrderErrors = array_merge([
                MageQL_Sales_Model_Order_Result::SUCCESS => [
                    "description" => "Order was placed successfully.",
                ],
                MageQL_Sales_SubmitOrderException::NO_ITEMS => [
                    "description" => "The quote does not contain any items.",
                ],
                MageQL_Sales_SubmitOrderException::NO_PAYMENT_METHOD => [
                    "description" => "The quote does not have a payment method set.",
                ],
                MageQL_Sales_SubmitOrderException::PAYMENT_NOT_AVAILABLE => [
                    "description" => "The quote payment method is no longer available for the given quote.",
                ],
                MageQL_Sales_SubmitOrderException::INVALID_PAYMENT_METHOD => [
                    "description" => "The selected payment method is not valid.",
                ],
                MageQL_Sales_SubmitOrderException::INVALID_EMAIL_ADDRESS => [
                    "description" => "The supplied email on the quote is not valid.",
                ],
                MageQL_Sales_SubmitOrderException::NO_SHIPPING_METHOD => [
                    "description" => "The quote does not have a shipping method set",
                ],
                MageQL_Sales_SubmitOrderException::SHIPPING_ADDRESS_VALIDATION_FAILED => [
                    "description" => "The quote shipping address is not valid.",
                ],
                MageQL_Sales_SubmitOrderException::BILLING_ADDRESS_VALIDATION_FAILED => [
                    "description" => "The quote billing address is not valid.",
                ],
                MageQL_Sales_SubmitOrderException::PRODUCTS_NOT_AVAILABLE => [
                    "description" => "Not all products in quote are available in the requested quantity.",
                ],
                MageQL_Sales_SubmitOrderException::FRAUD_SHOP => [
                    "description" => "The whole shop has been marked as fraud by a rule",
                ],
                MageQL_Sales_SubmitOrderException::FRAUD_ORDER => [
                    "description" => "The quote has been marked as fraud by an order rule",
                ],
                MageQL_Sales_SubmitOrderException::FRAUD_PRODUCT => [
                    "description" => "The quote has been marked as fraud by a product rule",
                ],
                MageQL_Sales_SubmitOrderException::FRAUD_BLACKLIST => [
                    "description" => "The quote has been marked as fraud by a blacklist rule",
                ],
            ], $paymentManager->getPlaceOrderErrors());

            return $this->enum("Type of result from attempting to place an order", array_map(
                /**
                 * @param array{description:string} $e
                 */
                function($e) {
                    // Filter out any incompatible values
                    return [
                        "description" => $e["description"],
                    ];
                }, $placeOrderErrors));

        case "Quote":
            return $this->object("Object containing the current state of the checkout and cart")
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "QuoteAddress":
            return $this->interface("Address information for a quote")->setTypeResolver(
                /**
                 * @param Mage_Sales_Model_Quote_Address|Mage_Sales_Model_Order_Address $src
                 */
                function(Mage_Customer_Model_Address_Abstract $src) {
                    switch($src->getAddressType()) {
                    case Mage_Customer_Model_Address_Abstract::TYPE_BILLING:
                        return "QuoteAddressBilling";
                    case Mage_Customer_Model_Address_Abstract::TYPE_SHIPPING:
                        return "QuoteAddressShipping";
                    }
                }
            );

        case "QuoteAddressBilling":
            return $this->object("Billing address for the quote")
                ->setInterfaces(["QuoteAddress", "Address"])
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "QuoteAddressShipping":
            return $this->object("Billing address for the quote")
                ->setInterfaces(["QuoteAddress", "Address"])
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "QuoteAddressType":
            return $this->enum("Enumeration over address-types", [
                Mage_Customer_Model_Address_Abstract::TYPE_BILLING => [
                    "description" => "The address is a billing address",
                ],
                Mage_Customer_Model_Address_Abstract::TYPE_SHIPPING => [
                    "description" => "The address is a shipping address",
                ],
            ]);

        case "QuoteItem":
            return $this->interface("An item in the quote")
                ->setTypeResolver(function(Mage_Sales_Model_Quote_Item $src) {
                    switch($src->getProduct()->getTypeId()) {
                    case "configurable":
                        return "QuoteItemConfigurable";

                    case "bundle":
                        return "QuoteItemBundle";

                    case "simple":
                    case "virtual":
                        return "QuoteItemSimple";

                    default:
                        throw new Exception(sprintf(
                            "Unknown product type-id %s for quote item",
                            $src->getProduct()->getTypeId()
                        ));
                    }
                });

        case "QuoteItemBundle":
            return $this->object("An item in the quote with a bundle selection applied")
                ->setInterfaces(["QuoteItem"])
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "QuoteItemConfigurable":
            return $this->object("An item in the quote with a configuration option applied")
                ->setInterfaces(["QuoteItem"])
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "QuoteItemSimple":
            return $this->object("An item in the quote without any configuration options")
                ->setInterfaces(["QuoteItem"])
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "QuoteShipping":
            return $this->object("Contains information about shipping");

        case "QuoteShippingMethod":
            return $this->object("The selected shipping method");

        case "QuoteShippingPrice":
            return $this->object("The price of the selected method")
                ->setInterfaces(["Price"]);

        case "QuoteTotal":
            return $this->object("A total in the quote")
                ->setInterfaces(["Price"])
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "QuoteValidationError":
            return $this->object("A validation error for the quote")
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "TaxRate":
            return $this->object("A tax rate and amount");

        case "RemoveQuoteItemResult":
            return $this->object("Object containing the result of removing a quote item");

        case "RemoveQuoteItemResultType":
            return $this->enum("Enumeration indicating if removeQuoteItem succeeded or what kind of error occurred", [
                MageQL_Sales_Model_Quote_Item::SUCCESS => [
                    "description" => "Product was added to the quote successfully",
                ],
                MageQL_Sales_Model_Quote_Item::ERROR_DECODE => [
                    "description" => "The supplied buyRequest was invalid",
                ],
                MageQL_Sales_Model_Quote_Item::ERROR_ITEM_NOT_FOUND => [
                    "description" => "The supplied itemBuyRequest refers to a quote item which no longer exists",
                ],
                MageQL_Sales_Model_Quote_Item::ERROR_PRODUCT_NOT_FOUND => [
                    "description" => "The product to remove was not found",
                ],
            ]);

        case "SetQuoteBillingAddressPlaceholderResult":
            return $this->object("Object containing the result from setQuoteBillingAddressPlaceholder mutation")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "SetQuoteBillingAddressPlaceholderResultType":
            return $this->enum("Enumeration indicating the result of setQuoteBillingAddressPlaceholderResult mutation", [
                MageQL_Sales_Model_Quote_Address_Result_Placeholder::SUCCESS => [
                    "description" => "Successfully updated the address",
                ],
                MageQL_Sales_Model_Quote_Address_Result_Placeholder::NOT_MODIFIED => [
                    "description" => "Address was not modified, contains data",
                ],
                MageQL_Sales_Model_Quote_Address_Result_Placeholder::ERROR_IS_PHYSICAL => [
                    "description" => "Quote contains physical products, placeholder address not permitted",
                ],
                MageQL_Sales_Model_Quote_Address_Result_Placeholder::ERROR_DISABLED => [
                    "description" => "Setting placeholder billing address is disabled",
                ],
            ]);


        case "SetQuoteBillingAddressResult":
            return $this->object("Object containing the result from setQuoteBillingAddress mutation")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "SetQuoteBillingAddressResultType":
            return $this->enum("Enumeration indicating the result of setQuoteBillingAddress", [
                MageQL_Sales_Model_Quote_Address_Result::SUCCESS => [
                    "description" => "Address was successfully modified",
                ],
                MageQL_Sales_Model_Quote_Address_Result::NOT_MODIFIED => [
                    "description" => "Address was not modified",
                ],
            ]);

        case "SetQuoteBillingAddressAsShippingAddressResult":
            return $this->object("Object containing the result from setQuoteBillingAddressAsShippingAddress mutation")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "SetQuoteBillingAddressAsShippingAddressResultType":
            return $this->enum("Enumeration indicating the result of setQuoteBillingAddressAsShippingAddress", [
                MageQL_Sales_Model_Quote_Address_Result::SUCCESS => [
                    "description" => "Address was successfully set as shipping address",
                ],
            ]);

        case "SetQuoteDefaultAddressResult":
            return $this->object("Object containing the result from setQuoteAddressesToCustomerDefaults mutation");

        case "SetQuoteDefaultAddressResultType":
            return $this->enum("Enum indicating success for setting quote addresses to customer defaults", [
                MageQL_Sales_Model_Quote_Address_Result::SUCCESS => [
                    "description" => "Address was successfully modified",
                ],
                MageQL_Sales_Model_Quote_Address_Result::NOT_MODIFIED => [
                    "description" => "Address was not modified",
                ],
                MageQL_Sales_Model_Quote_Address_Result::ERROR_NOT_LOGGED_IN => [
                    "description" => "Customer is not logged in",
                ],
            ]);

        case "SetQuoteEmailResult":
            return $this->object("Object containing the result from setQuoteEmail mutation");

        case "SetQuoteEmailResultType":
            return $this->enum("Enum indicating success for setting quote email", [
                MageQL_Sales_Model_Quote::SUCCESS => [
                    "description" => "Successfully set the email address.",
                ],
                MageQL_Sales_Model_Quote::ERROR_INVALID_EMAIL_ADDRESS => [
                    "description" => "The supplied email address is invalid.",
                ],
                MageQL_Sales_Model_Quote::ERROR_CUSTOMER_IS_LOGGED_IN => [
                    "description" => "A customer is logged in and the quote email is locked, please update the customer email using updateCustomerEmail mutation instead.",
                ],
            ]);

        case "SetQuoteAddressInput":
            return $this->inputObject("Object containing a partially or fully populated address");

        case "SetQuoteShippingAddressResult":
            return $this->object("Object containing the result from setQuoteBillingAddress mutation")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "SetQuoteShippingAddressResultType":
            return $this->enum("Enumeration indicating the result of setQuoteBillingAddress", [
                MageQL_Sales_Model_Quote_Address_Result::SUCCESS => [
                    "description" => "Address was successfully modified",
                ],
                MageQL_Sales_Model_Quote_Address_Result::NOT_MODIFIED => [
                    "description" => "Address was not modified",
                ],
            ]);

        case "SetQuoteShippingMethodResult":
            return $this->object("Object containing the result of attempting to set the shipping method");

        case "SetQuoteShippingMethodResultType":
            return $this->enum(
                "Enumeration indicating if setQuoteShippingMethod or setQuoteShippingMethodToCheapest succeeded",
                [
                    MageQL_Sales_Model_Quote_Shipping::SUCCESS => [
                        "description" => "Shipping method was set successfully",
                    ],
                    MageQL_Sales_Model_Quote_Shipping::NOT_MODIFIED => [
                        "description" => "Shipping method was not modified",
                    ],
                    MageQL_Sales_Model_Quote_Shipping::NO_ITEMS => [
                        "description" => "The quote did not contain any items which require shipping",
                    ],
                    MageQL_Sales_Model_Quote_Shipping::NOT_FOUND => [
                        "description" => "No available shipping methods were found",
                    ],
                ]
            );

        case "ShippingMethod":
            return $this->object("Object containing information about a shipping method option")
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "ShippingMethodPrice":
            return $this->object("Contains information about the shipping method price")
                ->setInterfaces(["Price"]);

        case "UpdateQuoteItemQtyResult":
            return $this->object("Object containing the result of updating a quote item");

        case "UpdateQuoteItemQtyResultType":
            // TODO: Make this enum extendable
            return $this->enum(
                "Enumeration indicating if updateQuoteItemQty succeeded or what kind of error occurred",
                array_merge($this->getAddQuoteItemResultTypeOptions(), [
                    MageQL_Sales_Model_Quote_Item::ERROR_ITEM_NOT_FOUND => [
                        "description" => "The supplied itemBuyRequest refers to a quote item which no longer exists",
                    ],
                ])
            );
        }

        return null;
    }

    public function getTypeFields(string $typeName, Registry $registry): array {
        switch($typeName) {
        case "AddQuoteItemResult":
            return [
                "result" => $this->field("AddQuoteItemResultType!", "Type of AddQuoteItemResult")
                    ->setResolver(
                        /**
                         * @param string|Mage_Sales_Model_Quote_Item $src
                         */
                        function($src) {
                            return is_string($src) ? $src : MageQL_Sales_Model_Quote_Item::SUCCESS;
                       }),
            ];

        case "AddQuoteItemCustomOption":
            return [
                "optionId" => $this->field("ID!", "The option id"),
                "value" => $this->field("AddQuoteItemCustomOptionValue!", "Value for option; one of selectionId or text must be set"),
            ];
        case "AddQuoteItemCustomOptionValue":
            return [
                "selectionId" => $this->field("Int", "Selected value for option; must be set for 'select' option types"),
                "text" => $this->field("String", "Text content for option; must be set for 'text' option types"),
            ];

        case "AddQuoteItemBundleOption":
            return [
                "optionId" => $this->field("ID!", "The option id"),
                "qty" => $this->field("Float", "Selection quantity, if different from default"),
                "selectionId" => $this->field("ID", "The selections for this option, null if no option should be selected (eg. if a default option-selection should not be picked)."),
            ];

        case "ItemBundleOption":
            return [
                "type" => $this->field("BundleOptionType!", "The type of input used for this option"),
                "title" => $this->field("String!", "Title for the option"),
                "products" => $this->field("[ItemBundleOptionSelection!]!", "The selected products for this option"),
            ];

        case "ItemBundleOptionSelection":
            return [
                "product" => $this->field("ListProduct!", "The selected products")
                    ->setResolver(function(array $a): Mage_Catalog_Model_Product {
                        return $a[0];
                    }),
                "qty" => $this->field("Float!", "The selected quantity")
                    ->setResolver(function(array $a): float {
                        return $a[1];
                    }),
            ];

        case "ItemConfigurableAttribute":
            return [
                "attribute" => $this->field("String!", "Attribute property")
                    ->setResolver("MageQL_Sales_Model_Item::resolveConfigurableAttributeCode"),
                "label" => $this->field("String!", "Attribute label")
                    ->setResolver("MageQL_Sales_Model_Item::resolveConfigurableAttributeLabel"),
                "value" => $this->field("String!", "Attribute value picked for the configuration")
                    ->setResolver("MageQL_Sales_Model_Item::resolveConfigurableAttributeValue"),
            ];

        case "ItemConfigurableOption":
            return [
                "attributes" => $this->field("[ItemConfigurableAttribute!]!", "List of configurable attributes and their selected options")
                    ->setResolver("MageQL_Sales_Model_Item::resolveConfigurableOptionAttributes"),
                "product" => $this->field("ListProduct!", "Selected product option")
                    ->setResolver("MageQL_Sales_Model_Item::resolveConfigurableOptionProduct"),
            ];

        case "ItemCustomOption":
            return [
                "option" => $this->field("ProductCustomOption!", "Custom option used on item"),
                "value" => $this->field("ItemCustomOptionValue!", "Custom option value used on item; one of selected or text is set"),
            ];

        case "ItemCustomOptionValue":
            return [
                "selected" => $this->field("ProductCustomOptionValue", "Selected value for option; is set for 'select' option types"),
                "text" => $this->field("String", "Text content for option; is set for 'text' option types"),
            ];

        case "ItemRowTotal":
            return [
                "exVat" => $this->field("Float!", "Price excluding VAT")
                    ->setResolver("MageQL_Sales_Model_Item::resolveRowTotalExVat"),
                "incVat" => $this->field("Float!", "Price including VAT")
                    ->setResolver("MageQL_Sales_Model_Item::resolveRowTotalIncVat"),
                "vat" => $this->field("Float!", "VAT amount")
                    ->setResolver("MageQL_Sales_Model_Item::resolveRowTotalVat"),
            ];

        case "ListProduct":
            return [
                "buyRequest" => $this->field("ID", "A buyRequest which can be used to add this product to cart, will only be present on this level for simple and virtual products which are saleable, or bundle products which have all required options satisifed by defaults.")
                    ->setResolver("MageQL_Sales_Model_Product::resolveBuyRequestSimple"),
            ];

        case "ListProductBundle":
            return [
                "bundleBuyRequestBase" => $this->field("ID!", "A basic buy request to be completed with bundle options.")
                    ->setResolver("MageQL_Sales_Model_Product::resolveBuyRequestBaseBundle"),
            ];

        case "Mutation":
            return [
                // TODO: Replace the array with something better typed
                "addQuoteItem" => $this->field(
                    "AddQuoteItemResult!",
                    "Attempts to add a product to the session quote, will replace any existing row with the same product"
                )
                    ->addArgument("buyRequest", $this->argument("ID!", "A buy request from a product or product variant"))
                    ->addArgument("qty", $this->argument("Int!", "The requested quantity"))
                    ->addArgument("bundleOptions", $this->argument("[AddQuoteItemBundleOption!]", "If the buyRequest is from a bundle product this contains the configuration for the product. Only required if there are required non-default options."))
                    ->addArgument("customOptions", $this->argument("[AddQuoteItemCustomOption!]", "If the buyRequest contains custom options. Only required if there are required non-default options."))
                    ->setResolver("MageQL_Sales_Model_Quote_Item::mutateAdd"),
                "updateQuoteItemQty" => $this->field(
                    "UpdateQuoteItemQtyResult!",
                    "Attempts to update an existing product item in the quote, it will overwrite the quantity."
                )
                    ->addArgument("itemBuyRequest", $this->argument("ID!", "The quote item identifier"))
                    ->addArgument("qty", $this->argument("Int!", "The requested quantity"))
                    ->setResolver("MageQL_Sales_Model_Quote_Item::mutateUpdateQty"),
                "removeQuoteItem" => $this->field(
                    "RemoveQuoteItemResult!",
                    "Attempts to remove an item in the quote."
                )
                    ->addArgument("itemBuyRequest", $this->argument("ID!", "The quote item identifier"))
                    ->setResolver("MageQL_Sales_Model_Quote_Item::mutateRemove"),
                "removeQuote" => $this->field(
                    "Boolean!",
                    "Removes the current quote and marks it for deletion."
                )
                    ->setResolver("MageQL_Sales_Model_Quote::mutateRemove"),
                "removeUnorderableQuoteItems" => $this->field(
                    "[QuoteItem!]!",
                    "Removes any items which cannot be ordered based on available stock quantities."
                )
                    ->setResolver("MageQL_Sales_Model_Quote::mutateRemoveUnorderableItems"),
                "setQuoteAddressesToCustomerDefaults" => $this->field("SetQuoteDefaultAddressResult!", "Attempts to set the quote billing- and shipping-address to the customer default addresses")
                    ->addArgument("billing", $this->argument("Boolean", "If to set the billing address to the default billing address")->setDefaultValue(true))
                    ->addArgument("shipping", $this->argument("Boolean", "If to set the shipping address to the default shipping address")->setDefaultValue(true))
                    ->setResolver("MageQL_Sales_Model_Quote_Address::mutateSetToCustomerDefaults"),
                "setQuoteBillingAddress" => $this->field(
                    "SetQuoteBillingAddressResult!",
                    "Updates the billing address on the quote. If the billing adress is being used as the shipping address any modifications using this mutation will propagate to the shipping address."
                )
                    ->addArgument("address", $this->argument("SetQuoteAddressInput!", "A partially or fully populated billing address"))
                    ->setResolver("MageQL_Sales_Model_Quote_Address::mutateSetBilling"),
                "setQuoteBillingAddressAsShippingAddress" => $this->field(
                    "SetQuoteBillingAddressAsShippingAddressResult!",
                    "Makes the shipping address use the billing address information. Any modifications to the billing address will be automatically propagated to the shipping address. If the shipping address is later modified this flag will be turned off until this mutation is run again."
                )
                    ->setResolver("MageQL_Sales_Model_Quote_Address::mutateUseBillingAsShipping"),
                "setQuoteEmail" => $this->field("SetQuoteEmailResult!", "Sets the email on the quote")
                    ->addArgument("email", $this->argument("String!", "The new email address"))
                    ->setResolver("MageQL_Sales_Model_Quote::mutateSetEmail"),
                "setQuotePaymentMethod" => $this->field(
                    "MutationSetQuotePaymentMethod!",
                    "Sets the selected nested field as the selected payment method for the quote. Only use one mutation"
                )
                    ->setResolver("MageQL_Sales_Model_Quote::resolve"),
                "setQuoteShippingAddress" => $this->field(
                    "SetQuoteShippingAddressResult!",
                    sprintf("Updates the shipping address on the quote. If the billing address was set to be used as shipping address this will turn that setting off if the shipping address is actually modified (ie. a '%s' response).", MageQL_Sales_Model_Quote_Address_Result::SUCCESS)
                )
                    ->addArgument("address", $this->argument("SetQuoteAddressInput!", "A partially or fully populated billing address"))
                    ->setResolver("MageQL_Sales_Model_Quote_Address::mutateSetShipping"),
                "setQuoteShippingMethod" => $this->field(
                    "SetQuoteShippingMethodResult!",
                    "Attempts to set the shipping method to the supplied shipping method code"
                )
                    ->addArgument("shippingMethodCode", $this->argument("ID!", "The shipping method code to pick"))
                    ->setResolver("MageQL_Sales_Model_Quote_Shipping::mutateSet"),
                "setQuoteShippingMethodToCheapest" => $this->field(
                    "SetQuoteShippingMethodResult!",
                    "Attempts to find the cheapest shipping method and set it"
                )
                    ->addArgument("overwrite",
                        $this->argument("Boolean", "If to overwrite an existing shipping method, default false")
                            ->setDefaultValue(false)
                    )
                    ->setResolver("MageQL_Sales_Model_Quote_Shipping::mutateSetToCheapest"),
                "placeOrder" => $this->field(
                    "PlaceOrderResult!",
                    "Attempts to place an order with the current session quote."
                )
                    ->setResolver("MageQL_Sales_Model_Quote::mutatePlaceOrder"),
                "setQuoteBillingAddressPlaceholder" => $this->field(
                    "SetQuoteBillingAddressPlaceholderResult!",
                    "Attempts to set dummy billing address."
                )
                    ->setResolver("MageQL_Sales_Model_Quote_Address::mutateSetBillingPlaceholder"),
            ];

        case "PlaceOrderResult":
            return [
                // TODO: More data
                "result" => $this->field("PlaceOrderResultType!", "Result of the attempt to place the order"),
            ];

        case "PlaceOrderResultError":
            return $registry->getFieldBuilders("PlaceOrderResult");

        case "PlaceOrderResultSuccess":
            return array_merge($registry->getFieldBuilders("PlaceOrderResult"), [
                "order" => $this->field("Order!", "The newly placed order"),
                "paymentRedirectUrl" => $this->field("String", "Payment gateway URL which is used to complete the order if not null"),
            ]);

        case "ProductCustomOptionValue":
            return [
                "buyRequest" => $this->field("ID", "A buyRequest which can be used to add this product to cart, with this custom option variant.")
                    ->setResolver("MageQL_Sales_Model_Product::resolveBuyRequestSimpleCustomOption"),
            ];

        case "ProductDetail":
            return [
                // TODO: Should this really be a part of the base type, and not virtual/simple?
                "buyRequest" => $this->field("ID", "A buyRequest which can be used to add this product to cart, will only be present on this level for simple and virtual products which are saleable, or bundle products which have all required options satisifed by defaults.")
                    ->setResolver("MageQL_Sales_Model_Product::resolveBuyRequestSimple"),
            ];

        case "ProductDetailBundle":
            return [
                "bundleBuyRequestBase" => $this->field("ID!", "A basic buy request to be completed with bundle options.")
                    ->setResolver("MageQL_Sales_Model_Product::resolveBuyRequestBaseBundle"),
            ];

        case "ConfigurationOptionItem":
            return [
                "buyRequest" => $this->field("ID", "A buyRequest which can be used to add this product to cart, present when saleable")
                    ->setResolver("MageQL_Sales_Model_Product::resolveBuyRequestConfigItem"),
            ];

        case "Query":
            return [
                // TODO: "checkoutAgreements" => $this->field("[CheckoutAgreement!]!", "List of agreements, some of which are mandatory")
                "quote" => $this->field("Quote!", "The current quote.")
                    ->setResolver("MageQL_Sales_Model_Quote::resolve"),
                "shippingMethods" => $this->field(
                    "[ShippingMethod!]!",
                    "A list of shipping methods for the store, with price information and availability depending on the current quote."
                )
                    ->setresolver("MageQL_Sales_Model_Shipping::resolveAll"),
            ];

        case "Quote":
            return [
                // TODO: Discount
                // TODO: Coupon
                // TODO: hasVirtualItems?
                "addresses" => $this->field("[QuoteAddress!]!", "List of addresses related to this quote")
                    ->setResolver("MageQL_Sales_Model_Quote::resolveAddresses"),
                "email" => $this->field("String", "Quote email address")
                    ->setResolver("MageQL_Sales_Model_Quote::resolveEmail"),
                "currencyCode" => $this->field("String!", "Currency code used by the quote"),
                "grandTotal" => $this->field("QuoteTotal!", "The total amount for the quote")
                    ->setResolver("MageQL_Sales_Model_Quote::resolveGrandTotal"),
                "items" => $this->field("[QuoteItem!]!", "The items in the cart")
                    ->setResolver("MageQL_Sales_Model_Quote::resolveItems"),
                "isVirtual" => $this->field("Boolean!", "If true this quote does not have any physical items present")
                    ->setResolver("MageQL_Sales_Model_Quote::resolveIsVirtual"),
                "payment" => $this->field("QuotePayment", "Payment data if any payment has been selected")
                    ->setResolver("MageQL_Sales_Model_Quote_Payment::resolve"),
                "shipping" => $this->field("QuoteShipping", "Current shipping information for the quote")
                    ->setResolver("MageQL_Sales_Model_Quote::resolveShipping"),
                "subTotal" => $this->field("QuoteTotal!", "The sub-total amount for the quote")
                    ->setResolver("MageQL_Sales_Model_Quote::resolveSubTotal"),
                "taxRates" => $this->field("[TaxRate!]!", "Tax rates and their amounts in the quote")
                    ->setResolver("MageQL_Sales_Model_Quote::resolveTaxRates"),
                "validationErrors" => $this->field("[QuoteValidationError!]!", "List of validation errors for the quote, does not include payment-related verifications")
                    ->setResolver("MageQL_Sales_Model_Quote::resolveValidationErrors"),
            ];

        case "QuoteAddress":
            return array_merge([
                "type" => $this->field("QuoteAddressType!", "Type of address")
                    ->setResolver("MageQL_Sales_Model_Quote_Address::resolveType"),
            ], $registry->getFieldBuilders("Address"));

        case "QuoteAddressBilling":
            return array_merge($registry->getFieldBuilders("QuoteAddress"), [
                "isUsedAsShipping" => $this->field("Boolean!", "If true, then this address is also used for shipping, or it is a virtual quote")
                    ->setResolver("MageQL_Sales_Model_Quote_Address::resolveIsUsedAsShipping"),
            ]);

        case "QuoteAddressShipping":
            return array_merge($registry->getFieldBuilders("QuoteAddress"), [
                // TODO: Stuff?
            ]);

        case "QuoteItem":
            return [
                "itemBuyRequest" => $this->field("ID!", "Buy request to modify or delete this item")
                    ->setResolver("MageQL_Sales_Model_BuyRequest::fromQuoteItem"),
                "qty" => $this->field("Int!", "Quantity of this product item in the quote"),
                "rowTotal" => $this->field("ItemRowTotal!", "Row total")
                    ->setResolver("MageQL\\forwardResolver"),
                "canOrder" => $this->field("Boolean!", "If this item row can be ordered in the requested quantity")
                    ->setResolver("MageQL_Sales_Model_Quote_Item::resolveCanOrder"),
                // TODO: Should we use this? This is the actual SKU (aka. child-product sku for config)
                // "sku" => $this->field("String!", "SKU of the selected product"),
                "product" => $this->field("ListProduct!", "The selected product")
                    ->setResolver("MageQL_Sales_Model_Item::resolveProduct"),
                "customOptions" => $this->field("[ItemCustomOption!]", "Custom options set on item")
                    ->setResolver("MageQL_Sales_Model_Item::resolveCustomOptions"),
            ];

        case "QuoteItemBundle":
            return array_merge($registry->getFieldBuilders("QuoteItem"), [
                "bundleOptions" => $this->field("[ItemBundleOption!]!", "The selected bundle options")
                    ->setResolver("MageQL_Sales_Model_Item::resolveQuoteBundleOptions"),
            ]);

        case "QuoteItemConfigurable":
            return array_merge($registry->getFieldBuilders("QuoteItem"), [
                "configOption" => $this->field("ItemConfigurableOption!", "The selected product option")
                    ->setResolver("MageQL_Sales_Model_Item::resolveQuoteConfigOption"),
            ]);

        case "QuoteItemSimple":
            return $registry->getFieldBuilders("QuoteItem");

        case "QuoteShipping":
            return [
                "method" => $this->field("QuoteShippingMethod!", "Selected method for shipping")
                    ->setResolver("MageQL\\forwardResolver"),
                "total" => $this->field("QuoteShippingPrice!", "Total for shipping")
                    ->setResolver("MageQL\\forwardResolver"),
            ];

        case "QuoteShippingPrice":
            return [
                "exVat" => $this->field("Float!", "Price excluding VAT")
                    ->setResolver("MageQL_Sales_Model_Quote_Shipping::resolvePriceExVat"),
                "incVat" => $this->field("Float!", "Price including VAT")
                    ->setResolver("MageQL_Sales_Model_Quote_Shipping::resolvePriceIncVat"),
                "vat" => $this->field("Float!", "VAT amount")
                    ->setResolver("MageQL_Sales_Model_Quote_Shipping::resolvePriceVat"),
            ];

        case "QuoteShippingMethod":
            return [
                "code" => $this->field("ID!", "Shipping code")
                    ->setResolver("MageQL_Sales_Model_Quote_Shipping::resolveMethod"),
                "description" => $this->field("String", "Shipping method description")
                    ->setResolver("MageQL_Sales_Model_Quote_Shipping::resolveDescription"),
            ];

        case "QuoteTotal":
            return [
                "exVat" => $this->field("Float!", "Price excluding VAT"),
                "incVat" => $this->field("Float!", "Price including VAT"),
                "vat" => $this->field("Float!", "VAT amount"),
            ];

        case "QuoteValidationError":
            return [
                "code" => $this->field("String!", "Error code"),
                "message" => $this->field("String!", "Error message"),
            ];

        case "RemoveQuoteItemResult":
            return [
                "result" => $this->field("RemoveQuoteItemResultType!", "Type of RemoveQuoteItemResult")
                    ->setResolver("MageQL\\forwardResolver"),
            ];

        case "SetQuoteBillingAddressPlaceholderResult":
            return [
                "result" => $this->field("SetQuoteBillingAddressPlaceholderResultType!", "Type of SetQuoteBillingAddressPlaceholderResult"),
                "address" => $this->field("QuoteAddressBilling", "The address after updates have been performed."),
                "validationErrors" => $this->field("[AddressValidationError!]!", "List of validation errors from the newly updated address"),
            ];

        case "SetQuoteBillingAddressResult":
            return [
                "result" => $this->field("SetQuoteBillingAddressResultType!", "Type of SetQuoteBillingAddressResult"),
                "address" => $this->field("QuoteAddressBilling!", "The address after updates have been performed."),
                "validationErrors" => $this->field("[AddressValidationError!]!", "List of validation errors from the newly updated address")
            ];

        case "SetQuoteBillingAddressAsShippingAddressResult":
            return [
                "result" => $this->field("SetQuoteBillingAddressAsShippingAddressResultType!", "Type of SetQuoteBillingAddressAsShippingAddressResult"),
                "address" => $this->field("QuoteAddressBilling!", "The address after updates have been performed."),
                "validationErrors" => $this->field("[AddressValidationError!]!", "List of validation errors from the newly updated address"),
            ];

        case "SetQuoteDefaultAddressResult":
            return [
                "result" => $this->field("SetQuoteDefaultAddressResultType!", "Type of SetQuoteEmailResult")
                    ->setResolver("MageQL\\forwardResolver"),
            ];

        case "SetQuoteEmailResult":
            return [
                "result" => $this->field("SetQuoteEmailResultType!", "Type of SetQuoteEmailResult")
                    ->setResolver("MageQL\\forwardResolver"),
            ];

        case "SetQuoteAddressInput":
            return Mage::getSingleton("mageql_sales/form_address")->getInputFields();

        case "SetQuoteShippingAddressResult":
            return [
                "result" => $this->field("SetQuoteShippingAddressResultType!", "Type of SetQuoteShippingAddressResult"),
                "address" => $this->field("QuoteAddressShipping!", "The address after updates have been performed."),
                "validationErrors" => $this->field("[AddressValidationError!]!", "List of validation errors from the newly updated address"),
            ];

        case "SetQuoteShippingMethodResult":
            return [
                "result" => $this->field("SetQuoteShippingMethodResultType!", "Type of SetQuoteShippingMethodResult")
                    ->setResolver("MageQL\\forwardResolver"),
            ];

        case "ShippingMethod":
            return [
                "code" => $this->field("ID!", "The shipping method code"),
                "carrierTitle" => $this->field("String!", "The shipping carrier title"),
                "methodTitle" => $this->field("String!", "The method title"),
                "methodDescription" => $this->field("String", "The method description"),
                "price" => $this->field("ShippingMethodPrice!", "The price of using this shipping method with the current quote")
                    ->setResolver("MageQL\\forwardResolver"),
            ];

        case "ShippingMethodPrice":
            return [
                "exVat" => $this->field("Float!", "Price excluding VAT")
                    ->setResolver("MageQL_Sales_Model_Shipping::resolvePriceExVat"),
                "incVat" => $this->field("Float!", "Price including VAT")
                    ->setResolver("MageQL_Sales_Model_Shipping::resolvePriceIncVat"),
                "vat" => $this->field("Float!", "VAT amount")
                    ->setResolver("MageQL_Sales_Model_Shipping::resolvePriceVat"),
            ];

        case "TaxRate":
            return [
                "percent" => $this->field("Float!", "The tax rate in percentage-units")
                    ->setResolver("MageQL_Sales_Model_Quote::resolveTaxRatePercent"),
                "amount" => $this->field("Float!", "The amount using this tax rate")
                    ->setResolver("MageQL_Sales_Model_Quote::resolveTaxRateAmount"),
            ];

        case "UpdateQuoteItemQtyResult":
            return [
                "result" => $this->field("UpdateQuoteItemQtyResultType!", "Type of UpdateQuoteItemQtyResult")
                    ->setResolver(
                        /**
                         * @param string|Mage_Sales_Model_Quote_Item $src
                         */
                        function($src) {
                            return is_string($src) ? $src : MageQL_Sales_Model_Quote_Item::SUCCESS;
                        }),
            ];
        }

        return [];
    }

    public function getUnreachableTypes(): array {
        return [
            "PlaceOrderResultError",
            "PlaceOrderResultSuccess",
            "QuoteAddressBilling",
            "QuoteAddressShipping",
            "QuoteItemBundle",
            "QuoteItemConfigurable",
            "QuoteItemSimple",
        ];
    }
}
