<?php

declare(strict_types=1);

namespace MageQL\Sales;

use Mage;
use Mage_Core_Model_Resource_Setup;
use Mage_Core_Model_Store;
use Throwable;
use Varien_Object;

use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use PHPUnit\Framework\TestCase;
use Spatie\Snapshots\MatchesSnapshots;

class OrderTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
        MagentoManager::reset();
        MagentoManager::initAdmin();

        // Got to reset these
        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");

        $setupModel->deleteConfigData("carriers/freeshipping/active", "websites");
        $setupModel->deleteConfigData("carriers/freeshipping/free_shipping_subtotal", "websites");

        /** @var Mage_Core_Model_Store $store */
        $store = Mage::app()->getStore();

        // Reset stock on the product, modified by testPlaceOrderOutOfStock
        $product = Mage::getModel("catalog/product")
            ->setStoreId($store->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        Mage::getModel("cataloginventory/stock_item")
            ->loadByProduct($product)
            ->addData([
                "is_in_stock" => 1,
                "qty" => 999,
            ])
            ->save();

        // we have to reindex the stock to not affect the other tests
        MagentoManager::reset();
        MagentoManager::reindex();
    }

    public function testNoLastOrder(): void {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            lastOrder {
                status
                id
                items {
                    qty
                    rowTotal {
                        incVat
                    }
                    product {
                        sku
                        attributes {
                            shortDescription
                            smallImage {
                                src
                            }
                        }
                    }
                }
                addresses {
                    type
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    company
                    street
                    city
                    postcode
                    region {
                        code
                        name
                    }
                    country {
                        code
                    }
                }
                email
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testLastOrder(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $order = Mage::getModel("sales/order")->loadByIncrementId("TEST-1");
        $orderId = (int)$order->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setLastOrderId($orderId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            lastOrder {
                status
                id
                items {
                    qty
                    rowTotal {
                        incVat
                    }
                    product {
                        sku
                        name
                        attributes {
                            shortDescription
                            smallImage {
                                src
                            }
                        }
                    }
                    ... on OrderItemConfigurable {
                        configOption {
                            attributes {
                                attribute
                                label
                                value
                            }
                            product {
                                sku
                                name
                                attributes {
                                    shortDescription
                                        smallImage {
                                        src
                                    }
                                }
                            }
                        }
                    }
                    ... on OrderItemBundle {
                        bundleOptions {
                            type
                            title
                            products {
                                qty
                                product {
                                    sku
                                    name
                                    attributes {
                                        shortDescription
                                        smallImage {
                                            src
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
                addresses {
                    type
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    company
                    street
                    city
                    postcode
                    region {
                        code
                        name
                    }
                    country {
                        code
                    }
                }
                email
                discountTotal
                grandTotal {
                    incVat
                    exVat
                    vat
                }
                subTotal {
                    incVat
                    exVat
                    vat
                }
                currencyCode
                virtual
                shipping {
                    method {
                        code
                        description
                    }
                    total {
                        incVat
                        exVat
                        vat
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testLastOrderInvalidId(): void {
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setLastOrderId(1234567890);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            lastOrder {
                status
                id
                items {
                    qty
                    rowTotal {
                        incVat
                    }
                    product {
                        sku
                        attributes {
                            shortDescription
                            smallImage {
                                src
                            }
                        }
                    }
                }
                addresses {
                    type
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    company
                    street
                    city
                    postcode
                    region {
                        code
                        name
                    }
                    country {
                        code
                    }
                }
                email
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testPlaceOrder(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();


        // Decrease so we get a zero subtotal checkout
        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");
        /** @var Mage_Core_Model_Store $store */
        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);

        $setupModel->setConfigData("carriers/freeshipping/free_shipping_subtotal", 0, "websites", (int)$store->getWebsiteId());

        MagentoManager::reset();
        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId($store->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        // Add the product and then modify the item to zero the price
        $item = $quote->addProduct($product, $request);

        $this->assertIsObject($item);

        $item->setOriginalCustomPrice(0)
            ->setCustomPrice(0)
            ->setIsSuperMode(true);

        $quote->getPayment()->importData([
            "method" => "free",
        ]);
        $quote->getShippingAddress()->setShippingMethod("freeshipping_freeshipping");
        $quote->setCustomerEmail("test@example.com");

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = (int)$quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            placeOrder {
                result
                ...on PlaceOrderResultSuccess {
                    paymentRedirectUrl
                    order {
                        status
                        items {
                            qty
                            rowTotal {
                                incVat
                            }
                            product {
                                sku
                                attributes {
                                    shortDescription
                                    smallImage {
                                        src
                                    }
                                }
                            }
                        }
                        addresses {
                            type
                            prefix
                            firstname
                            middlename
                            lastname
                            suffix
                            company
                            street
                            city
                            postcode
                            region {
                                code
                                name
                            }
                            country {
                                code
                            }
                        }
                        email
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testPlaceOrderInvalidPayment(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        /** @var Mage_Core_Model_Store $store */
        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId($store->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        // Add the product and then modify the item to zero the price
        $item = $quote->addProduct($product, $request);

        $this->assertIsObject($item);

        $item->setOriginalCustomPrice(0)
            ->setCustomPrice(0)
            ->setIsSuperMode(true);

        $quote->getPayment()->importData([
            "method" => "free",
        ]);
        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");
        $quote->setCustomerEmail("test@example.com");

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = (int)$quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            placeOrder {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testPlaceOrderNoItems(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));

        $quote->getPayment()->importData([
            "method" => "free",
        ]);
        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");
        $quote->setCustomerEmail("test@example.com");

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = (int)$quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            placeOrder {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testPlaceOrderFaultyShipping(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        /** @var Mage_Core_Model_Store $store */
        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId($store->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        // Add the product and then modify the item to zero the price
        $item = $quote->addProduct($product, $request);

        $this->assertIsObject($item);

        $item->setOriginalCustomPrice(0)
            ->setCustomPrice(0)
            ->setIsSuperMode(true);

        $quote->getPayment()->importData([
            "method" => "free",
        ]);
        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");
        $quote->setCustomerEmail("test@example.com");

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = (int)$quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            placeOrder {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testPlaceOrderMissingShippingAddress(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        /** @var Mage_Core_Model_Store $store */
        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId($store->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        // Add the product and then modify the item to zero the price
        $item = $quote->addProduct($product, $request);

        $this->assertIsObject($item);

        $item->setOriginalCustomPrice(0)
            ->setCustomPrice(0)
            ->setIsSuperMode(true);

        $quote->getPayment()->importData([
            "method" => "free",
        ]);
        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");
        $quote->setCustomerEmail("test@example.com");

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = (int)$quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            placeOrder {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testPlaceOrderMissingShipping(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        /** @var Mage_Core_Model_Store $store */
        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId($store->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        // Add the product and then modify the item to zero the price
        $item = $quote->addProduct($product, $request);

        $this->assertIsObject($item);

        $item->setOriginalCustomPrice(0)
            ->setCustomPrice(0)
            ->setIsSuperMode(true);

        $quote->getPayment()->importData([
            "method" => "free",
        ]);
        $quote->setCustomerEmail("test@example.com");

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = (int)$quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            placeOrder {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testPlaceOrderMissingPayment(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        /** @var Mage_Core_Model_Store $store */
        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId($store->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        // Add the product and then modify the item to zero the price
        $item = $quote->addProduct($product, $request);

        $this->assertIsObject($item);

        $item->setOriginalCustomPrice(0)
            ->setCustomPrice(0)
            ->setIsSuperMode(true);

        // It is not a virtual quote, use shipping address for payment
        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");
        $quote->setCustomerEmail("test@example.com");

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = (int)$quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            placeOrder {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testPlaceOrderInvalidBilling(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        /** @var Mage_Core_Model_Store $store */
        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId($store->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        // Add the product and then modify the item to zero the price
        $item = $quote->addProduct($product, $request);

        $this->assertIsObject($item);

        $item->setOriginalCustomPrice(0)
            ->setCustomPrice(0)
            ->setIsSuperMode(true);

        $quote->getPayment()->importData([
            "method" => "free",
        ]);
        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");
        $quote->setCustomerEmail("test@example.com");

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = (int)$quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            placeOrder {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testPlaceOrderInvalidEmail(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // Decrease so we get a zero subtotal checkout
        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");
        /** @var Mage_Core_Model_Store $store */
        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);

        $setupModel->setConfigData("carriers/freeshipping/free_shipping_subtotal", 0, "websites", (int)$store->getWebsiteId());

        MagentoManager::reset();
        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId($store->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        // Add the product and then modify the item to zero the price
        $item = $quote->addProduct($product, $request);

        $this->assertIsObject($item);

        $item->setOriginalCustomPrice(0)
            ->setCustomPrice(0)
            ->setIsSuperMode(true);

        $quote->getPayment()->importData([
            "method" => "free",
        ]);
        $quote->getShippingAddress()->setShippingMethod("freeshipping_freeshipping");
        $quote->setCustomerEmail("testfail");

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = (int)$quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            placeOrder {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testPlaceOrderOutOfStock(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // Decrease so we get a zero subtotal checkout
        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");
        /** @var Mage_Core_Model_Store $store */
        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);

        $setupModel->setConfigData("carriers/freeshipping/free_shipping_subtotal", 0, "websites", (int)$store->getWebsiteId());

        MagentoManager::reset();
        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId($store->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        // Add the product and then modify the item to zero the price
        $item = $quote->addProduct($product, $request);

        $this->assertIsObject($item);

        $item->setOriginalCustomPrice(0)
            ->setCustomPrice(0)
            ->setIsSuperMode(true);

        $quote->getPayment()->importData([
            "method" => "free",
        ]);
        $quote->getShippingAddress()->setShippingMethod("freeshipping_freeshipping");
        $quote->setCustomerEmail("test@example.com");

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = (int)$quote->getId();

        // Update stock on the product
        Mage::getModel("cataloginventory/stock_item")
            ->loadByProduct($product)
            ->addData([
                "is_in_stock" => 0,
                "qty" => 0,
            ])
            ->save();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            placeOrder {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testCustomerOrders(): void {
        MagentoManager::init();

        /** @var Mage_Core_Model_Store $store */
        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);

        $customer = Mage::getModel("customer/customer")
            ->setWebsiteId($store->getWebsiteId())
            ->loadByEmail("test-customer@example.com");

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], "query {
            customer {
                orders {
                    items {
                        id
                        status
                        items {
                            qty
                            rowTotal {
                                incVat
                            }
                            product {
                                sku
                                attributes {
                                    shortDescription
                                    smallImage {
                                        src
                                    }
                                }
                            }
                        }
                        addresses {
                            type
                            prefix
                            firstname
                            middlename
                            lastname
                            suffix
                            company
                            street
                            city
                            postcode
                            region {
                                code
                                name
                            }
                            country {
                                code
                            }
                        }
                        email
                        createdAt
                        discountTotal
                        grandTotal {
                            incVat
                            exVat
                            vat
                        }
                        subTotal {
                            incVat
                            exVat
                            vat
                        }
                        currencyCode
                        virtual
                        shipping {
                            method {
                                code
                                description
                            }
                            total {
                                incVat
                                exVat
                                vat
                            }
                        }
                    }
                }
            }
        }"));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testCustomerOrderById(): void {
        MagentoManager::init();

        /** @var Mage_Core_Model_Store $store */
        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);

        $customer = Mage::getModel("customer/customer")
            ->setWebsiteId($store->getWebsiteId())
            ->loadByEmail("test-customer@example.com");

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], "query {
            orderById(id: \"TEST-1\") {
                id
                status
                items {
                    qty
                    rowTotal {
                        incVat
                    }
                    product {
                        sku
                        attributes {
                            shortDescription
                            smallImage {
                                src
                            }
                        }
                    }
                }
                addresses {
                    type
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    company
                    street
                    city
                    postcode
                    region {
                        code
                        name
                    }
                    country {
                        code
                    }
                }
                email
                createdAt
                discountTotal
                grandTotal {
                    incVat
                    exVat
                    vat
                }
                subTotal {
                    incVat
                    exVat
                    vat
                }
                currencyCode
                virtual
                shipping {
                    method {
                        code
                        description
                    }
                    total {
                        incVat
                        exVat
                        vat
                    }
                }
            }
        }"));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testCustomerOrderByIdNotLoggedIn(): void {
        MagentoManager::init();

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], "query {
            orderById(id: \"TEST-1\") {
                id
                email
                createdAt
            }
        }"));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testCustomerOrderByIdNotFound(): void {
        MagentoManager::init();

        /** @var Mage_Core_Model_Store $store */
        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);

        $customer = Mage::getModel("customer/customer")
            ->setWebsiteId($store->getWebsiteId())
            ->loadByEmail("test-customer@example.com");

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], "query {
            orderById(id: \"TEST-1234\") {
                id
                email
                createdAt
            }
        }"));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }
}
