<?php

declare(strict_types=1);

use GraphQL\Deferred;
use GraphQL\Type\Definition\ResolveInfo;
use MageQL\Context;

use function MageQL\snakeCaseToCamel;

/**
 * @psalm-type ProductFilterInput array{
 *   code: string,
 *   value: ?string,
 *   minValue: ?float,
 *   maxValue: ?float,
 *   incVat: ?bool,
 * }
 * @psalm-type ProductSortInput array {
 *   code: string,
 *   order: MageQL_Catalog_Model_Sort::ORDER_ASC|MageQL_Catalog_Model_Sort::ORDER_DESC
 * }
 */
class MageQL_Sales_Model_Wishlist {

    const SUCCESS = "success";
    const SUCCESS_REMOVING_PRODUCT = "successRemovingProduct";
    const ERROR_ADDING_PRODUCT = "errorAddingProduct";
    const ERROR_REMOVING_PRODUCT = "errorRemovingProduct";
    const ERROR_NOT_LOGGED_IN = "errorNotLoggedIn";
    const ERROR_UNKNOWN_PRODUCT = "errorUnknownProduct";

    public static function registryKey(): string
    {
        return "mageql_sales_wishlist";
    }

    public static function instance(): self
    {
        $wishlist = Mage::registry(static::registryKey());

        if (!$wishlist) {
            $wishlist = new self();

            Mage::register(static::registryKey(), $wishlist);
        }

        return $wishlist;
    }

    /**
     * Clears this deferred container by removing the global instance.
     */
    public static function clear(): void
    {
        Mage::unregister(static::registryKey());
    }

    private function __construct()
    {
        // Empty
    }

    /**
     * @var Array<int, array{ selected:bool, added_at: ?string, wishlist_item_id: ?int}>
     */
    protected $data = [];

    /**
     * @var Array<int>
     */
    protected $productIds = [];

    /**
     * Queues the given product to fetch wishlist data.
     */
    public function add(Mage_Catalog_Model_Product $product): self
    {
        if (!in_array((int) $product->getId(), $this->productIds)) {
            $this->productIds[] = (int) $product->getId();
        }

        return $this;
    }

    public function load(): self
    {
        if (empty($this->productIds)) {
            // Nothing new to load
            return $this;
        }
        /** @var Mage_Core_Model_Store $store */
        $store = Mage::app()->getStore();
        $storeId = $store->getId();
        $customerSession = Mage::getSingleton("customer/session");
        $customerId = 0;

        if ($customerSession->isLoggedIn()) {
            $customerId = $customerSession->getCustomerId();
        }

        if (!empty($customerId) && !empty($storeId)) {
            $db = Mage::getSingleton('core/resource')->getConnection('core_read');

            if( ! $db) {
                throw new RuntimeException("Missing connection core_read");
            }

            $select = $db->select()
                    ->from(["w" => "wishlist"], [])
                    ->join(['wi' => 'wishlist_item'], "wi.wishlist_id = w.wishlist_id", ["wi.product_id", "wi.added_at", "wi.wishlist_item_id"])
                    ->where("w.customer_id = ?", $customerId)
                    ->where("wi.store_id = ?", $storeId)
                    ->where("wi.product_id IN (?)", $this->productIds);

            $result = $db->fetchAssoc($select->__toString());

            if (!empty($result)) {
                $this->data += $result;
            }

            $this->productIds = [];
        }

        return $this;
    }

    /**
     * Fetches data
     *
     * @return ?array{ selected:bool, added_at: ?string, wishlist_item_id: ?int}
     */
    public function get(Mage_Catalog_Model_Product $product): ?array
    {
        return $this->data[(int) $product->getId()] ?? null;
    }

    public static function resolveIsProductInWishlist(Mage_Catalog_Model_Product $product): Deferred
    {
        $wishlist = MageQL_Sales_Model_Wishlist::instance();

        $wishlist->add($product);

        return new Deferred(function() use($product, $wishlist) {
            $wishlist->load();

            $data = $wishlist->get($product);
            $date = $data['added_at'] ?? null;
            $wishlistItemId = $data['wishlist_item_id'] ?? null;

            return [
                "selected" => $data ? true : false,
                "addedAt" => empty($date) ? null : gmdate("Y-m-d\TH:i:s\Z", strtotime($date)),
                "itemId" => $wishlistItemId
            ];
        });
    }

    /**
     * @param mixed $unusedSrc
     * @param array{
     *   page: ?int,
     *   pageSize: ?int,
     *   filter?:?Array<ProductFilterInput>,
     *   sort: ?ProductSortInput,
     * } $args
     */
    public static function resolveProductsInWishlist(
        $unusedSrc,
        array $args,
        Context $ctx
    ): MageQL_Catalog_Model_Product_SortableCollectionInterface {
        $session = Mage::getSingleton("customer/session");

        // If customer is not logged in, default customer has customerId = 0,
        // since that generates empty collection it works out fine.
        $collection = new MageQL_Sales_Model_Wishlist_Collection($ctx->getStore(), $session->getCustomer());

        $collection->setFilters($args["filter"] ?? []);
        $collection->setSort($args["sort"]);
        $collection->setPage($args["page"], $args["pageSize"]);

        return $collection;
     }

    /**
     * @param Mage_Wishlist_Model_Item|string $src
     * @return string
     */
    public static function resolveResult($src) {
        return is_string($src) ? $src : self::SUCCESS;
    }

    /**
     * @param Mage_Wishlist_Model_Item|string $src
     * @return ?array{ selected:bool, addedAt: string, itemId: string }
     */
    public static function resolveResultItem($src): ?array {
        if(is_string($src)) {
            return null;
        }

        return [
            "selected" => true,
            "addedAt" => gmdate("Y-m-d\TH:i:s\Z", strtotime($src->getAddedAt())),
            "itemId" => (string)$src->getId(),
        ];
    }

    /**
     * Add product to wishlist
     *
     * @param mixed $unusedSrc
     * @return Mage_Wishlist_Model_Item|string
     */
    public static function mutateAddToWishlist($unusedSrc, array $args, Context $ctx) {
        try {
            if (! Mage::getSingleton('customer/session')->isLoggedIn()) {
                return self::ERROR_NOT_LOGGED_IN;
            }

            $customerId = Mage::getSingleton('customer/session')->getCustomer()->getId();
            $request = MageQL_Sales_Model_BuyRequest::fromString($ctx->getStore(), $args["buyRequest"]);

            if( ! $request instanceof MageQL_Sales_Model_BuyRequest_Product) {
                // FIXME: ClientAware exception
                throw new Exception("BuyRequest must contain a product buy request");
            }

            // Create Magento version of buyRequest
            $buyRequest = $request->getBuyRequest();
            $product = $request->getProduct();
            $buyRequest["qty"] = 1;

            // Add product to wishlist
            $wishlist = Mage::getModel("wishlist/wishlist")->loadByCustomer($customerId, true);

            $result = $wishlist->addNewItem($product, new Varien_Object($buyRequest));

            if(is_string($result)) {
                throw new Exception($result);
            }

            $wishlist->save();

            Mage::dispatchEvent("wishlist_add_product", array(
                "wishlist" => $wishlist,
                "product" => $product,
                "item" => $result
            ));

            Mage::helper("wishlist")->calculate();

            return $result;
        }
        catch(MageQL_Sales_Model_BuyRequest_Exception_ProductNotFound $e) {
            return self::ERROR_UNKNOWN_PRODUCT;
        }
        catch (Exception $ex) {
            Mage::logException($ex);

            return self::ERROR_ADDING_PRODUCT;
        }
    }

    /**
     * Remove product from wishlist
     *
     * @param mixed $unusedSrc
     * @param array{ itemId?: int } $args
     * @return String
     */
    public static function mutateRemoveFromWishlist($unusedSrc, array $args) {
        $wishlistItemId = $args["itemId"] ?? null;

        if(empty($wishlistItemId)) {
            return self::ERROR_UNKNOWN_PRODUCT;
        }

        if( ! Mage::getSingleton('customer/session')->isLoggedIn()) {
            return self::ERROR_NOT_LOGGED_IN;
        }

        $customerId = Mage::getSingleton('customer/session')->getCustomer()->getId();

        try {
            $sqlQuery = "
                SELECT wi.wishlist_item_id
                FROM wishlist_item wi
                JOIN wishlist w ON w.wishlist_id = wi.wishlist_id
                WHERE w.customer_id = :customerId AND wi.wishlist_item_id = :wishlistItemId
            ";
            $params = [
                "customerId" => $customerId,
                "wishlistItemId" => $wishlistItemId
            ];
            $conn = Mage::getSingleton('core/resource')->getConnection('core_read');

            if( ! $conn) {
                throw new RuntimeException("core_read connection is missing");
            }

            $result = $conn->fetchOne($sqlQuery, $params);

            if (empty($result) || (int)$result !== (int)$wishlistItemId) {
                return self::ERROR_UNKNOWN_PRODUCT;
            }

            $item = Mage::getModel('wishlist/item')->load($wishlistItemId);
            $item->delete();

            return self::SUCCESS;
        }
        catch (Exception $ex) {
            Mage::logException($ex);

            return self::ERROR_REMOVING_PRODUCT;
        }
    }
}
