<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

use function MageQL\snakeCaseToCamel;

use MageQL\Registry;
use MageQL\Type\AbstractBuilder;
use MageQL\Type\FieldBuilder;
use MageQL\Type\InterfaceBuilder;
use MageQL\Type\ObjectBuilder;

/**
 */
class MageQL_Sales_Model_Schema_Default_Wishlist extends MageQL_Core_Model_Schema_Abstract {

    public function getTypeBuilder(string $typeName, Registry $registry): ?AbstractBuilder {

        switch($typeName) {
        case "IsProductInWishlist":
            return $this->object("Product wishlist status");

        case "AddToWishlistResult":
            return $this->object("Result from addToWishlist");

        case "RemoveFromWishlistResult":
            return $this->object("Result from removeFromWishlist");

        case "AddToWishlistStatusCode":
            return $this->enum(
                "Enum indicating success for adding product to wishlist",
                [
                    MageQL_Sales_Model_Wishlist::SUCCESS => [
                        "description" => "Successfully added product to wishlist",
                    ],
                    MageQL_Sales_Model_Wishlist::ERROR_ADDING_PRODUCT => [
                        "description" => "Error while adding product to wishlist",
                    ],
                    MageQL_Sales_Model_Wishlist::ERROR_NOT_LOGGED_IN => [
                        "description" => "Customer not logged in",
                    ],
                    MageQL_Sales_Model_Wishlist::ERROR_UNKNOWN_PRODUCT => [
                        "description" => "Product was not found",
                    ],
                ]
            );

        case "RemoveFromWishlistStatusCode":
            return $this->enum(
                "Enum indicating success for removing product from wishlist",
                [
                    MageQL_Sales_Model_Wishlist::SUCCESS => [
                        "description" => "Successfully removed product from wishlist",
                    ],
                    MageQL_Sales_Model_Wishlist::ERROR_REMOVING_PRODUCT => [
                        "description" => "Error while removing product from wishlist",
                    ],
                    MageQL_Sales_Model_Wishlist::ERROR_NOT_LOGGED_IN => [
                        "description" => "Customer not logged in",
                    ],
                    MageQL_Sales_Model_Wishlist::ERROR_UNKNOWN_PRODUCT => [
                        "description" => "Product was not found",
                    ],
                ]
            );

        default:
            return null;
        }
    }

    public function getTypeFields(string $typeName, Registry $registry): array {

        switch($typeName) {
        case "ProductDetail":
            return [
                "wishlist" => $this->field("IsProductInWishlist!", "Product wishlist status")
                    ->setResolver("MageQL_Sales_Model_Wishlist::resolveIsProductInWishlist"),
            ];

        case "ListProduct":
            return [
                "wishlist" => $this->field("IsProductInWishlist!", "Product wishlist status")
                    ->setResolver("MageQL_Sales_Model_Wishlist::resolveIsProductInWishlist"),
            ];

        case "IsProductInWishlist":
            return [
                "selected" => $this->field("Boolean!", "Is product in wishlist"),
                "addedAt" => $this->field("String", "Date product was added to wishlist"),
                "itemId" => $this->field("ID", "Wishlist item id"),
            ];

        case "Query":
            return [
                "productsInWishlist" => $this->field("SortableProductList!", "All products in wishlist")
                    ->addArgument(
                        "filter",
                        $this->argument("[ProductFilterInput!]", "Filter products by these filters"))
                    ->addArgument(
                        "sort",
                        $this->argument("ProductSortInput", "Sorting to use for the products")
                             ->setDefaultValue(MageQL_Sales_Model_Wishlist_Collection::DEFAULT_SORT))
                    ->addArgument(
                        "pageSize",
                        $this->argument("Int", "Maximum number of products to list")
                            ->setDefaultValue(MageQL_Catalog_Model_Product_CollectionInterface::DEFAULT_PAGE_SIZE))
                    ->addArgument(
                        "page",
                        $this->argument("Int", "Which page to show")
                            ->setDefaultValue(MageQL_Catalog_Model_Product_CollectionInterface::DEFAULT_PAGE))
                    ->setResolver("MageQL_Sales_Model_Wishlist::resolveProductsInWishlist"),
            ];

        case "Mutation":
            return [
                "addToWishlist" => $this->field("AddToWishlistResult!", "Result of adding product to wishlist")
                    ->addArgument("buyRequest", $this->argument("ID!", "A buy request from a product or product variant"))
                    ->setResolver("MageQL_Sales_Model_Wishlist::mutateAddToWishlist"),
                "removeFromWishlist" => $this->field("RemoveFromWishlistResult!", "Result of removing product from wishlist")
                    ->addArgument("itemId", $this->argument("ID!", "Wishlist item id"))
                    ->setResolver("MageQL_Sales_Model_Wishlist::mutateRemoveFromWishlist"),
            ];

        case "AddToWishlistResult":
            return [
                "result" => $this->field("AddToWishlistStatusCode", "Status code of addToWishlist")
                    ->setResolver("MageQL_Sales_Model_Wishlist::resolveResult"),
                "wishlistItem" => $this->field("IsProductInWishlist", "Newly created wishlist item")
                    ->setResolver("MageQL_Sales_Model_Wishlist::resolveResultItem"),
            ];

        case "RemoveFromWishlistResult":
            return [
                "result" => $this->field("RemoveFromWishlistStatusCode", "Status code of addToWishlist")
                    ->setResolver("MageQL\\forwardResolver"),
            ];

        default:
            return [];
        }
    }

}
