<?php

declare(strict_types=1);

/**
 * Abstract class with static facade methods to create or parse buy-requests.
 *
 * @psalm-type BundleOption array{
 *   optionId: string,
 *   qty?: float,
 *   selectionId?: string,
 * }
 * @psalm-type CustomOption array{
 *     optionId: string,
 *     valueId?: string,
 *     field?: string,
 *     valueIds?: array<string>,
 * }
 */
abstract class MageQL_Sales_Model_BuyRequest implements JsonSerializable {
    abstract public function getPayload(): array;

    public function __toString(): string {
        return base64_encode(json_encode($this->getPayload()) ?: "");
    }

    /**
     * @return string
     */
    #[ReturnTypeWillChange]
    public function jsonSerialize() {
        return $this->__toString();
    }

    /**
     * Creates a buy request from a single product, will return null if it is
     * not possible to create a request for the specific product (either not
     * saleable or requires additional options).
     */
    public static function fromProduct(
        Mage_Catalog_Model_Product $product,
        Mage_Core_Model_Store $store
    ): ?MageQL_Sales_Model_BuyRequest_Product {
        if(
            // Exclude children of configurable products in quote items
            $product->getParentId() ||
            // Products which require an active choice cannot be simple buyRequests
            self::isProductSelectionRequired($product, $store) ||
            // And we cannot show buyRequests if the product is not in stock
            //
            // NOTE: Important that this one is last since configurable products
            // should fail the above selection check and isSalable triggers
            // configurable isSalable which loads all child-products of the
            // configurable product. Same goes for bundles, though
            // isProductSelectionRequired loads items.
            ! $product->isSalable()
        ) {
            return null;
        }

        return new MageQL_Sales_Model_BuyRequest_Product($product);
    }

    public static function getProductOptionSelections(
        Mage_Bundle_Model_Option $option,
        Mage_Core_Model_Store $store
    ): Mage_Bundle_Model_Resource_Selection_Collection {
        $selections = Mage::getResourceModel("bundle/selection_collection");

        $selections->setOptionIdsFilter([$option->getId()]);

        // No visibility filter here since bundle option selections can be invisible
        $selections->addStoreFilter($store);
        $selections->addMinimalPrice();
        $selections->joinPrices((int)$store->getWebsiteId());

        return $selections;
    }

    public static function isProductSelectionRequired(
        Mage_Catalog_Model_Product $product,
        Mage_Core_Model_Store $store
    ): bool {
        if($product->getTypeId() === "bundle") {
            /**
             * @var Mage_Bundle_Model_Product_Type
             */
            $instance = $product->getTypeInstance(true);
            $options = $instance->getOptions($product);

            foreach($options as $opt) {
                if($opt->getType() === "hidden" || !$opt->getRequired()) {
                    continue;
                }

                $hasSelection = false;

                foreach(self::getProductOptionSelections($opt, $store) as $sel) {
                    if($sel->getIsDefault() && $sel->getSelectionQty() > 0) {
                        $hasSelection = true;
                    }
                }

                if( ! $hasSelection) {
                    return true;
                }
            }
        }
        else if($product->getTypeId() === "configurable" && $product->getRequiredOptions()) {
            return true;
        }

        return false;
    }

    /**
     * Creates a buy request from a child-product and configuration attributes.
     */
    public static function fromChildProduct(
        MageQL_Catalog_Model_Product_Configurable_Option $option
    ): ?MageQL_Sales_Model_BuyRequest_Product_Configurable {
        if( ! $option->getProduct()->isSalable()) {
            return null;
        }

        return new MageQL_Sales_Model_BuyRequest_Product_Configurable($option->getParent(), $option->getProduct(), $option->getAttributes());
    }

    public static function fromQuoteItem(
        Mage_Sales_Model_Quote_Item $item
    ): MageQL_Sales_Model_BuyRequest_Item {
        return new MageQL_Sales_Model_BuyRequest_Item($item);
    }

    /**
     * @param array<BundleOption> $bundleOptions
     * @param array<CustomOption> $customOptions
     */
    public static function fromString(
        Mage_Core_Model_Store $store,
        string $input,
        array $bundleOptions = [],
        array $customOptions = []
    ): self {
        $decode = base64_decode($input);

        if( ! $decode) {
            throw new MageQL_Sales_Model_BuyRequest_Exception_ParseError("Failed to decode");
        }

        $request = json_decode($decode, true, 3);

        if(json_last_error()) {
            throw new MageQL_Sales_Model_BuyRequest_Exception_ParseError(json_last_error_msg());
        }

        $productId = (int)($request["p"] ?? 0);
        $itemId = (int)($request["i"] ?? 0);
        $attributes = $request["a"] ?? [];
        $options = $request["o"] ?? [];

        $product = Mage::getModel("catalog/product");

        $product->setStoreId($store->getId());
        $product->load($productId);

        if( ! Mage::helper("mageql_catalog")->isProductVisible($product, $store)) {
            throw new MageQL_Sales_Model_BuyRequest_Exception_ProductNotFound();
        }

        if($itemId) {
            $model = Mage::getSingleton("mageql_sales/quote");
            $quote = $model->getQuote();

            /**
             * @var ?Mage_Sales_Model_Quote_Item
             */
            $item = $quote->getItemById($request["i"]);

            if( ! $item) {
                throw new MageQL_Sales_Model_BuyRequest_Exception_ItemNotFound();
            }

            return new MageQL_Sales_Model_BuyRequest_Item($item);
        }

        switch($product->getTypeId()) {
        case "configurable":
            return self::fromConfigurable($store, $product, $attributes);

        case "bundle":
            return self::fromBundle($store, $product, $bundleOptions);

        default:
            return self::fromSimple($product, $options, $customOptions);
        }
    }

    protected static function fromSimple(
        Mage_Catalog_Model_Product $product,
        array $options,
        array $customOptions
    ): MageQL_Sales_Model_BuyRequest_Product {
        // TODO: modify to account for more custom-option types
        $isScalar =
            /**
             * @param mixed $i
             */
            function(bool $a, $i): bool {
                return $a && is_scalar($i);
            };

        // Verify options input data
        if (!array_reduce($options, $isScalar, true)) {
            throw new MageQL_Sales_Model_BuyRequest_Exception_InvalidOptions();
        }

        $productOptions = $product->getOptions();
        $parsedOptions = [];

        // Overwrite options data with custom options data
        foreach ($customOptions as $customOption) {
            $options[$customOption['optionId']] = $customOption;
        }

        // Verify valid options and option values
        foreach ($options as $optionId => $optionValue) {
            // Verify option
            if (!array_key_exists($optionId, $productOptions)) {
                // No such product option
                throw new MageQL_Sales_Model_BuyRequest_Exception_InvalidOptions("Invalid optionId '{$optionId}'");
            }
            $productOption = $productOptions[$optionId];

            switch ($productOption->getType()) {
                case Mage_Catalog_Model_Product_Option::OPTION_TYPE_DROP_DOWN:
                case Mage_Catalog_Model_Product_Option::OPTION_TYPE_RADIO:
                case Mage_Catalog_Model_Product_Option::OPTION_TYPE_CHECKBOX:
                    // Single select types
                    $selectionId = (is_array($optionValue) ? $optionValue['value']['selectionId'] : $optionValue) ?? null;
                    if (!is_null($selectionId) && !array_key_exists($selectionId, $productOption->getValues())) {
                        // Invalid product option value
                        throw new MageQL_Sales_Model_BuyRequest_Exception_InvalidOptions("Invalid selectionId '{$selectionId}'");
                    }
                    $parsedOptions[$optionId] = $selectionId;
                    break;

                case Mage_Catalog_Model_Product_Option::OPTION_TYPE_FIELD:
                case Mage_Catalog_Model_Product_Option::OPTION_TYPE_AREA:
                    // Text types
                    $textContent = (is_array($optionValue) ? $optionValue['value']['text'] : $optionValue) ?? null;
                    $parsedOptions[$optionId] = $textContent;
                    break;

                default:
                    // Unhandled type
                    throw new MageQL_Sales_Model_BuyRequest_Exception_InvalidOptions("Unimplemented option type '{$productOption->getType()}'");
            }
        }

        // Check required
        foreach ($productOptions as $productOption) {
            if (!$productOption->getIsRequire()) {
                continue;
            }
            if (!array_key_exists((int)$productOption->getId(), $parsedOptions)) {
                // Required option missing
                throw new MageQL_Sales_Model_BuyRequest_Exception_MissingRequiredOptions();
            }
            if (is_null($parsedOptions[(int)$productOption->getId()])) {
                // Required option is null
                throw new MageQL_Sales_Model_BuyRequest_Exception_MissingRequiredOptions();
            }
        }

        return new MageQL_Sales_Model_BuyRequest_Product($product, $parsedOptions);
    }

    /**
     * @param mixed $attributes
     */
    protected static function fromConfigurable(
        Mage_Core_Model_Store $store,
        Mage_Catalog_Model_Product $product,
        $attributes
    ): MageQL_Sales_Model_BuyRequest_Product_Configurable {
        $isScalar =
            /**
             * @param mixed $i
             */
            function(bool $a, $i): bool {
                return $a && is_scalar($i);
            };

        if( ! is_array($attributes) || ! array_reduce($attributes, $isScalar , true)) {
            throw new MageQL_Sales_Model_BuyRequest_Exception_InvalidAttributes();
        }

        /**
         * @var $attributes Array<string, int>
         */

        /**
         * @var Mage_Catalog_Model_Product_Type_Configurable
         */
        $instance = $product->getTypeInstance(true);
        $prodAttrs = $instance->getConfigurableAttributes($product);
        $child = null;

        // Local implementation of Mage_Catalog_Model_Product_Type_Configurable::getUsedProducts
        $children = $instance->getUsedProductCollection($product);

        $children->addFilterByRequiredOptions();
        $children->addStoreFilter($store);
        $children->addMinimalPrice();
        $children->addUrlRewrite();

        // Add the attributes we use for the configurating
        foreach($prodAttrs as $attr) {
            $children->addAttributeToSelect($attr->getProductAttribute()->getAttributeCode());
        }

        // Validate options
        foreach($children as $c) {
            foreach($prodAttrs as $attr) {
                $prodAttr = $attr->getProductAttribute();

                if(empty($attributes[$prodAttr->getAttributeId()]) ||
                    (int)$attributes[$prodAttr->getAttributeId()] !== (int)$c->getData($prodAttr->getAttributeCode())) {

                    // Mismatch, try next product
                    continue 2;
                }
            }

            // TODO: Check for extra options in $prodAttrs
            $child = $c;

            break;
        }

        if( ! $child) {
            throw new MageQL_Sales_Model_BuyRequest_Exception_ProductNotFound();
        }

        return new MageQL_Sales_Model_BuyRequest_Product_Configurable($product, $child, $prodAttrs);
    }

    /**
     * @param array<BundleOption> $userOptions
     */
    protected static function fromBundle(
        Mage_Core_Model_Store $store,
        Mage_Catalog_Model_Product $product,
        array $userOptions
    ): MageQL_Sales_Model_BuyRequest_Product_Bundle {
        $bundleOptions = [];
        $bundleOptionQty = [];
        /**
         * @var Mage_Bundle_Model_Product_Type
         */
        $instance = $product->getTypeInstance(true);

        // Add a store filter to ensure bundle option title is loaded correctly.
        // If the bundle option title is not loaded correctly then any modifications
        // to the quote item will duplicate any child items.
        $instance->setStoreFilter($store->getId(), $product);

        $options = $instance->getOptions($product);

        foreach($options as $opt) {
            $selections = MageQL_Sales_Model_BuyRequest::getProductOptionSelections(
                $opt,
                $store
            );
            $qty = null;
            $selected = [];

            foreach($userOptions as $userSelection) {
                if($userSelection["optionId"] === $opt->getId()) {
                    if (empty($userSelection["selectionId"])) {
                        $qty = null;
                        $selected = null;
                    }
                    else {
                        foreach($selections as $s) {
                            if($s->getSelectionId() === $userSelection["selectionId"]) {
                                $selected = $selected ?: [];
                                $selected[] = $s->getSelectionId();
                                $qty = $s->getSelectionQty();

                                if( ! empty($userSelection["qty"])) {
                                    if( ! $s->getSelectionCanChangeQty() && $userSelection["qty"] != $qty) {
                                        throw new MageQL_Sales_Model_BuyRequest_Exception_BundleSelectionQtyImmutable();
                                    }

                                    $qty = $userSelection["qty"];
                                }

                                continue 2;
                            }
                        }
                    }
                }
            }

            // We fill in the defaults after if we have nothing
            if(empty($selected) && $selected !== null) {
                foreach($selections as $sel) {
                    if($sel->getIsDefault()) {
                        $selected[] = $sel->getSelectionId();
                        $qty = $sel->getSelectionQty();
                    }
                }
            }

            if( ! empty($selected) && ! empty($qty)) {
                if($opt->isMultiSelection()) {
                    $bundleOptions[(int)$opt->getId()] = $selected;
                }
                else {
                    if (count($selected) > 1) {
                        throw new MageQL_Sales_Model_BuyRequest_Exception_BundleOptionSingleGotMultiple();
                    }

                    $bundleOptions[(int)$opt->getId()] = $selected[0];
                }

                $bundleOptionQty[(int)$opt->getId()] = $qty;
            }
            else if($opt->getRequired()) {
                throw new MageQL_Sales_Model_BuyRequest_Exception_BundleOptionRequired();
            }
        }

        if(empty($bundleOptions)) {
            throw new MageQL_Sales_Model_BuyRequest_Exception_BundleMissingOptions();
        }
        /** @psalm-suppress InvalidArgument */
        return new MageQL_Sales_Model_BuyRequest_Product_Bundle($product, $bundleOptions, $bundleOptionQty);
    }

    public static function fromProductOptionValue(
        Mage_Catalog_Model_Product_Option_Value $value
    ): ?MageQL_Sales_Model_BuyRequest_Product {
        /**
         * @var ?Mage_Catalog_Model_Product_Option
         */
        $option = $value->getOption();

        if( ! $option) {
            throw new RuntimeException("%s: Option value instance is missing option instance");
        }

        /**
         * @var ?Mage_Catalog_Model_Product
         */
        $product = $option->getProduct();

        if( ! $product) {
            throw new RuntimeException("%s: Option instance is missing product instance");
        }

        if( ! $product->isSalable()) {
            return null;
        }

        /** @psalm-suppress InvalidScalarArgument */
        return new MageQL_Sales_Model_BuyRequest_Product($product, [(int)$option->getId() => (int)$value->getId()]);
    }
}
