<?php

declare(strict_types=1);

namespace MageQL\Sales\Wishlist;

use Mage;
use Mage_Core_Model_Store;
use MageQL_Sales_Model_BuyRequest;

use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use Spatie\Snapshots\MatchesSnapshots;

require_once __DIR__."/Base.php";

class NotLoggedInTest extends Base {
    use MatchesSnapshots;

    public function testProducts(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            products {
                items {
                    sku
                    wishlist { selected, addedAt, itemId }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductDetail(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku(sku: "test-simple") {
              sku
              wishlist { selected, addedAt, itemId }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testRouteProduct(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            route(path: "test-for-simple-product.html") {
                ... on RouteProduct {
                    product {
                        sku
                        wishlist { selected, addedAt, itemId }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductBySku(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku(sku: "test-config") {
                sku
                wishlist { selected, addedAt, itemId }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testBySearch(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productsBySearch(term: "test") {
                items {
                    sku
                    wishlist { selected, addedAt }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testAdd(): void {
        MagentoManager::init();

         /** @var Mage_Core_Model_Store $store */
        $store = Mage::app()->getStore();

        $product = Mage::getModel("catalog/product");
        $product->load($product->getIdBySku("test-simple"));

        $buyRequest = MageQL_Sales_Model_BuyRequest::fromProduct($product, $store);

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'mutation {
            addToWishlist(buyRequest:'.json_encode($buyRequest).') {
                result
                wishlistItem { selected addedAt itemId }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testRemove(): void {
        MagentoManager::init();

        // Try to remove item from wishlist
        $wishlistItemId = $this->fetchWishlistItemId("test-simple");

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'mutation {
            removeFromWishlist(itemId:'.json_encode($wishlistItemId).') {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }
}
