<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

/**
 * @psalm-type TaxRate array{amount:float, percent:float}
 */
class MageQL_Sales_Model_Quote {
    const SUCCESS = "success";
    const ERROR_INVALID_EMAIL_ADDRESS = "errorInvalidEmailAddress";
    const ERROR_CUSTOMER_IS_LOGGED_IN = "errorCustomerIsLoggedIn";

    public static function resolve(): Mage_Sales_Model_Quote {
        return Mage::getSingleton("mageql_sales/quote")->getQuote();
    }

    /**
     * @return Array<Mage_Sales_Model_Quote_Address>
     */
    public static function resolveAddresses(
        Mage_Sales_Model_Quote $src
    ): array {
        $addresses = [];

        foreach($src->getAllAddresses() as $addr) {
            if($addr->getAddressType() !== Mage_Customer_Model_Address_Abstract::TYPE_SHIPPING ||
                ! $addr->getSameAsBilling()) {
                $addresses[] = $addr;
            }
        }

        return $addresses;
    }

    public static function resolveEmail(Mage_Sales_Model_Quote $src): ?string {
        $email = $src->getCustomerEmail();

        if( ! $email) {
            return null;
        }

        return $email;
    }

    /**
     * @return Varien_Object
     */
    public static function resolveGrandTotal(
        Mage_Sales_Model_Quote $src,
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx
    ) {
        $store = $ctx->getStore();
        $tax = 0.0;

        foreach($src->getAllAddresses() as $addr) {
            $tax += (float)$addr->getTaxAmount();
        }

        return new Varien_Object([
            "inc_vat" => (float) $src->getGrandTotal(),
            "ex_vat" => (float) $store->roundPrice($src->getGrandTotal() - $tax),
            "vat" => (float)$store->roundPrice($tax),
        ]);
    }

    public static function resolveIsVirtual(Mage_Sales_Model_Quote $src): bool {
        return $src->isVirtual();
    }

    /**
     * @param TaxRate $src
     */
    public static function resolveTaxRatePercent($src): float {
        return $src["percent"];
    }

    /**
     * @param TaxRate $src
     */
    public static function resolveTaxRateAmount(
        $src,
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx
    ): float {
        $store = $ctx->getStore();

        return $store->roundPrice($src["amount"]);
    }

    /**
     * @return Array<array{percent:float, amount:float}>
     */
    public static function resolveTaxRates(Mage_Sales_Model_Quote $src) {
        // No idea why it is on shipping address sometimes and other times on billing,
        // something related to tax-settings.
        $rates = [];

        foreach($src->getAllAddresses() as $addr) {
            $appliedRates = $addr->getAppliedTaxes();

            if( ! empty($appliedRates)) {
                $rates = array_merge($rates, $appliedRates);
            }
        }

        return array_reduce($rates,
            /**
             * @param Array<array{percent:float, amount:float}> $taxes
             * @param array{percent:float, amount:float} $tax
             * @return Array<array{percent:float, amount:float}>
             */
            function($taxes, $tax) {
                for($i = 0, $c = count($taxes); $i < $c; $i++) {
                    if($taxes[$i]["percent"] === round($tax["percent"], 2)) {
                        $taxes[$i]["amount"] += $tax["amount"];

                        return $taxes;
                    }
                }

                $taxes[] = [
                    "percent" => round($tax["percent"], 2),
                    "amount" => (float) $tax["amount"]
                ];

                return $taxes;
            }, []);
    }

    public static function resolveShipping(Mage_Sales_Model_Quote $src): ?Mage_Sales_Model_Quote_Address {
        if($src->isVirtual()) {
            return null;
        }

        $model = Mage::getSingleton("mageql_sales/quote");
        $shipping = $model->getShippingAddress();

        if( ! $shipping || ! $shipping->getShippingMethod()) {
            return null;
        }

        return $shipping;
    }

    /**
     * @return Varien_Object
     */
    public static function resolveSubTotal(
        Mage_Sales_Model_Quote $src,
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx
    ) {
        $totals = $src->getTotals();
        $store = $ctx->getStore();
        $subTotal = 0;
        $exclTax = 0;

        if(array_key_exists("subtotal", $totals)) {
            $subTotal = (float) $totals["subtotal"]->getValueInclTax();
            $exclTax = (float) $totals["subtotal"]->getValueExclTax();
        }

        return new Varien_Object([
            "ex_vat" => $exclTax,
            "inc_vat" => $subTotal,
            "vat" => (float) $store->roundPrice($subTotal - $exclTax),
        ]);
    }

    /**
     * @return Array<Mage_Sales_Model_Quote_Item>
     */
    public static function resolveItems(
        Mage_Sales_Model_Quote $src,
        array $unusedArgs,
        MageQL_Core_Model_Context $unusedCtx,
        ResolveInfo $info
    ): array {
        $config = Mage::getSingleton("mageql_catalog/attributes_product");
        /**
         * 2 levels deep to get attributes (1 item, 2 product, 3 attributes)
         *
         * @var array{product:?array{attributes:Array<string, bool>}}
         */
        $fields = $info->getFieldSelection(3);
        // Merge attributes
        $fields = array_merge($fields["product"] ?? [], $fields["product"]["attributes"] ?? []);

        // We use all attribute sets since we cannot be certain about which sets we will get
        $toSelect = $config->getUsedAttributes(
            $config->getAttributesByArea(MageQL_Catalog_Model_Attributes_Abstract::AREA_LIST),
            $fields
        );

        // Set the attributes so they are loaded when we get the quote items
        Mage::getSingleton("mageql_sales/attributes_product")->setQuoteAttributes($toSelect);

        // Do not use a cache since we want to use the previously set attributes
        // Note that the $useCache flag does not work on quote getItemsCollection
        $collection = Mage::getModel("sales/quote_item")->getCollection()->setQuote($src);

        $items = $collection->getItems();

        return array_values(array_filter($items,
            /**
             * @param Mage_Sales_Model_Quote_Item $item
             */
            function($item) {
                return ! $item->isDeleted() && ! $item->getParentItemId();
            }));
    }

    /**
     * @return Array<Varien_Object>
     */
    public static function resolveValidationErrors(Mage_Sales_Model_Quote $quote): array {
        $errors = [];
        $paymentManager = MageQL_Sales_Model_Payment::instance($quote->getStore());

        if( ! $quote->hasItems()) {
            $errors[] = new Varien_Object([
                "code" => "quote_empty",
                "message" => "No products present in quote",
            ]);
        }

        if( ! $quote->isVirtual()) {
            // Physical items require a shipping address and a method
            $shippingAddress = $quote->getShippingAddress();
            $shippingValidation = $shippingAddress->validate();

            if($shippingValidation !== true) {
                foreach($shippingValidation as $error) {
                    // TODO: convert to proper errors and field
                    $errors[] = new Varien_Object([
                        "code" => "quote_shipping_address_error",
                        "message" => $error,
                    ]);
                }
            }

            $method = $shippingAddress->getShippingMethod();
            $rate = $shippingAddress->getShippingRateByCode($method);

            if ( ! $method || ! $rate) {
                $errors[] = new Varien_Object([
                    "code" => "quote_shipping_method_missing",
                    "message" => "No shipping method selected",
                ]);
            }
        }

        $billingValidation = $quote->getBillingAddress()->validate();

        if($billingValidation !== true) {
            foreach($billingValidation as $error) {
                // TODO: convert to proper errors and field
                $errors[] = new Varien_Object([
                    "code" => "quote_billing_address_error",
                    "message" => $error,
                ]);
            }
        }

        $payment = $quote->getPayment();

        if( ! $payment->getMethod()) {
            $errors[] = new Varien_Object([
                "code" => "quote_payment_method_missing",
                "message" => "No payment method selected",
            ]);
        }
        else if( ! $paymentManager->hasPaymentMethod($payment->getMethod())) {
            $errors[] = new Varien_Object([
                "code" => "quote_payment_method_invalid",
                "message" => "Invalid payment method selected",
            ]);
        }

        if( ! Zend_Validate::is($quote->getCustomerEmail(), "EmailAddress")) {
            $errors[] = new Varien_Object([
                "code" => "quote_email_invalid",
                "message" => "Invalid quote email",
            ]);
        }

        // TODO: Verify agreements

        return $errors;
    }

    public static function mutateRemove(): bool {
        $model = Mage::getSingleton("mageql_sales/quote");

        $model->removeQuote();

        return true;
    }

    /**
     * @param mixed $unusedSrc
     * @param array{email:string} $args
     */
    public static function mutateSetEmail($unusedSrc, array $args): string {
        $model = Mage::getSingleton("mageql_sales/quote");
        $customerSession = Mage::getSingleton("customer/session");
        $email = $args["email"];
        $quote = $model->getQuote();

        if($customerSession->isLoggedIn()) {
            return self::ERROR_CUSTOMER_IS_LOGGED_IN;
        }

        if( ! Zend_Validate::is($email, "EmailAddress")) {
            return self::ERROR_INVALID_EMAIL_ADDRESS;
        }

        $quote->setCustomerEmail($email);

        foreach($quote->getAllAddresses() as $addr) {
            $addr->setEmail($email);
        }

        $model->saveSessionQuote();

        return self::SUCCESS;
    }

    /**
     * @param mixed $unusedSrc
     * @return Array<Mage_Sales_Model_Quote_Item>
     */
    public static function mutateRemoveUnorderableItems(
        $unusedSrc,
        array $unusedArgs,
        MageQL_Core_Model_Context $unusedCtx,
        ResolveInfo $info
    ): array {
        /**
         * @var Array<Mage_Sales_Model_Quote_Item>
         */
        $removedItems = [];
        $model = Mage::getSingleton("mageql_sales/quote");
        $quote = $model->getQuote();

        $items = self::resolveItems($quote, $unusedArgs, $unusedCtx, $info);

        foreach($items as $i) {
            if( ! MageQL_Sales_Model_Quote_Item::resolveCanOrder($i)) {
                $quote->removeItem($i->getId());

                $removedItems[] = $i;
            }
        }

        if( ! empty($removedItems)) {
            // Recollect since we might have removed stuff related to shipping
            $model->saveSessionQuoteWithShippingRates();

            // TODO: Event
        }

        return $removedItems;
    }

    public static function mutatePlaceOrder(): MageQL_Sales_Model_Order_AbstractResult {
        try {
            $session = Mage::getSingleton("checkout/session");
            $helper = Mage::helper("mageql_sales/quote");
            $quote = $session->getQuote();

            $order = $helper->submitOrder($quote);

            $orderPayment = $order->getPayment();
            $redirect = $quote->getPayment()->getOrderPlaceRedirectUrl();
            $agreement = $orderPayment ? $orderPayment->getBillingAgreement() : null;

            $result = new MageQL_Sales_Model_Order_Result($order);

            if($redirect) {
                $result->setPaymentRedirectUrl($redirect);
            }

            $session->unsetAll();
            $session->clearHelperData();
            $session->setLastQuoteId($quote->getId());
            $session->setLastSuccessQuoteId($quote->getId());
            $session->setLastOrderId($order->getId());
            $session->setLastRealOrderId($order->getIncrementId());

            if($agreement) {
                $session->setLastBillingAgreementId($agreement->getId());
            }

            return $result;
        }
        catch(MageQL_Sales_SubmitOrderException $e) {
            return new MageQL_Sales_Model_Order_Error($e->getErrorCode());
        }
        catch(MageQL_Sales_ErrorInterface $e) {
            // Use normal
            return new MageQL_Sales_Model_Order_Error($e->getErrorCode());
        }
    }

    public function getQuote(): Mage_Sales_Model_Quote {
        $session = Mage::getSingleton("checkout/session");
        $quote = $session->getQuote();

        // Ensure we have defaults for addresses
        $this->setQuoteDefaults($quote);

        return $quote;
    }

    public function removeQuote(): void {
        $session = Mage::getSingleton("checkout/session");
        $quote = $session->getQuote();

        // Only save it if we need to
        if($quote->getId()) {
            $quote->setIsActive(0);
            $quote->save();
        }

        $session->unsetAll();
    }

    /**
     * Variant of Mage_Sales_Model_Quote::getBillingAddress() which will NOT
     * automatically create an address if none exists.
     */
    public function getBillingAddress(): ?Mage_Sales_Model_Quote_Address {
        $quote = $this->getQuote();

        foreach($quote->getAllAddresses() as $addr) {
            if($addr->getAddressType() === Mage_Customer_Model_Address_Abstract::TYPE_BILLING) {
                return $addr;
            }
        }

        return null;
    }

    /**
     * Variant of Mage_Sales_Model_Quote::getShippingAddress() which will NOT
     * automatically create an address if none exists.
     */
    public function getShippingAddress(): ?Mage_Sales_Model_Quote_Address {
        $quote = $this->getQuote();

        foreach($quote->getAllAddresses() as $addr) {
            if($addr->getAddressType() === Mage_Customer_Model_Address_Abstract::TYPE_SHIPPING) {
                return $addr;
            }
        }

        return null;
    }

    /**
     * Gets billing address, if it does not already exists it will be created, with defaults.
     */
    public function makeBillingAddress(): Mage_Sales_Model_Quote_Address {
        $quote = $this->getQuote();
        $billing = $quote->getBillingAddress();

        $this->setAddressDefaults($quote, $billing);

        return $billing;
    }

    /**
     * Gets shipping address, if it does not already exists it will be created, with defaults.
     */
    public function makeShippingAddress(): Mage_Sales_Model_Quote_Address {
        $quote = $this->getQuote();
        $shipping = $quote->getShippingAddress();

        $this->setAddressDefaults($quote, $shipping);

        return $shipping;
    }

    /**
     * Fills default values in a quote, will not instantiate new addresses.
     */
    public function setQuoteDefaults(Mage_Sales_Model_Quote $quote): void {
        $session = Mage::getSingleton("customer/session");
        $customer = $session->getCustomerId() ? $session->getCustomer() : null;

        foreach($quote->getAllAddresses() as $addr) {
            $this->setAddressDefaults($quote, $addr, $customer);
        }
    }

    public function setAddressDefaults(
        Mage_Sales_Model_Quote $quote,
        Mage_Sales_Model_Quote_Address $address,
        Mage_Customer_Model_Customer $customer = null
    ): void {
        $store = $quote->getStore();
        $form = Mage::getSingleton("mageql_sales/form_address");

        if($customer && MageQL_Sales_Model_Quote_Address::isAddressEmpty($address, $store)) {
            $isShipping = $address->getAddressType() === Mage_Customer_Model_Address_Abstract::TYPE_SHIPPING;
            /**
             * @var ?Mage_Customer_Model_Address_Abstract $customerAddress
             */
            $customerAddress = $isShipping ?
                $customer->getDefaultShippingAddress() :
                $customer->getDefaultBillingAddress();

            // Check address country vs valid countries, skip copying if invalid
            if(
                $customerAddress &&
                $this->isValidCountryId($customerAddress->getCountryId(), $store) &&
                $form->copyData($customerAddress, $address)
            ) {
                $address->setCustomerAddressId(null);
                $address->setSaveInAddressBook(false);
                $address->setCollectShippingRates(true);
                $address->removeAllShippingRates();
            }
        }

        // Ensure that we always have a country
        if( ! $address->getCountryId()) {
            $countryId = $store->getConfig(Mage_Core_Helper_Data::XML_PATH_DEFAULT_COUNTRY);

            if( ! $countryId) {
                throw new Exception(sprintf(
                    "%s: The default country is not set for store '%s'.",
                    __METHOD__,
                    $store->getCode()
                ));
            }

            $address->setCountryId($countryId);
        }

        // Adapted from Mage_Checkout_Model_Cart::init(), just make sure to not create
        // addresses early
        if(
            $address->getAddressType() === Mage_Customer_Model_Address_Abstract::TYPE_SHIPPING &&
            ! $quote->hasItems()
        ) {
            $address->setCollectShippingRates(false);
            $address->removeAllShippingRates();
            $address->setShippingMethod("");
            $address->setShippingDescription("");
        }
    }

    public function saveSessionQuoteWithShippingRates(): void {
        $this->saveSessionQuote(true);
    }

    public function saveSessionQuote(bool $forceCollect = null): void {
        // TODO: Break out to make a separate save for quotes in general (unit testing)
        $quote = $this->getQuote();

        // Ensure that we have both a billing and a shipping address, and that
        // we collect rates since they are both new
        $quote->getBillingAddress();

        if( ! $quote->isVirtual()) {
            // Recollect shipping rates since we have a new product
            $shipping = $quote->getShippingAddress();

            if($forceCollect !== null) {
                $shipping->setCollectShippingRates($forceCollect);
            }
        }

        $this->setQuoteDefaults($quote);

        $quote->collectTotals();

        // Only perform this when we save
        if($quote->hasItems()) {
            Mage::helper("mageql_sales/quote")->trySetFreeIfAvailable($quote);
        }

        $quote->save();

        Mage::getSingleton("checkout/session")->setQuoteId($quote->getId());
    }

    /**
     * Adds a product to the current quote.
     */
    public function addProduct(
        Mage_Catalog_Model_Product $product,
        array $buyRequest
    ): Mage_Sales_Model_Quote_Item {
        $session = Mage::getSingleton("checkout/session");
        $quote = $this->getQuote();
        $req = new Varien_Object($buyRequest);

        $result = $quote->addProduct($product, $req);

        if(is_string($result)) {
            // Throw to let the mutation handle it
            throw new Mage_Core_Exception($result);
        }

        // Added quote item, affects shipping
        $this->saveSessionQuoteWithShippingRates();

        $session->setLastAddedProductId((int)$product->getId());

        return $result;
    }

    public function updateItem(
        Mage_Sales_Model_Quote_Item $item,
        Mage_Catalog_Model_Product $product,
        array $buyRequest
    ): Mage_Sales_Model_Quote_Item {
        $quote = $this->getQuote();
        $req = new Varien_Object($buyRequest);
        $result = $quote->updateItem($item->getId(), $req);

        // Update the item row specifically, saving the quote and session does
        // not do this
        $result->save();

        // Recollect shipping rates since we have modified a quantity
        $this->saveSessionQuoteWithShippingRates();

        // Wishlist listens for this
        Mage::dispatchEvent("checkout_cart_update_item_complete", [
            "product"  => $product,
            "request"  => Mage::app()->getRequest(),
            "response" => Mage::app()->getResponse(),
        ]);

        return $result;
    }

    /**
     * Sets defaults in the quote when the customer logs in to automatically
     * populate the address.
     */
    public function onCustomerLogin(Varien_Event_Observer $event): void {
        // getQuote sets defaults
        $this->getQuote()->save();
    }

    /**
     * @deprecated Use Mage::helper("mageql_sales/quote")->importPaymentData()
     * @psalm-suppress PossiblyUnusedReturnValue
     */
    public function importPaymentData(
        Mage_Sales_Model_Quote $quote,
        string $method,
        array $data
    ): Mage_Sales_Model_Quote_Payment {
        return Mage::helper("mageql_sales/quote")->importPaymentData($quote, $method, $data);
    }

    public function isValidCountryId(
        ?string $countryId,
        Mage_Core_Model_Store $store
    ): bool {
        if(!$countryId) {
            return false;
        }

        $approvedIds = Mage::helper("mageql/address")->getAllowedCountryCodes($store);

        return in_array($countryId, $approvedIds);
    }
}
