<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

use function MageQL\snakeCaseToCamel;

class MageQL_Sales_Model_Quote_Item extends Mage_Core_Model_Abstract {
    const SUCCESS = "success";
    const ERROR_PRODUCT_NOT_FOUND = "errorProductNotFound";
    const ERROR_PRODUCT_NOT_IN_STOCK = "errorProductNotInStock";
    const ERROR_PRODUCT_QUANTITY_NOT_AVAILABLE = "errorProductQuantityNotAvailable";
    const ERROR_PRODUCT_MAX_QUANTITY = "errorProductMaxQuantity";
    const ERROR_PRODUCT_VARIANT_NOT_AVAILABLE = "errorProductVariantNotAvailable";
    const ERROR_ITEM_NOT_FOUND = "errorQuoteItemNotFound";
    const ERROR_DECODE = "errorInvalidBuyRequest";

    public static function resolveBuyRequest(Mage_Sales_Model_Quote_Item $item): string {
        $request = [
            "i" => (int) $item->getId(),
            "p" => (int) $item->getProductId(),
        ];

        if($item->getProductType() === "configurable") {
            $request["a"] = array_map("intval", $item->getBuyRequest()->getSuperAttribute() ?: []);
        }

        return json_encode($request);
    }

    /**
     * @param mixed $unusedSrc
     * @param array{buyRequest:string, qty:float} $args
     * @return Mage_Sales_Model_Quote_Item|string
     */
    public static function mutateAdd(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx
    ) {
        $model = Mage::getSingleton("mageql_sales/quote");
        $qty = (float) $args["qty"];
        $request = json_decode($args["buyRequest"], true, 3);

        if(json_last_error()) {
            return self::ERROR_DECODE;
        }

        $productId = (int)($request["p"] ?? 0);
        $product = Mage::getModel("catalog/product")
            ->setStoreId($ctx->getStore()->getId())
            ->load($productId);

        if( ! Mage::helper("mageql_catalog")->isProductVisible($product, $ctx->getStore())) {
            return self::ERROR_PRODUCT_NOT_FOUND;
        }

        // TODO: Deal with bundle optioins
        // TODO: Deal with custom options (probably additional argument?)
        $buyRequest = [
            "qty" => self::clampProductQty($model->getQuote(), $product, $qty),
        ];

        if(array_key_exists("a", $request)) {
            // TODO: Validate?
            $buyRequest["super_attribute"] = $request["a"];
        }

        try {
            return $model->addProduct($product, $buyRequest);
        }
        catch(Mage_Core_Exception $e) {
            return self::translateAddProductError($e->getMessage(), $product, $e);
        }
    }

    /**
     * @param mixed $unusedSrc
     * @param array{itemBuyRequest:string, qty:float} $args
     * @return Mage_Sales_Model_Quote_Item|string
     */
    public static function mutateUpdateQty(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx
    ) {
        $model = Mage::getSingleton("mageql_sales/quote");
        $qty = (float) $args["qty"];
        $request = json_decode($args["itemBuyRequest"], true, 3);

        if(json_last_error()) {
            return self::ERROR_DECODE;
        }

        $itemId = (int)($request["i"] ?? 0);
        $productId = (int)($request["p"] ?? 0);

        if( ! $itemId) {
            // TODO: Different error sicne this is a buyRequest
            return self::ERROR_DECODE;
        }

        $quote = $model->getQuote();
        $product = Mage::getModel("catalog/product")
            ->setStoreId($ctx->getStore()->getId())
            ->load($productId);

        if( ! Mage::helper("mageql_catalog")->isProductVisible($product, $ctx->getStore())) {
            return self::ERROR_PRODUCT_NOT_FOUND;
        }

        // TODO: Ensure that it is not a bundle
        // TOOD: Custom options
        $buyRequest = [
            "product" => $product->getId(),
            "qty" => self::clampProductQty($quote, $product, $qty),
            // Needed to prevent magento from attempting to fetch the original qty + the new qty
            // if we do not have id here the old quote item is considered different and will be
            // first summed with this request and then deleted after the validation has been performed
            // It also needs to be a string to pass === in Mage_Sales_Model_Quote
            // If it is a different item,
            "id" => (string)$itemId,
        ];

        if(array_key_exists("a", $request)) {
            $buyRequest["super_attribute"] = $request["a"];
        }

        /**
         * @var ?Mage_Sales_Model_Quote_Item
         */
        $item = $quote->getItemById($request["i"]);

        if( ! $item) {
            return self::ERROR_ITEM_NOT_FOUND;
        }

        try {
            return $model->updateItem($item, $product, $buyRequest);
        }
        catch(Mage_Core_Exception $e) {
            return self::translateAddProductError($e->getMessage(), $product, $e);
        }
    }

    /**
     * @param mixed $unusedSrc
     * @param array{itemBuyRequest:string} $args
     */
    public static function mutateRemove($unusedSrc, array $args): string {
        $model = Mage::getSingleton("mageql_sales/quote");
        $request = json_decode($args["itemBuyRequest"], true, 3);

        if(json_last_error() || ! ($request["i"] ?? 0)) {
            return self::ERROR_DECODE;
        }

        $quote = $model->getQuote();
        /**
         * @var ?Mage_Sales_Model_Quote_Item
         */
        $item = $quote->getItemById($request["i"] ?? 0);

        if( ! $item || ! $item->getId()) {
            return self::ERROR_ITEM_NOT_FOUND;
        }

        $quote->removeItem($item->getId());

        // Recollect shipping rates since we have removed an item
        $model->saveSessionQuoteWithShippingRates();

        return self::SUCCESS;
    }

    /**
     * Clamps the product quantity if it is not a configurable product.
     *
     * @param float $qty
     */
    public static function clampProductQty(
        Mage_Sales_Model_Quote $quote,
        Mage_Catalog_Model_Product $product,
        $qty
    ): float {
        if($product->isConfigurable() ||
           !$product->getStockItem() ||
           $quote->hasProductId($product->getId())) {
            return $qty;
        }

        return max($product->getStockItem()->getMinSaleQty(), $qty);
    }

    public static function translateAddProductError(
        string $errorMsg,
        Mage_Catalog_Model_Product $product,
        Exception $prev = null
    ): string {
        $error = explode("\n", $errorMsg)[0];

        // TODO: Add option errors if bundle-products need it
        // Option errors cannot happen since we construct the attributes on the server

        if($error === Mage::helper("checkout")->__("The product could not be found.")) {
            return self::ERROR_PRODUCT_NOT_FOUND;
        }

        if($error === Mage::helper('cataloginventory')->__('This product is currently out of stock.')) {
            return self::ERROR_PRODUCT_NOT_IN_STOCK;
        }

        if(stripos($error, Mage::helper('sales')->__('Item qty declaration error.')) !== false ||
           stripos($error, Mage::helper('cataloginventory')->__('Item qty declaration error.\nThe requested quantity for \"%s\" is not available.', $product->getName())) !== false ||
           stripos($error, Mage::helper('cataloginventory')->__('The requested quantity for "%s" is not available.', $product->getName())) !== false) {
           return self::ERROR_PRODUCT_QUANTITY_NOT_AVAILABLE;
        }

        if(stripos($error, strstr(Mage::helper('cataloginventory')->__('The maximum quantity allowed for purchase is %s.', "DEADBEEF"), "DEADBEEF", true) ?: "DEADBEEF") !== false) {
           return self::ERROR_PRODUCT_MAX_QUANTITY;
        }

        if($error === Mage::helper('cataloginventory')->__('This product with current option is not available')) {
            return self::ERROR_PRODUCT_VARIANT_NOT_AVAILABLE;
        }

        throw new Exception($errorMsg, 0, $prev);
    }
}
