<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

/**
 * @psalm-type ConfigurationOptionItem array{0:Mage_Catalog_Model_Product, 1:Array<Mage_Catalog_Model_Product_Type_Configurable_Attribute>}
 */
class MageQL_Sales_Model_Product extends Mage_Core_Model_Abstract {
    /**
     * Generates a buy request for a simple/virtual product.
     */
    public static function resolveBuyRequestSimple(Mage_Catalog_Model_Product $src): ?string {
        switch($src->getTypeId()) {
        case "bundle":
        case "configurable":
            // Bundles and configs require a buyRequest from a variant
            return null;
        case "simple":
        case "virtual":
            if( ! $src->isSaleable()) {
                return null;
            }

            if($src->getParentId()) {
                return null;
            }

            return json_encode([
                "p" => (int)$src->getId(),
            ]);
        default:
            throw new Exception(sprintf("%s: Unknown product type %s", __METHOD__, $src->getTypeId()));
        }
    }

    /**
     * Generates a buy-request string for a configurable item which can be used
     * to add a product to a quote.
     *
     * @param ConfigurationOptionItem $src
     */
    public static function resolveBuyRequestConfigItem($src): ?string {
        return self::createConfigurableChildBuyRequest($src[0], $src[1]);
    }

    /**
     * @param Mage_Catalog_Model_Product $child
     * @param Array<Mage_Catalog_Model_Product_Type_Configurable_Attribute> $configAttrs
     */
    public static function createConfigurableChildBuyRequest($child, $configAttrs): ?string {
        $requestAttrs = [];

        if( ! in_array($child->getTypeId(), ["simple", "virtual"])) {
            throw new Exception(sprintf("%s: Invalid product type %s for config child", __METHOD__, $child->getTypeId()));
        }

        if( ! $child->getParentId()) {
            throw new Exception(sprintf("%s: Invalid parent product id on %s (%d) for config child", __METHOD__, $child->getSku(), $child->getId()));
        }

        if( ! $child->isSaleable()) {
            return null;
        }

        foreach($configAttrs as $attr) {
            $prodAttr = $attr->getProductAttribute();

            $requestAttrs[$prodAttr->getAttributeId()] = (int)$child->getData($prodAttr->getAttributeCode());
        }

        return json_encode([
            "p" => (int)$child->getParentId(),
            "a" => $requestAttrs,
        ]);
    }
}
