<?php

declare(strict_types=1);

use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use PHPUnit\Framework\TestCase;
use Spatie\Snapshots\MatchesSnapshots;

class LoginTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
    }

    public function onNotSuccessfulTest(Throwable $e): void {
        MagentoManager::logQueries();

        throw $e;
    }

    protected function customerFixture(): Mage_Customer_Model_Customer {
        $store = Mage::app()->getStore();
        $customer = Mage::getModel("customer/customer");

        $customer->setStore($store);
        $customer->loadByEmail("test-customer@example.com");
        $customer->addData([
            "website_id" => $store->getWebsiteId(),
            "email" => "test-customer@example.com",
            "firstname" => "Test",
            "lastname" => "Customer",
        ]);
        $customer->setStore($store);
        $customer->setPassword("test-customer");
        $customer->save();

        return $customer;
    }

    public function testLoginEmpty(): void {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            adminLoginAsCustomer(key: "FAIL") {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertEquals(null, $resp->getHeader("Location"));
    }

    public function testLoginEmptyRedirect(): void {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            adminLoginAsCustomer(key: "FAIL" redirect:true) {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertEquals(null, $resp->getHeader("Location"));
    }

    public function testLogin(): void {
        MagentoManager::init();

        $customer = $this->customerFixture();

        $key = Mage::helper("mageql_loginascustomer")->encode([
            "cid" => $customer->getId(),
            "t" => time(),
        ]);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            adminLoginAsCustomer(key: "'.$key.'") {
                result
                customer {
                    firstname
                    email
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertEquals(null, $resp->getHeader("Location"));
    }

    public function testLoginRedirect(): void {
        MagentoManager::init();

        $customer = $this->customerFixture();

        $key = Mage::helper("mageql_loginascustomer")->encode([
            "cid" => $customer->getId(),
            "t" => time(),
        ]);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            adminLoginAsCustomer(redirect: true, key: '.json_encode($key).') {
                result
                customer {
                    firstname
                    email
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(302, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertEquals("https://example.com/", $resp->getHeader("Location"));
    }

    public function testLoginTimeout(): void {
        MagentoManager::init();

        $customer = $this->customerFixture();

        $key = Mage::helper("mageql_loginascustomer")->encode([
            "cid" => $customer->getId(),
            "t" => time() - 6,
        ]);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            adminLoginAsCustomer(key: '.json_encode($key).') {
                result
                customer {
                    firstname
                    email
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertEquals(null, $resp->getHeader("Location"));
    }

    public function testLoginForm(): void {
        MagentoManager::init();

        $customer = $this->customerFixture();
        $helper = Mage::helper("mageql_loginascustomer");

        $key = $helper->encode([
            "cid" => $customer->getId(),
            "t" => time(),
        ]);

        $query = $helper->createQuery($key, false);
        $req = new Request("POST /graphql", [
            "Content-Type" => "application/x-www-form-urlencoded",
        ]);
        $req->setPost([ "query" => $query ]);

        $resp = MagentoManager::runRequest($req);

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertEquals(null, $resp->getHeader("Location"));
    }

    public function testLoginFormRedirect(): void {
        MagentoManager::init();

        $customer = $this->customerFixture();
        $helper = Mage::helper("mageql_loginascustomer");

        $key = $helper->encode([
            "cid" => $customer->getId(),
            "t" => time(),
        ]);

        $query = $helper->createQuery($key, true);
        $req = new Request("POST /graphql", [
            "Content-Type" => "application/x-www-form-urlencoded",
        ]);
        $req->setPost([ "query" => $query ]);

        $resp = MagentoManager::runRequest($req);

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(302, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertEquals("https://example.com/", $resp->getHeader("Location"));
    }
}
