<?php

declare(strict_types=1);

use Spatie\Snapshots\MatchesSnapshots;

use Crossroads\Magento\Test\Integration\TestCase;
use Crossroads\Magento\Test\Integration\Request;

/**
 * @runTestsInSeparateProcesses
 * @preserveGlobalState disabled
 */
class CmsPageTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        parent::setUp();

        // Set up for this test
        $this->initMagento();

        $storeId = Mage::app()->getStore()->getId();
        $defaultStoreId = Mage::app()->getStore("default")->getId();
        $page = Mage::getModel("cms/page");

        Mage::getModel("cms/page")->load($page->checkIdentifier("content-heading-page", $defaultStoreId))->addData([
            "identifier" => "content-heading-page",
            "title" => "Page With Content Heading Wrong Store",
            "content_heading" => "The Content Heading Wrong Store",
            "content" => "Content in page, but wrong store",
            "stores" => [$defaultStoreId],
        ])->save();
        Mage::getModel("cms/page")->load($page->checkIdentifier("content-heading-page", $storeId))->addData([
            "identifier" => "content-heading-page",
            "title" => "Page With Content Heading",
            "content_heading" => "The Content Heading",
            "content" => "Content in page",
            "stores" => [$storeId],
        ])->save();
        Mage::getModel("cms/page")->load("inactive-page", "identifier")->addData([
            "identifier" => "inactive-page",
            "title" => "Page is inactive",
            "content_heading" => "The Content Heading",
            "content" => "Content in page",
            "stores" => [$storeId],
            "is_active" => 0,
        ])->save();
        Mage::getModel("cms/page")->load($page->checkIdentifier("active-in-other-store-page", $defaultStoreId))->addData([
            "identifier" => "active-in-other-store-page",
            "title" => "Page should not be visible",
            "content_heading" => "The Content Heading",
            "content" => "Content in page",
            "stores" => [$defaultStoreId],
            "is_active" => 1,
        ])->save();

        // Create in reverse order to be able to test position
        Mage::getModel("cms/page")->load($page->checkIdentifier("another-nav-menu", $storeId))->addData([
            "identifier" => "another-nav-menu",
            "title" => "The second one",
            "content_heading" => "The Content Heading Two",
            "content" => "Content in page Two",
            "meta_keywords" => "Some,keywords",
            "meta_description" => "The Meta Description here",
            "include_in_menu" => 1,
            "sort_order" => 5,
            "stores" => [$storeId],
        ])->save();
        Mage::getModel("cms/page")->load($page->checkIdentifier("nav-menu-first", $storeId))->addData([
            "identifier" => "nav-menu-first",
            "title" => "The first",
            "content_heading" => "The Content Heading",
            "content" => "Content in page",
            "include_in_menu" => 1,
            "sort_order" => 4,
            "stores" => [$storeId],
        ])->save();
    }

    public function testContentHeading() {
        $resp = $this->runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
    cmsPage (identifier: "content-heading-page") {
        title,
        contentHeading,
        content
    }
}'));

        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $this->assertMatchesJsonSnapshot($resp->getBody());
    }

    public function testInactivePage() {
        $resp = $this->runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
    cmsPage (identifier: "inactive-page") {
        title
    }
}'));

        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $this->assertMatchesJsonSnapshot($resp->getBody());
    }

    public function testWrongStorePage() {
        $resp = $this->runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
    cmsPage (identifier: "active-in-other-store-page") {
        title
    }
}'));

        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $this->assertMatchesJsonSnapshot($resp->getBody());
    }
    public function test404() {
        $resp = $this->runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
    cmsPage (identifier: "does-not-exist-at-all") {
        title,
        metaKeywords,
        metaDescription,
        contentHeading,
        content
    }
}'));

        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
    }

    public function testExisting() {
        $resp = $this->runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
    cmsPage (identifier: "customer-service") {
        title,
        url,
        metaKeywords,
        metaDescription,
        contentHeading,
        content
    }
}'));

        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
    }

    public function testExtraFields() {
        $resp = $this->runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
    cmsPage (identifier: "no-route") {
        title,
        url,
        metaKeywords,
        metaDescription,
        contentHeading,
        content
    }
}'));

        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
    }

    public function testCmsPageNav() {
        $resp = $this->runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
    cmsPageNav {
        title,
        url,
    }
}'));

        $this->assertMatchesJsonSnapshot($resp->getBody());

        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertEquals(200, $resp->getHttpResponseCode());
    }

    public function testCmsPageNavAllFields() {
        $resp = $this->runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
    cmsPageNav {
        title,
        url,
        metaKeywords,
        metaDescription,
        contentHeading,
        content
    }
}'));

        $this->assertMatchesJsonSnapshot($resp->getBody());

        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertEquals(200, $resp->getHttpResponseCode());
    }

    public function testRoute() {
        $resp = $this->runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
    route (path: "customer-service") {
        type
        ...on RouteCmsPage {
            cmsPage {
                    title,
                    url,
                    metaKeywords,
                    metaDescription,
                    contentHeading,
                    content
            }
        }
    }
}'));

        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
    }

    public function testRoute2() {
        $resp = $this->runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
    route (path: "content-heading-page") {
        type
        ...on RouteCmsPage {
            cmsPage {
                    title,
                    url,
                    metaKeywords,
                    metaDescription,
                    contentHeading,
                    content
            }
        }
    }
}'));

        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
    }
}
