<?php

declare(strict_types=1);

use Spatie\Snapshots\MatchesSnapshots;

use Crossroads\Magento\Test\Integration\TestCase;
use Crossroads\Magento\Test\Integration\Request;

/**
 * @runTestsInSeparateProcesses
 * @preserveGlobalState disabled
 */
class CmsPageTest extends TestCase {
    use MatchesSnapshots;

    public function testContentHeading() {
        // Set up for this test
        $this->initMagento();

        $page = Mage::getModel("cms/page")->load("content-heading-page", "identifier");

        $page->addData([
            "identifier" => "content-heading-page",
            "title" => "Page With Content Heading",
            "content_heading" => "The Content Heading",
            "content" => "Content in page",
            "stores" => [Mage::app()->getStore()->getId()],
        ])->save();

        $resp = $this->runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
    cmsPage (identifier: "content-heading-page") {
        title,
        contentHeading,
        content
    }
}'));

        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $this->assertMatchesJsonSnapshot($resp->getBody());
    }

    public function testInactivePage() {
        // Set up for this test
        $this->initMagento();

        $page = Mage::getModel("cms/page")->load("inactive-page", "identifier");

        $page->addData([
            "identifier" => "inactive-page",
            "title" => "Page should not be visible",
            "content_heading" => "The Content Heading",
            "content" => "Content in page",
            "stores" => [Mage::app()->getStore()->getId()],
            "is_active" => 0,
        ])->save();

        $resp = $this->runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
    cmsPage (identifier: "inactive-page") {
        title
    }
}'));

        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $this->assertMatchesJsonSnapshot($resp->getBody());
    }

    public function testWrongStorePage() {
        // Set up for this test
        $this->initMagento();

        $page = Mage::getModel("cms/page")->load("active-in-other-store-page", "identifier");

        $page->addData([
            "identifier" => "active-in-other-store-page",
            "title" => "Page should not be visible",
            "content_heading" => "The Content Heading",
            "content" => "Content in page",
            "stores" => [],
            "is_active" => 1,
        ])->save();

        $resp = $this->runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
    cmsPage (identifier: "active-in-other-store-page") {
        title
    }
}'));

        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $this->assertMatchesJsonSnapshot($resp->getBody());
    }
    public function test404() {
        $resp = $this->runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
    cmsPage (identifier: "does-not-exist-at-all") {
        title,
        metaKeywords,
        metaDescription,
        contentHeading,
        content
    }
}'));

        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
    }

    public function testExisting() {
        $resp = $this->runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
    cmsPage (identifier: "customer-service") {
        title,
        url,
        metaKeywords,
        metaDescription,
        contentHeading,
        content
    }
}'));

        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
    }

    public function testExtraFields() {
        $resp = $this->runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
    cmsPage (identifier: "no-route") {
        title,
        url,
        metaKeywords,
        metaDescription,
        contentHeading,
        content
    }
}'));

        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
    }

    public function testCmsPageNav() {
        $this->initMagento();

        // Create in reverse order to be able to test position
        Mage::getModel("cms/page")
            ->load("another-nav-menu", "identifier")
            ->addData([
                "identifier" => "another-nav-menu",
                "title" => "The second one",
                "content_heading" => "The Content Heading",
                "content" => "Content in page",
                "include_in_menu" => 1,
                "sort_order" => 5,
                "stores" => [Mage::app()->getStore()->getId()],
            ])->save();
        Mage::getModel("cms/page")
            ->load("nav-menu-first", "identifier")
            ->addData([
                "identifier" => "nav-menu-first",
                "title" => "The first",
                "content_heading" => "The Content Heading",
                "content" => "Content in page",
                "include_in_menu" => 1,
                "sort_order" => 4,
                "stores" => [Mage::app()->getStore()->getId()],
            ])->save();

        $resp = $this->runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
    cmsPageNav {
        title,
        url,
    }
}'));

        $this->assertMatchesJsonSnapshot($resp->getBody());

        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertEquals(200, $resp->getHttpResponseCode());
    }

    public function testCmsPageNavAllFields() {
        $this->initMagento();

        // Create in reverse order to be able to test position
        Mage::getModel("cms/page")
            ->load("another-nav-menu", "identifier")
            ->addData([
                "identifier" => "another-nav-menu",
                "title" => "The second one",
                "content_heading" => "The Content Heading Two",
                "content" => "Content in page Two",
                "meta_keywords" => "Some,keywords",
                "meta_description" => "The Meta Description here",
                "include_in_menu" => 1,
                "sort_order" => 5,
                "stores" => [Mage::app()->getStore()->getId()],
            ])->save();
        Mage::getModel("cms/page")
            ->load("nav-menu-first", "identifier")
            ->addData([
                "identifier" => "nav-menu-first",
                "title" => "The first",
                "content_heading" => "The Content Heading",
                "content" => "Content in page",
                "include_in_menu" => 1,
                "sort_order" => 4,
                "stores" => [Mage::app()->getStore()->getId()],
            ])->save();

        $resp = $this->runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
    cmsPageNav {
        title,
        url,
        metaKeywords,
        metaDescription,
        contentHeading,
        content
    }
}'));

        $this->assertMatchesJsonSnapshot($resp->getBody());

        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertEquals(200, $resp->getHttpResponseCode());
    }

    public function testRoute() {
        $resp = $this->runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
    route (path: "customer-service") {
        type
        ...on RouteCmsPage {
            cmsPage {
                    title,
                    url,
                    metaKeywords,
                    metaDescription,
                    contentHeading,
                    content
            }
        }
    }
}'));

        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
    }
}
