<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

use MageQL\Registry;
use MageQL\Type\AbstractBuilder;

class MageQL_Cms_Model_Schema_Default extends MageQL_Core_Model_Schema_Abstract {
    const CMS_PAGE_FIELDS = [
        "title" => "title",
        "metaKeywords" => "meta_keywords",
        "metaDescription" => "meta_description",
        "url" => "identifier",
        "contentHeading" => "content_heading",
        "content" => "content",
    ];

    public function resolveCmsPageUrl($src) {
        return "/".$src->getIdentifier();
    }

    public function resolveCmsPage($src, array $args, $ctx, ResolveInfo $info) {
        $page = Mage::getModel("cms/page");

        $page->setStoreId($ctx->getStore()->getId());
        $page->load($args["identifier"], "identifier");

        if( ! $page->getId() || ! $page->getIsActive()) {
            return null;
        }

        return $page;
    }

    public function resolveCmsPageNav($src, array $args, $ctx, ResolveInfo $info) {
        $pages = Mage::getModel("cms/page")->getCollection();

        $pages->addStoreFilter($ctx->getStore());
        $pages->addFieldToFilter("is_active", 1);
        $pages->addFieldToFilter("include_in_menu", 1);
        // Do not include root in menu
        $pages->addFieldToFilter("identifier", ["neq" => ""]);
        $pages->setOrder("main_table.sort_order", "ASC");

        // TODO: This pattern should be reusable
        $columns = array_unique(array_merge(["page_id"], array_intersect_key(self::CMS_PAGE_FIELDS, $info->getFieldSelection())));

        foreach($columns as $col) {
            $pages->addFieldToSelect($col);
        }

        return $pages->getItems();
    }

    public function getTypeBuilder(string $typeName, Registry $registry): ?AbstractBuilder {
        switch($typeName) {
        case "CmsPage":
            return $this->object("CMS Page")
                        ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "RouteCmsPage":
            return $this->object("An object containing a CMS page")
                ->setInterfaces(["Route"]);
        }

        return null;
    }

    public function getTypeFields(string $typeName, Registry $registry): array {
        switch($typeName) {
        case "CmsPage":
            return [
                "content" => $this->field("String!", "Content"),
                "contentHeading" => $this->field("String", "Content heading"),
                "metaDescription" => $this->field("String", "Meta description"),
                "metaKeywords" => $this->field("String", "Meta keywords"),
                "title" => $this->field("String!", "Page title"),
                "url" => $this->field("String!", "Absolute URL to the page")
                    ->setResolver([$this, "resolveCmsPageUrl"]),
            ];

        case "Query":
            return [
                "cmsPage" => $this->field("CmsPage", "A CMS page identified by its identifier")
                    ->addArgument("identifier", $this->argument("String!", "Cms page identifier (slug)"))
                    ->setResolver([$this, "resolveCmsPage"]),
                "cmsPageNav" => $this->field("[CmsPage!]!", "Pages to include in a navigation menu")
                    ->setResolver([$this, "resolveCmsPageNav"]),
            ];

        case "RouteCmsPage":
            return [
                "type" => $this->field("RouteType!", "Type of route")
                    ->setResolver(function() {
                        return "cms_page";
                    }),
                "cmsPage" => $this->field("CmsPage!", "The CMS Page")
                    ->setResolver(function($src) {
                        // The url rewrite is actually the CMS page
                        return $src;
                    }),
            ];
        }

        return [];
    }

    public function getUnreachableTypes(): array {
        return [
            "RouteCmsPage",
        ];
    }
}
