<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

class MageQL_Cms_Model_Page {
    const CMS_PAGE_FIELDS = [
        "title" => "title",
        "metaKeywords" => "meta_keywords",
        "metaDescription" => "meta_description",
        "url" => "identifier",
        "contentHeading" => "content_heading",
        "content" => "content",
    ];

    /**
     * @param mixed $unusedSrc
     * @param array{identifier:string} $args
     */
    public static function resolve(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx
    ): ?Mage_Cms_Model_Page {
        $page = Mage::getModel("cms/page");
        $pageId = (int)$page->checkIdentifier($args["identifier"], $ctx->getStore()->getId());

        if( ! $pageId) {
            return null;
        }

        $page->load($pageId);

        if( ! $page->getId() || ! $page->getIsActive()) {
            return null;
        }

        return $page;
    }

    /**
     * @param mixed $unusedSrc
     * @return Array<Mage_Cms_Model_Page>
     */
    public static function resolveNav(
        $unusedSrc,
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx,
        ResolveInfo $info
    ): array {
        $pages = Mage::getModel("cms/page")->getCollection();

        $pages->addStoreFilter($ctx->getStore());
        $pages->addFieldToFilter("is_active", 1);
        $pages->addFieldToFilter("include_in_menu", 1);
        // Do not include root in menu
        $pages->addFieldToFilter("identifier", ["neq" => ""]);
        $pages->setOrder("main_table.sort_order", "ASC");

        // TODO: This pattern should be reusable
        $columns = array_unique(array_merge(["page_id"], array_intersect_key(self::CMS_PAGE_FIELDS, $info->getFieldSelection())));

        foreach($columns as $col) {
            $pages->addFieldToSelect($col);
        }

        return $pages->getItems();
    }

    public static function resolveUrl(Mage_Cms_Model_Page $src): string {
        return "/".$src->getIdentifier();
    }

    public static function resolveContent(Mage_Cms_Model_Page $src): string {
        $processor = Mage::helper("cms")->getPageTemplateProcessor();

        return $processor->filter($src->getContent());
    }

    public static function resolveRouteType(): string {
        return "cms_page";
    }
}
