<?php

declare(strict_types=1);

class MageQL_Catalog_Model_Product_Filter_Price
    extends MageQL_Catalog_Model_Product_Filter_Abstract {

    const MAX_PLACEHOLDER = 999999;

    /**
     * @readonly
     * @var Mage_Catalog_Model_Resource_Product_Collection
     */
    protected $collection;

    public function __construct(
        Mage_Catalog_Model_Resource_Product_Collection $collection
    ) {
        $this->collection = $collection;
    }

    public function getTypeName(): string {
        return "ProductFilterPrice";
    }

    public function getCode(): string {
        return "price";
    }

    public function getLabel(): string {
        return Mage::helper("mageql_catalog")->__("Price");
    }

    public function hasData(): bool {
        // We always have price data
        return true;
    }

    protected function createBaseSelect(): Zend_Db_Select {
        // Clone select data
        $select = clone $this->collection->getSelect();

        // If price filter is used, override it
        $wasChanged = false;
        $where = $select->getPart(Zend_Db_Select::WHERE);

        if( ! empty($where)) {
            foreach($where as $key => $part) {
                if(stripos($part, "price_index.") !== false) {
                    if ($key == 0) {
                        $where[$key] = "TRUE";
                    }
                    else {
                        $where[$key] = "AND TRUE";
                    }

                    $wasChanged = true;
                }
            }

            if($wasChanged) {
                $select->setPart(Zend_Db_Select::WHERE, $where);
            }
        }

        // Reset column selection and pagination
        $select->reset(Zend_Db_Select::COLUMNS);
        $select->reset(Zend_Db_Select::LIMIT_COUNT);
        $select->reset(Zend_Db_Select::LIMIT_OFFSET);
        $select->reset(Zend_Db_Select::ORDER);

        // Find out if tax rate table is already joined and if so, remove it.
        $foundTCR = false;
        $from = $select->getPart(Zend_Db_Select::FROM);

        if( ! empty($from)) {
            foreach(array_keys($from) as $key) {
                if($key == "tcr" || $key == "tc") {
                    unset($from[$key]);

                    $foundTCR = true;
                }
            }
        }

        // Overwrite FROM if tax rate table was found and removed
        if($foundTCR) {
            $select->setPart(Zend_Db_Select::FROM, $from);
        }

        // Correctly join in tax rate table
        $select->joinLeft(["tc" => "tax_class"], "tc.class_id = price_index.tax_class_id", []);
        $select->joinLeft(["tcr" => "tax_calculation_rate"], "tcr.code = tc.class_name", []);

        return $select;
    }

    public function getIncVat(): MageQL_Catalog_Model_Filter_Range {
        $select = clone $this->createBaseSelect();

        $taxSetting = Mage::getStoreConfig(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX);
        if (empty($taxSetting)) {
            // Magento instance use prices excluding tax
            $select->columns([
                "MIN(price_index.min_price * (1 + (COALESCE(tcr.rate, 0) / 100))) AS min",
                "MAX(price_index.min_price * (1 + (COALESCE(tcr.rate, 0) / 100))) AS max",
            ]);
        } else {
            // Magento instance use prices including tax
            $select->columns([
                "MIN(price_index.min_price) AS min",
                "MAX(price_index.min_price) AS max",
            ]);
        }

        $row = $this->collection->getConnection()->fetchRow($select);

        return new MageQL_Catalog_Model_Filter_AttributeRange((float)($row["min"] ?? 0), (float)($row["max"] ?? self::MAX_PLACEHOLDER));
    }

    public function getExVat(): MageQL_Catalog_Model_Filter_Range {
        $select = clone $this->createBaseSelect();

        $taxSetting = Mage::getStoreConfig(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX);
        if (empty($taxSetting)) {
            // Magento instance use prices excluding tax
            $select->columns([
                "MIN(price_index.min_price) AS min",
                "MAX(price_index.min_price) AS max",
            ]);
        } else {
            // Magento instance use prices including tax
            $select->columns([
                "MIN(price_index.min_price / (1 + (COALESCE(tcr.rate, 0) / 100))) AS min",
                "MAX(price_index.min_price / (1 + (COALESCE(tcr.rate, 0) / 100))) AS max",
            ]);
        }

        // Run query and return result
        $row = $this->collection->getConnection()->fetchRow($select);

        return new MageQL_Catalog_Model_Filter_AttributeRange((float)($row["min"] ?? 0), (float)($row["max"] ?? self::MAX_PLACEHOLDER));
    }
}
