<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;
use MageQL\Context;

/**
 * @template-extends MageQL_Catalog_Model_Attributes_Abstract<Mage_Catalog_Model_Product, array{ code:string, backend_type:string, input:string, label:string, required:bool, not_system:bool, attribute_set:Array<string>, attribute_set_type:Array<string>, apply_to:Array<string>, show_in_detail:bool, show_in_list:bool, filterable:bool, source_model:?string }>
 */
class MageQL_Catalog_Model_Attributes_Product extends MageQL_Catalog_Model_Attributes_Abstract {
    const SUPPORTED_TYPE_IDS = [
        "bundle",
        "configurable",
        "simple",
        "virtual",
    ];

    /**
     * @var Array<string>
     */
    protected $baseAttributes = [
        // To be able to determine which product it actually is
        "entity_id",
        "sku",
        "type_id",
        "attribute_set_id",
        // For shipping
        "weight",
        "weight_type",
        // For Bundle/Configurable
        "has_options",
        // For Bundle only
        "price_type",
        "price_view",
        "required_options",
    ];

    protected function getBaseAttributes(): array {
        return $this->baseAttributes;
    }

    protected function getEntityType(): string {
        return "catalog_product";
    }

    protected function getFieldAttributeMap(): array {
        return [
            "url" => ["url_key", "url_path"],
        ];
    }

    protected function filterAttributeData(array $a): array {
        $a =  parent::filterAttributeData($a);

        // Categories do not have any special types
        $a["apply_to"] = array_intersect($a["apply_to"], self::SUPPORTED_TYPE_IDS);

        // If the attribute applies to all supported types we include it as a default
        if(empty(array_diff(self::SUPPORTED_TYPE_IDS, $a["apply_to"]))) {
            $a["apply_to"] = [];
        }

        return $a;
    }

    public function addBaseAttribute(string $attributeCode): void {
        $this->baseAttributes[] = $attributeCode;
    }

    /**
     * Constructor for image-wrapper.
     *
     * @psalm-suppress PossiblyUnusedParam
     * @param Mage_Catalog_Model_Product $src
     * @return MageQL_Core_Model_Attributes_Image_Abstract
     */
    public function createImageWrapper(
        $src,
        Context $ctx,
        ResolveInfo $info,
        string $attrCode,
        string $value
    ): MageQL_Core_Model_Attributes_Image_Abstract {
        return new MageQL_Catalog_Model_Attributes_Image_Product($src, $attrCode, $value);
    }

    /**
     * Returns all attributes which can filter products.
     */
    public function getFilterableAttributes(): array {
        $attrs = [];

        foreach($this->getAttributes() as $field => $a) {
            if($a["filterable"]) {
                $attrs[$field] = $a;
            }
        }

        return $attrs;
    }

    /**
     * @psalm-suppress PossiblyUnusedReturnValue
     * @param mixed $unusedSrc
     */
    public static function resolveAttribute(
        $unusedSrc,
        array $unusedArgs,
        Context $unusedCtx,
        ResolveInfo $info
    ): Mage_Eav_Model_Entity_Attribute_Abstract {
        $attributeConfig = Mage::getSingleton("mageql_catalog/attributes_product");
        $attr = $attributeConfig->getAttributes()[$info->fieldName];
        $eav = Mage::getSingleton("catalog/config");

        $result = $eav->getAttribute(Mage_Catalog_Model_Product::ENTITY, $attr["code"]);

        if( ! $result) {
            throw new Exception(sprintf(
                "%s: Product attribute '%s' cannot be loaded",
                __METHOD__,
                $attr["code"]
            ));
        }

        return $result;
    }

    public static function resolveLabel(
        Mage_Catalog_Model_Resource_Eav_Attribute $attr
    ): string {
        return $attr->getStoreLabel() ?: $attr->getFrontend()->getLabel() ?: ucwords(str_replace("_", " ", $attr->getAttributeCode()));
    }

    public static function resolveValues(
        Mage_Catalog_Model_Resource_Eav_Attribute $attr,
        array $unusedArgs,
        Context $ctx
    ): ?array {
        if( ! $attr->usesSource()) {
            return null;
        }

        $resource  = Mage::getResourceSingleton("catalog/product_flat");
        $session = Mage::getSingleton("customer/session");
        $attrValue = $attr->getAttributeCode()."_value";
        $db = $resource->getReadConnection();

        $select = $db->select();

        $select->distinct();
        $select->from(["p" => $resource->getFlatTableName()], [$attrValue]);

        $select->join(
            ["f" => "catalog_product_index_price"],
            sprintf(
                "p.entity_id = f.entity_id AND f.website_id = %d AND f.customer_group_id = %d",
                $ctx->getStore()->getWebsiteId() ?: 0,
                $session->getCustomerGroupId()), []);

        $select->where("p.status = ?", Mage_Catalog_Model_Product_Status::STATUS_ENABLED);
        $select->where("p.type_id IN (?)", ["simple", "virtual"]);
        $select->where("p.$attrValue IS NOT NULL");

        $values = $db->query($select)->fetchAll(PDO::FETCH_COLUMN);

        sort($values);

        return array_filter($values);
    }
}
