<?php

declare(strict_types=1);

/**
 * @psalm-type ProductSortInput array{
 *   code: string,
 *   order: MageQL_Catalog_Model_Sort::ORDER_ASC|MageQL_Catalog_Model_Sort::ORDER_DESC,
 * }
 */
class MageQL_Catalog_Model_Product_Collection_Search
    extends MageQL_Catalog_Model_Product_AbstractSortableCollection {
    const SORT_RELEVANCE = "relevance";
    const DEFAULT_SORT = [
        "code" => self::SORT_RELEVANCE,
        "order" => MageQL_Catalog_Model_Sort::ORDER_DESC,
    ];

    /**
     * @var string
     */
    protected $fieldOrderExpression;

    public function __construct(Mage_Core_Model_Store $store, string $term) {
        $query = Mage::getModel("catalogsearch/query");
        $fulltext = Mage::getModel("catalogsearch/fulltext");
        $resource = $fulltext->getResource();

        $query->setStoreId($store->getId());
        $query->loadByQuery($term);

        if( ! $query->getId()) {
            $query->setQueryText($term);
            $query->setStoreId($store->getId());
            $query->setPopularity(1);
        }
        else {
            $query->setPopularity($query->getPopularity() + 1);
        }

        $query->prepare();

        $resource->prepareResult($fulltext, $term, $query);

        $searchHelper = Mage::getResourceHelper("catalogsearch");
        $collection = Mage::getModel("catalog/product")->getCollection();
        $foundData = $resource->getFoundData();

        ksort($foundData);
        natsort($foundData);

        $foundIds = array_keys($foundData);

        if(empty($foundIds)) {
            // We do not have any found ids, make sure the magento collection is empty.
            $foundIds = [-1];
        }

        $collection->addIdFilter($foundIds);
        $collection->addUrlRewrite();

        $this->fieldOrderExpression = $searchHelper->getFieldOrderExpression("e.entity_id", $foundIds);

        // Call parent before saving the query to filter
        parent::__construct($collection, $store);

        // Clone is necessary since the collection otheriwse would cache the
        // non-filtered total count
        $query->setNumResults((clone $collection)->getSize());
        $query->save();
    }

    public function getSortableBy(): array {
        $list = parent::getSortableBy();

        $list[] = New MageQL_Catalog_Model_Sort("relevance", Mage::helper("mageql_catalog")->__("Relevance"));

        return $list;
    }

    /**
     * @return ProductSortInput
     */
    public function getDefaultSort(): array {
        return self::DEFAULT_SORT;
    }

    /**
     * @param ProductSortInput $sortBy
     */
    public function applySort(array $sortBy): void {
        if($sortBy["code"] == self::SORT_RELEVANCE) {
            $select = $this->collection->getSelect();

            $select->order(new Zend_Db_Expr(sprintf(
                "%s %s, e.entity_id ASC",
                $this->fieldOrderExpression,
                $sortBy["order"]
            )));
        } else {
            parent::applySort($sortBy);
        }
    }
}
