<?php

declare(strict_types=1);

namespace MageQL\Catalog;

use Mage;
use Throwable;
use Varien_Profiler;

use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use PHPUnit\Framework\TestCase;
use Spatie\Snapshots\MatchesSnapshots;

class ProductTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
    }

    public function onNotSuccessfulTest(Throwable $e): void {
        MagentoManager::logQueries();

        throw $e;
    }

    public function testMissingParameter(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku {
                sku
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(400, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testMissingProduct(): void {
        try {
            // We use the profiler to test the event
            Varien_Profiler::enable();

            $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
                productBySku (sku: "test-non-existing") {
                    sku
                }
            }'));

            $this->assertMatchesJsonSnapshot($res->getBody());
            $this->assertEquals(200, $res->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:catalog_controller_product_view", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testUnassignedProduct(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-unassigned") {
                sku
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testInvisibleChildProduct(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-config-child-1") {
                sku
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testSimpleProduct(): void {
        try {
            // We use the profiler to test the event
            Varien_Profiler::enable();

            $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
                productBySku (sku: "test-simple") {
                    sku
                    type
                    name
                    url
                    attributeSetName
                    attributes {
                        description
                        metaDescription
                        metaKeyword
                        metaTitle
                        minimalPrice
                        msrp
                        msrpEnabled
                        shortDescription
                        smallImage {
                            src
                            smaller: src (width: 50)
                            resized: src (width: 20, height: 20)
                            fill: src (width: 20, height: 20, fill: true)
                            fillSquare: src (width: 20, fill: true)
                        }
                        smallImageLabel
                        thumbnail {
                            src
                        }
                    }
                }
            }'));

            $this->assertMatchesJsonSnapshot($res->getBody());
            $this->assertEquals(200, $res->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
            $this->assertNotFalse(Varien_Profiler::fetch("DISPATCH EVENT:catalog_controller_product_view", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testVirtualProduct(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-virtual") {
                sku
                type
                name
                url
                attributeSetName
                attributes {
                    description
                    metaDescription
                    metaKeyword
                    metaTitle
                    minimalPrice
                    msrp
                    msrpEnabled
                    shortDescription
                    smallImage {
                        src
                        smaller: src (width: 50)
                        resized: src (width: 20, height: 20)
                        fill: src (width: 20, height: 20, fill: true)
                        fillSquare: src (width: 20, fill: true)
                    }
                    smallImageLabel
                    thumbnail {
                        src
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testConfigurableProduct(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-config") {
                sku
                type
                name
                url
                attributeSetName
                attributes {
                    description
                    metaDescription
                    metaKeyword
                    metaTitle
                    minimalPrice
                    msrp
                    msrpEnabled
                    shortDescription
                    smallImage {
                        src
                        smaller: src (width: 50)
                        resized: src (width: 20, height: 20)
                        fill: src (width: 20, height: 20, fill: true)
                        fillSquare: src (width: 20, fill: true)
                    }
                    smallImageLabel
                    thumbnail {
                        src
                    }
                }
                ... on ProductDetailConfigurable {
                    configOptions {
                        attributes { attribute label }
                        items {
                            values { attribute value }
                            price { incVat exVat vat }
                            product {
                                sku
                                name
                                attributes {
                                    smallImage { src }
                                    thumbnail {
                                        src
                                    }
                                    shortDescription
                                    ... on ListProductSimpleAttributesDefault {
                                        color
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testBundleProductDefault(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-bundle-default") {
                sku
                type
                name
                url
                attributeSetName
                attributes {
                    description
                    metaDescription
                    metaKeyword
                    metaTitle
                    minimalPrice
                    msrp
                    msrpEnabled
                    shortDescription
                    smallImage {
                        src
                        smaller: src (width: 50)
                        resized: src (width: 20, height: 20)
                        fill: src (width: 20, height: 20, fill: true)
                        fillSquare: src (width: 20, fill: true)
                    }
                    smallImageLabel
                    thumbnail {
                        src
                    }
                }
                price { incVat exVat vat }
                ... on ProductDetailBundle {
                    bundleOptions {
                        required
                        type
                        title
                        selections {
                            isDefault
                            qty
                            isQtyFixed
                            price { incVat exVat vat }
                            product {
                                sku
                                name
                                attributes {
                                    shortDescription
                                    smallImage { src }
                                }
                            }
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testBundleProductDefaultIds(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-bundle-default") {
                sku
                ... on ProductDetailBundle {
                    bundleOptions {
                        optionId
                        selections {
                            selectionId
                            product {
                                sku
                            }
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testBundleProductHidden(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-bundle-hidden") {
                sku
                type
                name
                url
                attributeSetName
                attributes {
                    description
                    metaDescription
                    metaKeyword
                    metaTitle
                    minimalPrice
                    msrp
                    msrpEnabled
                    shortDescription
                    smallImage {
                        src
                        smaller: src (width: 50)
                        resized: src (width: 20, height: 20)
                        fill: src (width: 20, height: 20, fill: true)
                        fillSquare: src (width: 20, fill: true)
                    }
                    smallImageLabel
                    thumbnail {
                        src
                    }
                }
                price { incVat exVat vat }
                ... on ProductDetailBundle {
                    bundleOptions {
                        required
                        type
                        title
                        selections {
                            isDefault
                            qty
                            isQtyFixed
                            price { incVat exVat vat }
                            product {
                                sku
                                name
                                attributes {
                                    shortDescription
                                    smallImage { src }
                                }
                            }
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductInterfaceAttributesNotFound(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-simple") {
                sku
                type
                attributes {
                    ... on ProductDetailSimpleAttributes {
                        color
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(400, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductInterfaceAttributes(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-simple") {
                sku
                type
                attributes {
                    ... on ProductDetailSimpleAttributes {
                        description
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductMediaGallery(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-simple") {
                sku
                type
                gallery {
                    label
                    image {
                       src
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductMediaGalleryResize(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-simple") {
                sku
                type
                gallery {
                    label
                    image {
                        src
                        small: src (width: 200)
                        thumb: src (width: 50, height: 50)
                        odd: src (width: 50, height: 50, fill: true)
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testCrossSell(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-config") {
                sku
                crossSellProducts {
                    totalCount
                    items {
                        sku
                        attributes {
                            shortDescription
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testUpSell(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-config") {
                sku
                upSellProducts {
                    totalCount
                    items {
                        sku
                        attributes {
                            shortDescription
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testRelatedProducts(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-config") {
                sku
                relatedProducts {
                    totalCount
                    items {
                        sku
                        attributes {
                            shortDescription
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testRelatedProductsPaginated(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-config") {
                sku
                relatedProducts (pageSize: 1, page: 1) {
                    totalCount
                    items {
                        sku
                        attributes {
                            shortDescription
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));

        MagentoManager::reset();

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-config") {
                sku
                relatedProducts (pageSize: 1, page: 2) {
                    totalCount
                    items {
                        sku
                        attributes {
                            shortDescription
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));

        MagentoManager::reset();

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-config") {
                sku
                relatedProducts (pageSize: 1, page: 3) {
                    totalCount
                    items {
                        sku
                        attributes {
                            shortDescription
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductsByManufacturer(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql",
            ["Content-Type" => "application/graphql"],
        'query {
            productsBy {
                manufacturer (value: "Manufacturer A") {
                    totalCount
                    items {
                        sku
                        attributes {
                            shortDescription
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductAttributeValues(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql",
            ["Content-Type" => "application/graphql"],
        'query {
            productAttributes {
                manufacturer {
                    label
                    values
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductsBySearch(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql",
            ["Content-Type" => "application/graphql"],
        'query {
            productsBySearch(term: "Test") {
                totalCount
                items {
                    sku
                    attributes {
                        shortDescription
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));

        $query = Mage::getModel("catalogsearch/query");

        $query->loadByQuery("Test");

        $this->assertEquals(8, $query->getNumResults());
    }

    public function testProductsBySearchTooShort(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql",
            ["Content-Type" => "application/graphql"],
        'query {
            productsBySearch(term: "T") {
                totalCount
                items {
                    sku
                    attributes {
                        shortDescription
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductsBySearchNotFound(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql",
            ["Content-Type" => "application/graphql"],
        'query {
            productsBySearch(term: "this string does not match anything") {
                totalCount
                items {
                    sku
                    attributes {
                        shortDescription
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductsBySearchSortableBy(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productsBySearch( term: "Test", sort: { code: "name", order: ASC }) {
                totalCount
                sortableBy {
                    code
                    label
                }
                items {
                    sku
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductsBySearchSortNull(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productsBySearch( term: "Test", sort: null) {
                totalCount
                sortableBy {
                    code
                    label
                }
                items {
                    sku
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testMultiSort(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
          products(
            sort: { code: "price", order: DESC }
          ) {
            sortableBy {
              code
            }
          }
          total: products {
            count: totalCount
          }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductCustomOptions(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-custom-options") {
                sku
                customOptions {
                    title
                    required
                    type
                    values {
                        sku
                        title
                        price {
                            incVat
                            exVat
                            vat
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductCustomOptionsRequired(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-custom-options-required") {
                sku
                customOptions {
                    title
                    required
                    type
                    values {
                        sku
                        title
                        price {
                            incVat
                            exVat
                            vat
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }
}
