<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

/**
 * @psalm-type ProductFilterInput array{
 *   code: string,
 *   value?: ?string,
 *   minValue?: ?float,
 *   maxValue?: ?float,
 *   incVat?: ?bool,
 * }
 */
abstract class MageQL_Catalog_Model_Product_AbstractFilterableCollection
    extends MageQL_Catalog_Model_Product_AbstractCollection
    implements MageQL_Catalog_Model_Product_FilterableCollectionInterface {
    /**
     * @return Array<MageQL_Catalog_Model_Product_Filter_Abstract>
     */
    public function getFilterableBy(): array {
        /**
         * Map indexed by key to eliminate duplicates.
         *
         * @var Array<string, MageQL_Catalog_Model_Product_Filter_Abstract>
         */
        $filters = [];
        $config = Mage::getSingleton("mageql_catalog/attributes_product");
        $eav = Mage::getSingleton("catalog/config");
        $helper = Mage::helper("mageql_catalog");
        $custom = $helper->getCustomProductFilters(Mage::app()->getStore());

        foreach($custom as $model) {
            $filters = array_merge($filters, $model->getFilterableBy($this->collection));
        }

        foreach(array_reverse($config->getFilterableAttributes()) as $key => $attr) {
            if(array_key_exists($key, $filters)) {
                // Do nothing, we do not want to unnecessarily calculate the price
                continue;
            }

            $eavAttr = $eav->getAttribute(Mage_Catalog_Model_Product::ENTITY, $attr["code"]);

            if( ! $eavAttr || ! $eavAttr->getId()) {
                continue;
            }

           if ($attr["source_model"] === "eav/entity_attribute_source_boolean") {
                $filters[$key] = new MageQL_Catalog_Model_Product_Filter_Attribute_Boolean($eavAttr, $this->collection);
                continue;
            }

            switch($attr["input"]) {
            case "datetime":
                $filters[$key] = new MageQL_Catalog_Model_Product_Filter_Attribute_Range($eavAttr, $this->collection);

                break;

            case "select":
            case "multiselect":
                $filters[$key] = new MageQL_Catalog_Model_Product_Filter_Attribute_Bucket($eavAttr, $this->collection);

                break;

            default:
                switch($attr["backend_type"]) {
                case "int":
                case "decimal":
                    $filters[$key] = new MageQL_Catalog_Model_Product_Filter_Attribute_Range($eavAttr, $this->collection);

                    break;

                default:
                    Mage::log(sprintf("%s: Cannot create ProductsByFilter for attribute code %s", __METHOD__, $key));
                }
            }
        }

        return array_values(array_reverse(array_filter($filters, function($filter): bool {
            return $filter->hasData();
        })));
    }

    /**
     * @param Array<ProductFilterInput> $filters
     */
    public function setFilters(array $filters): void {
        $config = Mage::getSingleton("catalog/config");
        $helper = Mage::helper("mageql_catalog");
        $custom = $helper->getCustomProductFilters(Mage::app()->getStore());

        foreach($filters as $filter) {
            // Sanity-check to avoid loading by id:
            if(is_numeric($filter["code"])) {
                throw new MageQL_Catalog_NoSuchFilterException($filter["code"]);
            }

            foreach($custom as $c) {
                if($c->applyFilter($this->collection, $filter)) {
                    continue 2;
                }
            }

            $this->applyFilter($config, $filter);
        }
    }

    /**
     * @param ProductFilterInput $filter
     */
    protected function applyFilter(Mage_Eav_Model_Config $config, array $filter): void {
        $attr = $config->getAttribute(Mage_Catalog_Model_Product::ENTITY, $filter["code"]);

        if( ! $attr || ! $attr->getIsFilterable()) {
            throw new MageQL_Catalog_NoSuchFilterException($filter["code"]);
        }

        // Fetch real attribute code if input was something strange
        $attributeCode = $attr->getAttributeCode();
        $minValue = array_key_exists("minValue", $filter) ? ( $filter["minValue"] !== null ? floatval($filter["minValue"]) : null) : null;
        $maxValue = array_key_exists("maxValue", $filter) ? ( $filter["maxValue"] !== null ? floatval($filter["maxValue"]) : null) : null;

        // Check input type of attribute
        switch($attr->getFrontendInput()) {
            case "select":
                $attributeCode .= "_value";

                break;
            case "boolean":
                // Booleans uses the same column name
                break;

            default:
                throw new MageQL_Catalog_BadFilterTypeException($attr);
        }

        if(array_key_exists("value", $filter) && $filter["value"] !== null) {
            // Precise value have priority over min and max values, if specified
            $this->collection->addAttributeToFilter($attributeCode, ["eq" => $filter["value"]]);
        }
        else {
            if($minValue !== null && $minValue > 0) {
                $this->collection->addAttributeToFilter($attributeCode, ["gteq" => $minValue]);
            }

            if ($maxValue !== null && $maxValue > 0) {
                $this->collection->addAttributeToFilter($attributeCode, ["lteq" => $maxValue]);
            }
        }
    }
}
