<?php

declare(strict_types=1);

use GraphQL\Deferred;
use GraphQL\Type\Definition\ResolveInfo;

use function MageQL\snakeCaseToCamel;

/**
 * @psalm-type ProductFilterInput array{
 *   code: string,
 *   value: ?string,
 *   minValue: ?float,
 *   maxValue: ?float,
 *   incVat: ?bool,
 * }
 * @psalm-type ProductSortInput array {
 *   code: string,
 *   order: MageQL_Catalog_Model_Sort::ORDER_ASC|MageQL_Catalog_Model_Sort::ORDER_DESC
 * }
 */
class MageQL_Catalog_Model_Product extends Mage_Core_Model_Abstract {
    /**
     * Type resolver for ProductDetail.
     */
    public static function typeResolverDetail(Mage_Catalog_Model_Product $product): string {
        return "ProductDetail".ucfirst($product->getTypeId());
    }

    /**
     * Type resolver for ListProduct.
     */
    public static function typeResolverList(Mage_Catalog_Model_Product $product): string {
        return "ListProduct".ucfirst($product->getTypeId());
    }

    /**
     * Type resolver for ProductOption.
     */
    public static function typeResolverOption(Mage_Catalog_Model_Product $product): string {
        switch($product->getTypeId()) {
        case "simple":
            return "ProductOptionSimple";

        case "virtual":
            return "ProductOptionVirtual";

        default:
            throw new RuntimeException(sprintf(
                "%s: Invalid type id for ProductOption product: ''.",
                __METHOD__,
                $product->getTypeId()
            ));
        }
    }

    public static function resolveType(Mage_Catalog_Model_Product $product): string {
        return $product->getTypeId();
    }

    /**
     * @param mixed $unusedSrc
     * @param array{sku:string} $args
     */
    public static function resolveBySku(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx
    ): ?Mage_Catalog_Model_Product {
        $product = Mage::getModel("catalog/product");
        $id = $product->getIdBySku($args["sku"]);

        if($id) {
            $product->load($id);
        }

        if( ! Mage::helper("mageql_catalog")->isProductVisible($product, $ctx->getStore())) {
            return null;
        }

        // Magento event to log product visit
        Mage::dispatchEvent("catalog_controller_product_view", ["product" => $product]);

        return $product;
    }

    /**
     * Obtains the category already populated by loadRouteCategory.
     *
     * @see MageQL_Core_Model_Route
     */
    public static function resolveByRoute(
        Mage_Core_Model_Url_Rewrite $rewrite
    ): ?Mage_Catalog_Model_Product {
        // Set in loadRouteProduct
        return $rewrite->getData("product");
    }

    /**
     * @return Array<Mage_Catalog_Model_Product_Type_Configurable_Attribute>
     */
    public static function resolveConfigurationOptionAttributes(
        Mage_Catalog_Model_Product $src
    ): array {
        /**
         * @var Mage_Catalog_Model_Product_Type_Configurable
         */
        $instance = $src->getTypeInstance(true);

        // This will be cached on the product
        return $instance->getUsedProductAttributes($src);
    }

    /**
     * @return Array<MageQL_Catalog_Model_Product_Configurable_Option>
     */
    public static function resolveConfigurationOptionItems(
        Mage_Catalog_Model_Product $product,
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx,
        ResolveInfo $info
    ): array {
        /**
         * @var Mage_Catalog_Model_Product_Type_Configurable
         */
        $instance = $product->getTypeInstance(true);
        $prodAttrs = $instance->getConfigurableAttributes($product);

        // Local implementation of Mage_Catalog_Model_Product_Type_Configurable::getUsedProducts
        $collection = $instance->getUsedProductCollection($product);

        $collection->addFilterByRequiredOptions();
        $collection->addStoreFilter($ctx->getStore());
        $collection->addPriceData();
        $collection->addTaxPercents();
        $collection->addUrlRewrite();
        // TODO: Can we sort for this?
        $collection->addAttributeToSort("position", "asc");

        // No limit

        // Add the attributes we use for the configurating
        foreach($prodAttrs as $attr) {
            $collection->addAttributeToSelect($attr->getProductAttribute()->getAttributeCode());
        }

        // We use all attribute sets since we cannot be certain about which
        // sets we will get, even though they should be the same as parent
        $fields = Mage::helper("mageql_catalog")->getSelectedFields($info, ["product"]);

        foreach($fields as $field) {
            $collection->addAttributeToSelect($field);
        }

        return array_map(function(Mage_Catalog_Model_Product $item) use($product, $prodAttrs) {
            return new MageQL_Catalog_Model_Product_Configurable_Option($product, $item, $prodAttrs);
        }, $collection->getItems());
    }

    /**
     * @param array{
     *   page: ?int,
     *   pageSize: ?int
     * } $args
     */
    public static function resolveRelatedProducts(
        Mage_Catalog_Model_Product $src,
        array $args,
        MageQL_Core_Model_Context $ctx
    ): MageQL_Catalog_Model_Product_CollectionInterface {
        $collection = new MageQL_Catalog_Model_Product_Collection_ProductLink($src->getRelatedProductCollection(), $ctx->getStore());

        $collection->setPage($args["page"], $args["pageSize"]);

        return $collection;
    }

    /**
     * @param array{
     *   page: ?int,
     *   pageSize: ?int
     * } $args
     */
    public static function resolveUpSellProducts(
        Mage_Catalog_Model_Product $src,
        array $args,
        MageQL_Core_Model_Context $ctx
    ): MageQL_Catalog_Model_Product_CollectionInterface {
        $collection = new MageQL_Catalog_Model_Product_Collection_ProductLink($src->getUpSellProductCollection(), $ctx->getStore());

        $collection->setPage($args["page"], $args["pageSize"]);

        return $collection;
    }

    /**
     * @param array{
     *   page: ?int,
     *   pageSize: ?int
     * } $args
     */
    public static function resolveCrossSellProducts(
        Mage_Catalog_Model_Product $src,
        array $args,
        MageQL_Core_Model_Context $ctx
    ): MageQL_Catalog_Model_Product_CollectionInterface {
        $collection = new MageQL_Catalog_Model_Product_Collection_ProductLink($src->getCrossSellProductCollection(), $ctx->getStore());

        $collection->setPage($args["page"], $args["pageSize"]);

        return $collection;
    }

    /**
     * @param mixed $unusedSrc
     * @param array{
     *   page: ?int,
     *   pageSize: ?int,
     *   filter?: ?Array<ProductFilterInput>,
     * } $args
     */
    public static function resolveBestsellingProducts(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx
    ): MageQL_Catalog_Model_Product_FilterableCollectionInterface {
        $collection = new MageQL_Catalog_Model_Product_Collection_BestsellingProducts($ctx->getStore());

        $collection->setFilters($args["filter"] ?? []);
        $collection->setPage($args["page"], $args["pageSize"]);

        return $collection;
    }

    public static function resolveCategories(
        Mage_Catalog_Model_Product $src,
        array $unusedArgs,
        MageQL_Core_Model_Context $unusedCtx,
        ResolveInfo $info
    ): Deferred {
        $catAttrs = Mage::getSingleton("mageql_catalog/attributes_category");
        $toSelect = $catAttrs->getUsedAttributes(
            $catAttrs->getAttributesByArea(MageQL_Catalog_Model_Attributes_Abstract::AREA_ANY),
            $info->getFieldSelection(1)
        );

        $categories = MageQL_Catalog_Model_Product_Categories::instance();

        $categories->add($src->getId(), $toSelect);

        return new Deferred(function() use($categories, $src) {
            return $categories->load()->get($src->getId());
        });
    }

    /**
     * @deprecated
     * @param mixed $unusedSrc
     * @param array{
     *   page: ?int,
     *   pageSize: ?int,
     *   value: string
     * } | array{
     *   page: ?int,
     *   pageSize: ?int,
     *   to: string,
     *   from: string
     * } $args
     */
    public static function resolveProductsByFilter(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx,
        ResolveInfo $info
    ): MageQL_Catalog_Model_Product_CollectionInterface {
        $attributeConfig = Mage::getSingleton("mageql_catalog/attributes_product");
        $attrs = $attributeConfig->getAttributes();

        if( ! array_key_exists($info->fieldName, $attrs)) {
            throw new Exception(sprintf(
                "%s: Unknown attribute field %s.",
                __METHOD__,
                $info->fieldName
            ));
        }

        $attr = $attrs[$info->fieldName];
        $collection = Mage::getModel("catalog/product")->getCollection();
        $resource = Mage::getResourceModel("catalog/product");
        $attribute = $resource->getAttribute($attr["code"]);

        if( ! $attribute) {
            throw new Exception(sprintf(
                "%s: Product attribute %s could not be found",
                __METHOD__,
                $attr["code"]
            ));
        }

        if(array_key_exists("value", $args)) {
            $value = $attribute->usesSource() ?
                $attribute->getSource()->getOptionId($args["value"]) :
                $args["value"];

            $collection->addAttributeToFilter($attr["code"], [
                "eq" => $value,
            ]);
        }
        else if($attr["input"] === "datetime") {
            if( ! empty($attr["from"])) {
                $collection->addAttributeToFilter($attr["code"], [
                    "gteq" => date("Y-m-d H:i:s", strtotime($attr["from"])),
                ]);
            }

            if( ! empty($attr["to"])) {
                $collection->addAttributeToFilter($attr["code"], [
                    "lteq" => date("Y-m-d H:i:s", strtotime($attr["to"])),
                ]);
            }
        }
        else {
            if( ! empty($attr["from"])) {
                $collection->addAttributeToFilter($attr["code"], [
                    "gteq" => (float)$attr["from"],
                ]);
            }

            if( ! empty($attr["to"])) {
                $collection->addAttributeToFilter($attr["code"], [
                    "lteq" => (float)$attr["to"],
                ]);
            }
        }

        return new MageQL_Catalog_Model_Product_Collection_ProductsByFilter($collection, $ctx->getStore());
    }

    /**
     * @param mixed $unusedSrc
     * @param array{
     *   term: string,
     *   page: ?int,
     *   pageSize: ?int,
     *   filter?:?Array<ProductFilterInput>,
     *   sort: ?ProductSortInput,
     * } $args
     */
    public static function resolveProductsBySearch(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx
    ): ?MageQL_Catalog_Model_Product_SortableCollectionInterface {
        $minLength = $ctx->getStore()->getConfig(Mage_CatalogSearch_Model_Query::XML_PATH_MIN_QUERY_LENGTH);

        if(strlen($args["term"]) < $minLength) {
            return null;
        }

        $collection = new MageQL_Catalog_Model_Product_Collection_Search($ctx->getStore(), $args["term"]);

        $collection->setFilters($args["filter"] ?? []);
        $collection->setSort($args["sort"]);
        $collection->setPage($args["page"], $args["pageSize"]);

        return $collection;
    }

    /**
     * @param mixed $unusedSrc
     * @param array{
     *   page: ?int,
     *   pageSize: ?int,
     *   filter?: ?Array<ProductFilterInput>,
     *   sort: ?ProductSortInput,
     * } $args
     */
    public static function resolveProducts(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx
    ): MageQL_Catalog_Model_Product_SortableCollectionInterface {
        $collection = new MageQL_Catalog_Model_Product_Collection_Products($ctx->getStore());

        $collection->setFilters($args["filter"] ?? []);
        $collection->setSort($args["sort"]);
        $collection->setPage($args["page"], $args["pageSize"]);

        return $collection;
    }

    /**
     * @param mixed $unusedSrc
     * @param array{
     *   page: ?int,
     *   pageSize: ?int,
     *   filter?: ?Array<ProductFilterInput>,
     * } $args
     */
    public static function resolveRecentlyViewedProducts(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx
    ): ?MageQL_Catalog_Model_Product_FilterableCollectionInterface {
        $session = Mage::getSingleton("customer/session");

        if( ! $session->isLoggedIn()) {
            return null;
        }

        $collection = new MageQL_Catalog_Model_Product_Collection_RecentlyViewed($session->getCustomer(), $ctx->getStore());

        $collection->setFilters($args["filter"] ?? []);
        $collection->setPage($args["page"], $args["pageSize"]);

        return $collection;
    }

    /**
     * Event observer which loads the specified product for a redirect, if the
     * load fails it will replace the rewrite with null (interpreted as 404).
     *
     * @see MageQL_Core_Model_Route
     */
    public function loadRouteProduct(Varien_Event_Observer $event): void {
        /**
         * @var Mage_Core_Model_Url_Rewrite
         */
        $rewrite = $event->getRewrite();
        /**
         * @var MageQL_Core_Model_Context
         */
        $ctx = $event->getContext();
        $result = $event->getResult();

        if($event->getIsRedirect() || !$rewrite->getProductId()) {
            return;
        }

        $product = Mage::getModel("catalog/product");

        $product->load($rewrite->getProductId());

        if(Mage::helper("mageql_catalog")->isProductVisible($product, $ctx->getStore())) {
            // Save for later in resolveByRoute
            $rewrite->setData("product", $product);

            // Magento event to log product visit
            Mage::dispatchEvent("catalog_controller_product_view", ["product" => $product]);
        }
        else {
            // Not Found
            $result->setRewrite(null);
        }
    }
}
