<?php

declare(strict_types=1);

namespace MageQL\Catalog;

use Mage;
use MageQL_Sales_Model_BuyRequest;
use Mage_Core_Helper_Data;
use Mage_Core_Model_Resource_Setup;
use Mage_Customer_Model_Customer;
use Mage_Tax_Model_Config;
use Mage_Shipping_Model_Config;
use Throwable;
use Varien_Object;

use Crossroads\Magento\Test\Integration\Config;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use PHPUnit\Framework\TestCase;
use Spatie\Snapshots\MatchesSnapshots;

class InvalidCountryTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
        MagentoManager::initAdmin();

        // Testing tax rates do not match US, and is not in accepted countries in testing store
        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);
        $customer = Mage::getModel("customer/customer")
            ->setWebsiteId($store->getWebsiteId())
            ->loadByEmail("test-customer@example.com");

        foreach($customer->getAddresses() as $address) {
            $address->setCountryId("US");
            $address->save();
        }

        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::reset();
        MagentoManager::initAdmin();

        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);
        $customer = Mage::getModel("customer/customer")
            ->setWebsiteId($store->getWebsiteId())
            ->loadByEmail("test-customer@example.com");

        foreach($customer->getAddresses() as $address) {
            $address->setCountryId("SE");
            $address->save();
        }

        MagentoManager::reset();
    }

    public function onNotSuccessfulTest(Throwable $e): void {
        MagentoManager::logQueries();

        $this->tearDown();

        throw $e;
    }

    protected function customerFixture(): Mage_Customer_Model_Customer {
        MagentoManager::init();

        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);

        return Mage::getModel("customer/customer")
            ->setWebsiteId($store->getWebsiteId())
            ->loadByEmail("test-customer@example.com");
    }

    public function testDefaultCountrySimple(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-simple") {
                sku
                type
                name
                url
                attributeSetName
                attributes {
                    description
                    metaDescription
                    metaKeyword
                    metaTitle
                    minimalPrice
                    msrp
                    msrpEnabled
                    shortDescription
                    smallImage {
                        src
                        smaller: src (width: 50)
                        resized: src (width: 20, height: 20)
                        fill: src (width: 20, height: 20, fill: true)
                        fillSquare: src (width: 20, fill: true)
                    }
                    smallImageLabel
                    thumbnail {
                        src
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testCustomerMatchingSimple(): void {
        $this->customerFixture();

        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $session = Mage::getSingleton("customer/session");

        $session->login("test-customer@example.com", "test-customer");

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-simple") {
                sku
                type
                name
                url
                attributeSetName
                attributes {
                    description
                    metaDescription
                    metaKeyword
                    metaTitle
                    minimalPrice
                    msrp
                    msrpEnabled
                    shortDescription
                    smallImage {
                        src
                        smaller: src (width: 50)
                        resized: src (width: 20, height: 20)
                        fill: src (width: 20, height: 20, fill: true)
                        fillSquare: src (width: 20, fill: true)
                    }
                    smallImageLabel
                    thumbnail {
                        src
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testDefaultCountry(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-config") {
                sku
                type
                name
                url
                attributeSetName
                attributes {
                    description
                    metaDescription
                    metaKeyword
                    metaTitle
                    minimalPrice
                    msrp
                    msrpEnabled
                    shortDescription
                    smallImage {
                        src
                        smaller: src (width: 50)
                        resized: src (width: 20, height: 20)
                        fill: src (width: 20, height: 20, fill: true)
                        fillSquare: src (width: 20, fill: true)
                    }
                    smallImageLabel
                    thumbnail {
                        src
                    }
                }
                ... on ProductDetailConfigurable {
                    configOptions {
                        attributes { attribute label }
                        items {
                            values { attribute value }
                            price { incVat exVat vat }
                            product {
                                sku
                                name
                                attributes {
                                    smallImage { src }
                                    thumbnail {
                                        src
                                    }
                                    shortDescription
                                    ... on ListProductSimpleAttributesDefault {
                                        color
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testCustomerMatching(): void {
        $this->customerFixture();

        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $session = Mage::getSingleton("customer/session");

        $session->login("test-customer@example.com", "test-customer");

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-config") {
                sku
                type
                name
                url
                attributeSetName
                attributes {
                    description
                    metaDescription
                    metaKeyword
                    metaTitle
                    minimalPrice
                    msrp
                    msrpEnabled
                    shortDescription
                    smallImage {
                        src
                        smaller: src (width: 50)
                        resized: src (width: 20, height: 20)
                        fill: src (width: 20, height: 20, fill: true)
                        fillSquare: src (width: 20, fill: true)
                    }
                    smallImageLabel
                    thumbnail {
                        src
                    }
                }
                ... on ProductDetailConfigurable {
                    configOptions {
                        attributes { attribute label }
                        items {
                            values { attribute value }
                            price { incVat exVat vat }
                            product {
                                sku
                                name
                                attributes {
                                    smallImage { src }
                                    thumbnail {
                                        src
                                    }
                                    shortDescription
                                    ... on ListProductSimpleAttributesDefault {
                                        color
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }
}
