<?php

declare(strict_types=1);

use MageQL\Registry;
use MageQL\Type\AbstractBuilder;

class MageQL_Catalog_Model_Schema_Category extends MageQL_Core_Model_Schema_Abstract {
    public function getTypeBuilder(string $typeName, Registry $registry): ?AbstractBuilder {
        switch($typeName) {
        case "Category":
            return $this->object("A category containing products")
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "CategoryDisplayMode":
            $modes = Mage::getSingleton("catalog/category_attribute_source_mode")->getAllOptions();
            $keys = array_map(function($m): string { return $m["value"]; }, $modes);
            $values = array_map(function($m) { return [ "description" => $m["label"] ]; }, $modes);

            return $this->enum("Setting for category display mode", array_combine($keys, $values));

        case "RouteCategory":
            return $this->object("An instruction to redirect to the supplied path to obtain the resource")
                ->setInterfaces(["Route"]);
        }

        return null;
    }

    public function getTypeFields(string $typeName, Registry $registry): array {
        switch($typeName) {
        case "Category":
            // Name, description and everything:
            $config = Mage::getSingleton("mageql_catalog/attributes_category");
            // Category index everything, including all attribute sets
            $attrs = $config->getAttributesByArea(MageQL_Catalog_Model_Attributes_Abstract::AREA_ANY);
            $fields = $config->createFields($attrs);

            return array_merge($fields, [
                "children" => $this->field("[Category!]!", "Child-categories which are included in the menu")
                    ->setResolver("MageQL_Catalog_Model_Category::resolveChildren"),
                "displayMode" => $this->field("CategoryDisplayMode", "Category display mode"),
                "parent" => $this->field("Category", "Parent category, if any")
                    ->setResolver("MageQL_Catalog_Model_Category::resolveParent"),
                "products" => $this->field("SortableProductList!", "List of products in this category")
                    ->addArgument(
                        "filter",
                        $this->argument("[ProductFilterInput!]", "Filter products by these filters"))
                    ->addArgument(
                        "sort",
                        $this->argument("ProductSortInput", "Sorting to use for the products")
                             ->setDefaultValue(MageQL_Catalog_Model_Product_Collection_Category::DEFAULT_SORT))
                    ->addArgument(
                        "pageSize",
                        $this->argument("Int", "Maximum number of products to list")
                            ->setDefaultValue(MageQL_Catalog_Model_Product_CollectionInterface::DEFAULT_PAGE_SIZE))
                    ->addArgument(
                        "page",
                        $this->argument("Int", "Which page to show")
                            ->setDefaultValue(MageQL_Catalog_Model_Product_CollectionInterface::DEFAULT_PAGE))
                    ->setResolver("MageQL_Catalog_Model_Category::resolveProducts"),
                "url" => $this->field("String!", "Category url")
                    ->setResolver(function(Mage_Catalog_Model_Category $src): string {
                        return "/".trim($src->getRequestPath() ?: "", "/");
                    }),
            ]);

        case "Query":
            return [
                "categoryNav" => $this->field("[Category!]!", "Categories which can be shown in the main menu")
                    ->setResolver("MageQL_Catalog_Model_Category::resolveNavItems"),
                "rootCategory" => $this->field("Category!", "Root category for the store")
                    ->setResolver("MageQL_Catalog_Model_Category::resolveRoot"),
            ];

        case "RouteCategory":
            return [
                "type" => $this->field("RouteType!", "Type of route")
                    ->setResolver(function(): string {
                        return "category";
                    }),
                "category" => $this->field("Category!", "The Category")
                    ->setResolver("MageQL_Catalog_Model_Category::resolveByRoute"),
            ];
        }

        return [];
    }

    public function getUnreachableTypes(): array {
        return [
            "RouteCategory",
        ];
    }
}
