<?php

declare(strict_types=1);

namespace MageQL\Catalog;

use Mage;
use Throwable;
use Mage_Tax_Model_Config;
use Mage_Core_Model_Resource_Setup;

use Crossroads\Magento\Test\Integration\Config;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use PHPUnit\Framework\TestCase;
use Spatie\Snapshots\MatchesSnapshots;

class FilterTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
    }

    public function onNotSuccessfulTest(Throwable $e): void {
        $this->tearDown();

        throw $e;
    }

    public function testProductFilterDoesNotExist(): void {
        Mage::setIsDeveloperMode(false);

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            products (
                filter: [ { code: "this is weird", value: "Manufacturer B" } ]
            ) {
                totalCount
                items {
                    sku
                    price { incVat exVat vat }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(400, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductFilterDoesNotExist2(): void {
        Mage::setIsDeveloperMode(false);

        // 83 is manufacturer, but we should always get the same error message anyway
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            products (
                filter: [ { code: "83", value: "Manufacturer B" } ]
            ) {
                totalCount
                items {
                    sku
                    price { incVat exVat vat }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(400, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductPriceFilter(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            products (
                filter: [ { code: "price", minValue: 0, maxValue: 10 } ]
            ) {
                items {
                    sku
                    price { incVat exVat vat }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductsPriceFilter2(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
                products (
                    filter: [ { code: "price", minValue: 10, maxValue: 15 } ]
                ) {
                    items {
                        sku
                        price { incVat exVat vat }
                    }
                }
            }'
        ));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductPriceFilter3(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
                products (
                    filter: [ { code: "price", minValue: 0, maxValue: 5 } ]
                ) {
                    items {
                        sku
                        price { incVat exVat vat }
                    }
                }
            }'
        ));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductAttributeFilter(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
                products (
                    filter: [ { code: "manufacturer", value: "Manufacturer A" } ]
                ) {
                    totalCount
                    items {
                        sku
                    }
                }
            }'
        ));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductAttributeFilter2(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
                products (
                    filter: [ { code: "manufacturer", value: "Manufacturer B" } ]
                ) {
                    totalCount
                    items {
                        sku
                    }
                }
            }'
        ));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductAttributeFilter3(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
                products (
                    filter: [ { code: "manufacturer", value: "Manufacturer X" } ]
                ) {
                    totalCount
                    items {
                        sku
                    }
                }
            }'
        ));


        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductAttributeAndPriceFilter(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            products (
                filter: [ { code: "manufacturer", value: "Manufacturer B" }, { code: "price", minValue: 10 } ]
            ) {
                totalCount
                items {
                    sku
                    price { incVat exVat vat }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductAttributeAndPriceFilterMax(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            products (
                filter: [ { code: "manufacturer", value: "Manufacturer B" }, { code: "price", maxValue: 10 } ]
            ) {
                totalCount
                items {
                    sku
                    price { incVat exVat vat }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductPriceFilterWoVAT(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            products (
                filter: [ { code: "price", maxValue: 10, incVat: false } ]
            ) {
                totalCount
                items {
                    sku
                    price { incVat exVat vat }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductsFilterableBy(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            products {
                totalCount
                filterableBy {
                    __typename
                    code
                    label
                    ... on ProductFilterBucket { values { value } }
                    ... on ProductFilterPrice { incVat { min max } exVat { min max } }
                    ... on ProductFilterRange { range {  min max } }
                }
                items {
                    sku
                    price { incVat exVat vat }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductsFilterableByWithFilters(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            products (
                filter: [ { code:"manufacturer", value:"Manufacturer B" }, { code: "price", minValue:10, maxValue:20 } ]
            ) {
                totalCount
                filterableBy {
                    __typename
                    code
                    label
                    ... on ProductFilterBucket { values { value } }
                    ... on ProductFilterPrice { incVat { min max } exVat { min max } }
                    ... on ProductFilterRange { range { min max } }
                }
                items {
                    sku
                    price { incVat exVat vat }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductsSortableBy(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            products {
                totalCount
                sortableBy {
                    code
                    label
                }
                items {
                    sku
                    price { incVat exVat vat }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductsBadSortableAttribute(): void {
        Mage::setIsDeveloperMode(false);
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            products(sort: { code: "what is this", order: ASC }) {
                totalCount
                sortableBy {
                    code
                    label
                }
                items {
                    sku
                    price { incVat exVat vat }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(400, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductsSortNull(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            products(sort: null) {
                totalCount
                sortableBy {
                    code
                    label
                }
                items {
                    sku
                    price { incVat exVat vat }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductsBySearchFilter(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql",
            ["Content-Type" => "application/graphql"],
        'query {
            productsBySearch(term: "Test", filter: [{ code:"manufacturer", value:"Manufacturer B" }]) {
                totalCount
                items {
                    sku
                    attributes {
                        shortDescription
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));

        $query = Mage::getModel("catalogsearch/query");

        $query->loadByQuery("Test");

        // Still 6 items since the query itself results in 6 before filtering
        $this->assertEquals($query->getNumResults(), 6);
    }

    public function testProductAttributeFilterBoolean1(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
                products (
                    filter: [ { code: "is_featured", value: "1" } ]
                ) {
                    totalCount
                    items {
                        sku
                    }
                }
            }'
        ));


        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductAttributeFilterBoolean2(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
                products (
                    filter: [ { code: "is_featured", value: "0" } ]
                ) {
                    totalCount
                    items {
                        sku
                    }
                }
            }'
        ));


        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }
}