<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

class MageQL_Catalog_Model_Product_Bundle_Option {
    const TYPE_CHECKBOX = "checkbox";
    const TYPE_RADIO = "radio";
    const TYPE_SELECT = "select";
    const TYPE_MULTI = "multi";

    /**
     * @readonly
     * @var Mage_Bundle_Model_Option
     */
    protected $option;

    /**
     * @readonly
     * @var Mage_Catalog_Model_Product
     */
    protected $product;

    public static function isVisibleOption(Mage_Bundle_Model_Option $option): bool {
        return in_array($option->getType(), [
            self::TYPE_CHECKBOX,
            self::TYPE_RADIO,
            self::TYPE_SELECT,
            self::TYPE_MULTI,
        ]);
    }

    public function __construct(
        Mage_Bundle_Model_Option $option,
        Mage_Catalog_Model_Product $product
    ) {
        $this->option = $option;
        $this->product = $product;
    }

    public function getOptionId(): string {
        return $this->option->getId();
    }

    public function getRequired(): bool {
        return (bool)$this->option->getRequired();
    }

    public function getType(): string {
        return $this->option->getType();
    }

    public function getTitle(): string {
        return $this->option->getTitle() ?: $this->option->getDefaultTitle() ?: "";
    }

    public function getSelections(array $unusedArgs, MageQL_Core_Model_Context $ctx, ResolveInfo $info): array {
        /**
         * @var Mage_Bundle_Model_Resource_Selection_Collection
         */
        $collection = Mage::getResourceModel("bundle/selection_collection");

        $collection->setOptionIdsFilter([$this->option->getId()]);

        // No visibility filter here since bundle option selections can be invisible
        // URLs in case they are their own products
        $collection->addUrlRewrite();
        $collection->setPositionOrder();
        $collection->addStoreFilter($ctx->getStore());
        $collection->addPriceData();
        $collection->addTaxPercents();
        $collection->joinPrices((int)$ctx->getStore()->getWebsiteId());

        $fields = Mage::helper("mageql_catalog")->getSelectedFields($info, ["product"]);

        foreach($fields as $field) {
            $collection->addAttributeToSelect($field);
        }

        return array_map(function(Mage_Catalog_Model_Product $p): MageQL_Catalog_Model_Product_Bundle_Selection {
            return new MageQL_Catalog_Model_Product_Bundle_Selection($this, $p);
        }, $collection->getItems());
    }

    public function getProduct(): Mage_Catalog_Model_Product {
        return $this->product;
    }
}
