<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

/**
 * @psalm-type ProductFilterInput array{
 *   code: string,
 *   value?: ?string,
 *   minValue?: ?float,
 *   maxValue?: ?float,
 *   incVat?: ?bool,
 * }
 */
abstract class MageQL_Catalog_Model_Product_AbstractFilterableCollection
    extends MageQL_Catalog_Model_Product_AbstractCollection
    implements MageQL_Catalog_Model_Product_FilterableCollectionInterface {
    /**
     * @return Array<MageQL_Catalog_Model_Product_Filter_Abstract>
     */
    public function getFilterableBy(): array {
        /**
         * Map indexed by key to eliminate duplicates.
         *
         * @var Array<string, MageQL_Catalog_Model_Product_Filter_Abstract>
         */
        $filters = [];
        $config = Mage::getSingleton("mageql_catalog/attributes_product");
        $eav = Mage::getSingleton("eav/config");

        foreach($config->getFilterableAttributes() as $key => $attr) {
            $eavAttr = $eav->getAttribute(Mage_Catalog_Model_Product::ENTITY, $key);

            if( ! $eavAttr) {
                continue;
            }

            switch($attr["input"]) {
            case "price":
                // Do nothing, we do not want to unnecessarily calculate the price
                break;

            case "datetime":
                $filters[$key] = new MageQL_Catalog_Model_Product_Filter_Attribute_Range($eavAttr, $this->collection);

                break;

            case "select":
            case "multiselect":
                $filters[$key] = new MageQL_Catalog_Model_Product_Filter_Attribute_Bucket($eavAttr, $this->collection);

                break;

            default:
                switch($attr["backend_type"]) {
                case "int":
                case "decimal":
                    $filters[$key] = new MageQL_Catalog_Model_Product_Filter_Attribute_Range($eavAttr, $this->collection);

                    break;

                default:
                    Mage::log(sprintf("%s: Cannot create ProductsByFilter for attribute code %s", __METHOD__, $key));
                }
            }
        }

        $filters["price"] = $this->getPriceRange();

        return array_values(array_filter($filters, function($filter): bool {
            return $filter->hasData();
        }));
    }

    protected function getPriceRange(): MageQL_Catalog_Model_Product_Filter_Price {
        return new MageQL_Catalog_Model_Product_Filter_Price($this->collection);
    }

    /**
     * @param Array<ProductFilterInput> $filters
     */
    public function setFilters(array $filters): void {
        $config = Mage::getSingleton("eav/config");

        foreach($filters as $filter) {
            // Sanity-check to avoid loading by id:
            if(is_numeric($filter["code"])) {
                throw new MageQL_Catalog_NoSuchFilterException($filter["code"]);
            }

            switch($filter["code"]) {
            case "price":
                $this->applyPriceFilter($filter);

                break;

            default:
                $this->applyFilter($config, $filter);
            }
        }
    }

    /**
     * @param ProductFilterInput $filter
     */
    protected function applyFilter(Mage_Eav_Model_Config $config, array $filter): void {
        $attr = $config->getAttribute(Mage_Catalog_Model_Product::ENTITY, $filter["code"]);

        if( ! $attr || ! $attr->getIsFilterable()) {
            throw new MageQL_Catalog_NoSuchFilterException($filter["code"]);
        }

        // Fetch real attribute code if input was something strange
        $attributeCode = $attr->getAttributeCode();
        $minValue = array_key_exists("minValue", $filter) ? ( $filter["minValue"] !== null ? floatval($filter["minValue"]) : null) : null;
        $maxValue = array_key_exists("maxValue", $filter) ? ( $filter["maxValue"] !== null ? floatval($filter["maxValue"]) : null) : null;

        // Check input type of attribute
        switch($attr->getFrontendInput()) {
            case "select":
                $attributeCode .= "_value";

                break;

            default:
                throw new MageQL_Catalog_BadFilterTypeException($attr);
        }

        if(array_key_exists("value", $filter) && $filter["value"] !== null) {
            // Precise value have priority over min and max values, if specified
            $this->collection->addAttributeToFilter($attributeCode, ["eq" => $filter["value"]]);
        }
        else {
            if($minValue !== null && $minValue > 0) {
                $this->collection->addAttributeToFilter($attributeCode, ["gteq" => $minValue]);
            }

            if ($maxValue !== null && $maxValue > 0) {
                $this->collection->addAttributeToFilter($attributeCode, ["lteq" => $maxValue]);
            }
        }
    }

    /**
     * @param ProductFilterInput $filter
     */
    public function applyPriceFilter(array $filter): void {
        // Minimum and maximum values have already been filtered to be Floats by GraphQL
        $minValue = $filter["minValue"] ?? null;
        $maxValue = $filter["maxValue"] ?? null;
        $incVat = $filter["incVat"] ?? $this->store->getConfig(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX);
        $select = $this->collection->getSelect();

        if($incVat) {
            if($minValue !== null) {
                $select->where("price_index.min_price >= ?", $minValue);
            }

            if($maxValue !== null) {
                $select->where("price_index.min_price <= ?", $maxValue);
            }
        } else {
            // If incVat == false, then supplied minValue/maxValue must be specified excluding VAT
            $select->join(["tc" => "tax_class"], "tc.class_id = price_index.tax_class_id", []);
            $select->join(["tcr" => "tax_calculation_rate"], "tcr.code = tc.class_name", []);

            if($minValue !== null) {
                $select->where("price_index.min_price >= ? * (1 + (tcr.rate / 100))", $minValue);
            }

            if($maxValue !== null) {
                $select->where("price_index.min_price <= ? * (1 + (tcr.rate / 100))", $maxValue);
            }
        }
    }
}
