<?php

declare(strict_types=1);

namespace MageQL\Catalog;

use Mage;
use Throwable;

use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use PHPUnit\Framework\TestCase;
use Spatie\Snapshots\MatchesSnapshots;

class CategoryTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
    }

    public function onNotSuccessfulTest(Throwable $e): void {
        MagentoManager::logQueries();

        throw $e;
    }

    public function testRootCategory(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            rootCategory {
                url
                name
                image {
                    src
                    smaller: src(width: 50)
                    resized: src(width: 20, height: 20)
                    fill: src(width: 20, height: 20, fill: true)
                    fillSquare: src(width: 20, fill: true)
                }
                displayMode
                description
                metaDescription
                metaTitle
                metaKeywords
                parent {
                    url
                    name
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testNav(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            categoryNav {
                url
                name
                image {
                    src
                    smaller: src(width: 50)
                    resized: src(width: 20, height: 20)
                }
                displayMode
                description
                metaDescription
                metaTitle
                metaKeywords
                parent {
                    url
                    name
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testChildren(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            rootCategory {
                name
                children {
                    url
                    name
                    children {
                        url
                        name
                        children {
                            url
                            name
                            parent {
                                url
                                name
                            }
                        }
                        parent {
                            url
                            name
                        }
                    }
                }
                parent {
                    url
                    name
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProducts(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            categoryNav {
                name
                products {
                    totalCount
                    items {
                        sku
                        name
                        price { incVat }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductsEmpty(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            rootCategory {
                name
                products {
                    totalCount
                    items {
                        sku
                        name
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductsWConfigurable(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            categoryNav {
                name
                products {
                    totalCount
                    items {
                        sku
                        name
                        ... on ListProductBundle {
                            bundleOptions {
                                required
                                type
                                title
                                selections {
                                    isDefault
                                    price { incVat }
                                    product {
                                        sku
                                        name
                                        attributes {
                                            smallImage {
                                                src
                                            }
                                        }
                                    }
                                }
                            }
                        }
                        ... on ListProductConfigurable {
                            configOptions {
                                attributes { attribute label }
                                items {
                                    values { attribute value }
                                    price { incVat }
                                    product {
                                        sku
                                        name
                                        attributes {
                                            shortDescription
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testNavProductUrls(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            categoryNav {
                url
                name
                products {
                    totalCount
                    items {
                        sku
                        name
                        url
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testDeferredChildren(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            categoryNav {
                url
                name
            }
            route(path: "test-category.html") {
                type
                ... on RouteCategory {
                    category {
                        url
                        name
                        children {
                            url
                            name
                            image {
                                src
                            }
                            description
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testDeferredChildren2(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            categoryNav {
                url
                name
            }
            route(path: "test-category.html") {
                type
                ... on RouteCategory {
                    category {
                        url
                        name
                        children {
                            url
                            name
                            image {
                                src
                            }
                            description
                        }
                    }
                }
            }
            another: route(path: "test-category.html") {
                type
                ... on RouteCategory {
                    category {
                        url
                        name
                        smaller: image { src (width: 50) }
                        children {
                            url
                            name
                            description
                            displayMode
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductCategories(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-simple") {
                sku
                name
                attributes {
                    shortDescription
                }
                categories {
                    name
                    url
                    description
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testListProductCategories(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            categoryNav {
                name
                products {
                    totalCount
                    items {
                        sku
                        name
                        attributes {
                            shortDescription
                        }
                        categories {
                            name
                            url
                            description
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }
}
