<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

use function MageQL\snakeCaseToCamel;

use MageQL\Registry;
use MageQL\Type\AbstractBuilder;
use MageQL\Type\FieldBuilder;
use MageQL\Type\InterfaceBuilder;
use MageQL\Type\ObjectBuilder;

/**
 * @psalm-type Area MageQL_Catalog_Model_Attributes_Abstract::AREA_DETAIL|MageQL_Catalog_Model_Attributes_Abstract::AREA_LIST
 * @psalm-type ConfigurationOptionItemValue array{attribute:string, value:string}
 * @psalm-type ConfigurationOptionItem array{0:Mage_Catalog_Model_Product, 1:Array<Mage_Catalog_Model_Product_Type_Configurable_Attribute>}
 */
class MageQL_Catalog_Model_Schema_Default_Product extends MageQL_Core_Model_Schema_Abstract {
    const DEFAULT_LINK_SIZE = 4;
    const DEFAULT_PAGE_SIZE = 20;

    /**
     * @var MageQL_Catalog_Model_Attributes_Product
     */
    protected $config;

    public function __construct(array $args) {
        parent::__construct($args);

        $this->config = Mage::getSingleton("mageql_catalog/attributes_product");
    }

    public function getTypeBuilder(string $typeName, Registry $registry): ?AbstractBuilder {
        switch($typeName) {
        case "BundleOption":
            return $this->object("An option and its product selections for a bundle product")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "BundleSelection":
            return $this->object("A selection for a bundle option which can be picked by the customer")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "BundleOptionType":
            return $this->enum("The type of bundle option input", [
                MageQL_Catalog_Model_Product_Bundle_Option::TYPE_CHECKBOX => [
                    "description" => "A checkbox should be used, or a yes-no selector, to pick zero or more selections",
                ],
                MageQL_Catalog_Model_Product_Bundle_Option::TYPE_RADIO => [
                    "description" => "A radio button should be used to pick a single selection from the option",
                ],
                MageQL_Catalog_Model_Product_Bundle_Option::TYPE_SELECT => [
                    "description" => "A dropdown should be used for the option allowing the customer to pick one selection",
                ],
                MageQL_Catalog_Model_Product_Bundle_Option::TYPE_MULTI => [
                    "description" => "A multiselect should be used for the option allowing the customer to pick zero or more selections",
                ],
            ]);

        case "ConfigurationAttribute":
            return $this->object("An attribute which can be configured on configurable products");

        case "ConfigurationOptionItem":
            return $this->object("A child-product/configuration-item available to pick for a configurable product");

        case "ConfigurationOptionItemValue":
            return $this->object("A value for the configurable attribute on configurable products");

        case "ConfigurationOptions":
            return $this->object("Available options and their attributes for a configurable product");

        case "GalleryItem":
            return $this->object("Media Gallery Image");

        case "ListProduct":
            return $this->interface("A partially populated product of unknown type")
                ->setTypeResolver("MageQL_Catalog_Model_Product::typeResolverList");

        case "ProductAttribute":
            return $this->object("Information about a filterable product attribute");

        case "ProductAttributes":
            return $this->object("Filterable product attributes and their values");

        case "ProductType":
            return $this->enum("Type indicating variant of product", [
                "bundle" => [
                    "description" => "Complex product containing multiple variants",
                ],
                "configurable" => [
                    "description" => "Complex product containing variants",
                ],
                "simple" => [
                    "description" => "Simple single product",
                ],
                "virtual" => [
                    "description" => "Simple product without physical representation",
                ],
            ]);

        case "PaginatedProducts":
            return $this->object("Object containing a list of partially populated products");

        case "ProductDetail":
            return $this->interface("A fully populated product of unknown type")
                ->setTypeResolver("MageQL_Catalog_Model_Product::typeResolverDetail");

        case "ProductOption":
            return $this->interface("A product which is a possible option or selection in a configurable or bundle product")
                ->setTypeResolver("MageQL_Catalog_Model_Product::typeResolverOption");

        case "ProductOptionSimple":
            return $this->object("A simple product available as option or selection for configurable product or bundle")
                ->setInterfaces(["ProductOption"]);

        case "ProductOptionVirtual":
            return $this->object("A virtual product available as option or selection for configurable product or bundle")
                ->setInterfaces(["ProductOption"]);

        case "ProductPrice":
            return $this->object("Price information for a product in base store currency")
                ->setInterfaces(["Price"]);

        case "ProductsBy":
            return $this->object("Available filters for products");

        case "RouteProduct":
            return $this->object("A response containing a detailed product")
                ->setInterfaces(["Route"]);

        case "ProductAttributeFilter":
            return $this->inputObject("Type describing a filter of products based on attribute code, if the attribute is a Bucket attribute use value, otherwise use at least one of minValue/maxValue for range-attributes");

        case "ProductPriceFilter":
            return $this->inputObject("Type describing a filter of products based on price min/max");
        }

        if(strpos($typeName, "ProductDetail") === 0) {
            return $this->createProductTypeBuilder(
                MageQL_Catalog_Model_Attributes_Abstract::AREA_DETAIL,
                substr($typeName, strlen("ProductDetail"))
            );
        }

        if(strpos($typeName, "ListProduct") === 0) {
            return $this->createProductTypeBuilder(
                MageQL_Catalog_Model_Attributes_Abstract::AREA_LIST,
                substr($typeName, strlen("ListProduct"))
            );
        }

        return null;
    }

    public function getTypeFields(string $typeName, Registry $registry): array {
        switch($typeName) {
        case "BundleOption":
            return [
                "optionId" => $this->field("ID!", "Bundle option id, used to select an option when adding a quote item"),
                "required" => $this->field("Boolean!", "If picking an option is required"),
                "type" => $this->field("BundleOptionType!", "The type of input used for this option"),
                "title" => $this->field("String!", "Title for the option"),
                "selections" => $this->field("[BundleSelection!]!", "The list of product selections for this option")
            ];

        case "BundleSelection":
            return [
                "selectionId" => $this->field("ID!", "Selection id to pick this product"),
                "isDefault" => $this->field("Boolean!", "If this selection is included by default"),
                "qty" => $this->field("Float!", "The number of items this selection has by default"),
                "isQtyFixed" => $this->field("Boolean!", "If the qty can be modified"),
                "price" => $this->field("ProductPrice!", "The price of this option"),
                "product" => $this->field("ProductOption!", "The product for this bundle selection"),
            ];

        case "ConfigurationAttribute":
            return [
                "attribute" => $this->field("String!", "Attribute property")
                    ->setResolver(function(Mage_Catalog_Model_Product_Type_Configurable_Attribute $src): string {
                        return snakeCaseToCamel($src->getProductAttribute()->getAttributeCode());
                    }),
                "label" => $this->field("String!", "Attribute label")
                    ->setResolver(function(Mage_Catalog_Model_Product_Type_Configurable_Attribute $src): string {
                        $prodAttr = $src->getProductAttribute();

                        return $prodAttr->getStoreLabel() ?: $prodAttr->getFrontend()->getLabel() ?: $src->getLabel();
                    }),
            ];

        case "ConfigurationOptionItem":
            return [
                // TODO: Do we have any special considerations for configurable price here?
                // SimpleConfigurableProducts might mess a bit with it
                "price" => $this->field("ProductPrice!", "Product final price")
                    ->setResolver(
                        /**
                         * @param ConfigurationOptionItem $src
                         */
                        function($src) {
                            $product = $src[0];

                            return new MageQL_Catalog_Model_Product_Price($product, $product->getFinalPrice());
                        }),
                "product" => $this->field("ProductOption!", "The product")
                    ->setResolver(
                        /**
                         * @param ConfigurationOptionItem $src
                         */
                        function($src): Mage_Catalog_Model_Product {
                            return $src[0];
                        }),
                "values" => $this->field("[ConfigurationOptionItemValue!]!", "List of values this item fulfills")
                    ->setResolver("MageQL_Catalog_Model_Product::resolveConfigurationOptionItemValues"),
            ];

        case "ConfigurationOptionItemValue":
            return [
                "attribute" => $this->field("String!", "Attribute property this value belongs to"),
                "value" => $this->field("String!", "Attribute value"),
            ];

        case "ConfigurationOptions":
            return [
                "attributes" => $this->field("[ConfigurationAttribute!]!", "List of configurable attributes")
                    ->setResolver("MageQL_Catalog_Model_Product::resolveConfigurationOptionAttributes"),
                "items" => $this->field("[ConfigurationOptionItem!]!", "List of items to pick from")
                    ->setResolver("MageQL_Catalog_Model_Product::resolveConfigurationOptionItems"),
            ];

        case "GalleryItem":
            return [
                "image" => $this->field("ImageData!", "Gallery item image data")
                    ->addArgument("width", $this->argument("Int", "Maximum width of the image"))
                    ->addArgument("height", $this->argument("Int", "Minimum width of the image"))
                    ->addArgument(
                        "fill",
                        $this->argument("Boolean", "If to fill the image to the given size")
                            ->setDefaultValue(false))
                    ->setResolver(
                        /**
                         * @param array{0:Mage_Catalog_Model_Product, 1:Varien_Object} $src
                         */
                        function($src, array $unusedArgs, MageQL_Core_Model_Context $ctx, ResolveInfo $info) {
                            [$product, $image] = $src;

                            $value = $image->getFile();

                            if( ! $value || $value === "/no_selection") {
                                return null;
                            }

                            return $this->config->createImageWrapper($product, $ctx, $info, "gallery", $value);
                        }),
                "label" => $this->field("String", "Image label")
                    ->setResolver(
                        /**
                         * @param array{0:Mage_Catalog_Model_Product, 1:Varien_Object} $src
                         */
                        function($src): string {
                            $image = $src[1];

                            return $image->getLabel();
                        }),
            ];

        case "ListProduct":
            return [
                // TODO: Some should be reusable here
                "sku" => $this->field("String!", "SKU"),
                "type" => $this->field("ProductType!", "Product type")
                    ->setResolver("MageQL_Catalog_Model_Product::resolveType"),
                "name" => $this->field("String!", "Product name"),
                "attributes" => $this->field("ListProductAttributes!", "Product attributes for list")
                    ->setResolver("MageQL\\forwardResolver"),
                "attributeSetName" => $this->field("String!", "Attribute set name")
                    ->setResolver(function(Mage_Catalog_Model_Product $src) {
                        return $this->config->getSetById($src->getAttributeSetId())["name"];
                    }),
                "categories" => $this->field("[Category!]!", "Categories this product is part of")
                    ->setResolver("MageQL_Catalog_Model_Product::resolveCategories"),
                "originalPrice" => $this->field("ProductPrice!", "Product original price")
                    ->setResolver(function(Mage_Catalog_Model_Product $src) {
                        return new MageQL_Catalog_Model_Product_Price($src, $src->getPrice());
                    }),
                "price" => $this->field("ProductPrice!", "Product final price")
                    ->setResolver(function(Mage_Catalog_Model_Product $src) {
                        return new MageQL_Catalog_Model_Product_Price($src, $src->getFinalPrice());
                    }),
                // TODO: Fields
                // stock
                // options (configurable/bundle)
                "url" => $this->field("String!", "URL")
                    ->setResolver(function(Mage_Catalog_Model_Product $src) {
                        return $this->context->stripBaseUrl($src->getProductUrl());
                    }),
            ];

        case "ListProductAttributes":
            return $this->config->createFields(
                $this->config->getSystemAttributes(MageQL_Catalog_Model_Attributes_Abstract::AREA_LIST)
            );

        case "ListProductBundle":
            // TODO: Specifc fields which are bundle only?
            return array_merge($registry->getFieldBuilders("ListProduct"), [
                "bundleOptions" => $this->field("[BundleOption!]!", "Options for a bundle product")
                    ->setResolver("MageQL_Catalog_Model_Product_Bundle::resolveOptions"),
            ]);

        case "ListProductConfigurable":
            // TODO: Specifc fields which are configurable only?
            return array_merge($registry->getFieldBuilders("ListProduct"), [
                "configOptions" => $this->field("ConfigurationOptions!", "Options for a configurable product")
                    ->setResolver("MageQL\\forwardResolver"),
            ]);

        case "ListProductSimple":
            // TODO: Specifc fields which are simple only?
            return $registry->getFieldBuilders("ListProduct");

        case "ListProductVirtual":
            // TODO: Specifc fields which are virtual only?
            return $registry->getFieldBuilders("ListProduct");

        case "PaginatedProducts":
            return [
                "items" => $this->field("[ListProduct!]!", "List of products")
                    ->setResolver("MageQL_Catalog_Model_Product::resolvePaginatedItems"),
                "totalCount" => $this->field("Int!", "Total number of products in this paginated collection")
                    ->setResolver(function(Varien_Data_Collection $collection) {
                        return $collection->getSize();
                    }),
            ];

        case "ProductAttribute":
            return [
                "label" => $this->field("String!", "The attribute label")
                    ->setResolver("MageQL_Catalog_Model_Attributes_Product::resolveLabel"),
                "values" => $this->field("[String!]", "If the attribute has different values which can be selected they will be listed here")
                    ->setResolver("MageQL_Catalog_Model_Attributes_Product::resolveValues"),
            ];

        case "ProductAttributes":
            return $this->createProductAttributesFields();

        case "ProductDetail":
            return [
                "sku" => $this->field("String!", "SKU"),
                "type" => $this->field("ProductType!", "Product type")
                    ->setResolver("MageQL_Catalog_Model_Product::resolveType"),
                "name" => $this->field("String!", "Product name"),
                "attributes" => $this->field("ProductDetailAttributes!", "Product attributes for detail page")
                    ->setResolver("MageQL\\forwardResolver"),
                "attributeSetName" => $this->field("String!", "Attribute set name")
                    ->setResolver(function(Mage_Catalog_Model_Product $src) {
                        return $this->config->getSetById((int)$src->getAttributeSetId())["name"];
                    }),
                "categories" => $this->field("[Category!]!", "Categories this product is part of")
                    ->setResolver("MageQL_Catalog_Model_Product::resolveCategories"),
                "gallery" => $this->field("[GalleryItem]!", "Media gallery")
                    ->setResolver(function(Mage_Catalog_Model_Product $product) {
                        return array_map(function($image) use($product) {
                            // The product instance is required to be able to generate a link
                            return [$product, $image];
                        }, $product->getMediaGalleryImages()->getItems());
                    }),
                "originalPrice" => $this->field("ProductPrice!", "Product original price")
                    ->setResolver(function(Mage_Catalog_Model_Product $src) {
                        return new MageQL_Catalog_Model_Product_Price($src, $src->getPrice());
                    }),
                "price" => $this->field("ProductPrice!", "Product final price")
                    ->setResolver(function(Mage_Catalog_Model_Product $src) {
                        return new MageQL_Catalog_Model_Product_Price($src, $src->getFinalPrice());
                    }),
                "url" => $this->field("String!", "URL")
                    ->setResolver(function(Mage_Catalog_Model_Product $src) {
                        return $this->context->stripBaseUrl($src->getProductUrl());
                    }),
                "crossSellProducts" => $this->field("PaginatedProducts!", "Cross-sell products")
                    ->addArgument(
                        "pageSize",
                        $this->argument("Int", "Maximum number of products to list")
                            ->setDefaultValue(self::DEFAULT_LINK_SIZE))
                    ->addArgument(
                        "page",
                        $this->argument("Int", "Which page to show")
                            ->setDefaultValue(1))
                    ->setResolver("MageQL_Catalog_Model_Product::resolveCrossSellProducts"),
                "relatedProducts" => $this->field("PaginatedProducts!", "Related products")
                    ->addArgument(
                        "pageSize",
                        $this->argument("Int", "Maximum number of products to list")
                            ->setDefaultValue(self::DEFAULT_LINK_SIZE))
                    ->addArgument(
                        "page",
                        $this->argument("Int", "Which page to show")
                            ->setDefaultValue(1))
                    ->setResolver("MageQL_Catalog_Model_Product::resolveRelatedProducts"),
                "upSellProducts" => $this->field("PaginatedProducts!", "Up-sell products")
                    ->addArgument(
                        "pageSize",
                        $this->argument("Int", "Maximum number of products to list")
                            ->setDefaultValue(self::DEFAULT_LINK_SIZE))
                    ->addArgument(
                        "page",
                        $this->argument("Int", "Which page to show")
                            ->setDefaultValue(1))
                    ->setResolver("MageQL_Catalog_Model_Product::resolveUpSellProducts"),
                // TODO: Fields
                // group prices
                // tier price
                // special price
                // available
                // stock
                // options (bundle)
                // custom options
            ];

        case "ProductDetailAttributes":
            return $this->config->createFields(
                $this->config->getSystemAttributes(MageQL_Catalog_Model_Attributes_Abstract::AREA_DETAIL)
            );

        case "ProductDetailBundle":
            // TODO: Specifc fields which are bundle only?
            return array_merge($registry->getFieldBuilders("ProductDetail"), [
                "bundleOptions" => $this->field("[BundleOption!]!", "Options for a bundle product")
                    ->setResolver("MageQL_Catalog_Model_Product_Bundle::resolveOptions"),
            ]);

        case "ProductDetailConfigurable":
            // TODO: Specifc fields which are configurable only?
            return array_merge($registry->getFieldBuilders("ProductDetail"), [
                "configOptions" => $this->field("ConfigurationOptions!", "Options for a configurable product")
                    ->setResolver("MageQL\\forwardResolver"),
            ]);

        case "ProductDetailSimple":
            // TODO: Specifc fields which are simple only?
            return $registry->getFieldBuilders("ProductDetail");

        case "ProductDetailVirtual":
            // TODO: Specifc fields which are virtual only?
            return $registry->getFieldBuilders("ProductDetail");

        case "ProductOption":
            // We cannot copy from ListProduct right away since we want to avoid some of the standard fields
            return [
                "sku" => $this->field("String!", "SKU"),
                "type" => $this->field("ProductType!", "Product type")
                    ->setResolver("MageQL_Catalog_Model_Product::resolveType"),
                "name" => $this->field("String!", "Product name"),
                "attributes" => $this->field("ListProductAttributes!", "Product attributes for list")
                    ->setResolver("MageQL\\forwardResolver"),
                "attributeSetName" => $this->field("String!", "Attribute set name")
                    ->setResolver(function(Mage_Catalog_Model_Product $src) {
                        return $this->config->getSetById($src->getAttributeSetId())["name"];
                    }),
            ];

        case "ProductOptionSimple":
            return $registry->getFieldBuilders("ProductOption");

        case "ProductOptionVirtual":
            return $registry->getFieldBuilders("ProductOption");

        case "ProductPrice":
            return $registry->getFieldBuilders("Price");

        case "ProductsBy":
            return $this->createProductsByFilters();

        case "Query":
            return [
                "bestsellingProducts" => $this->field("PaginatedProducts!", "All purchased products ordered by number of ordered products, from all users in the store")
                    ->addArgument(
                        "attributeFilter",
                        $this->argument("[ProductAttributeFilter!]", "Filter products by specified attribute and value(s)"))
                    ->addArgument(
                        "priceFilter",
                        $this->argument("ProductPriceFilter", "Filter products by min/max price"))
                    ->addArgument(
                        "pageSize",
                        $this->argument("Int", "Maximum number of products to list")
                            ->setDefaultValue(self::DEFAULT_PAGE_SIZE))
                    ->addArgument(
                        "page",
                        $this->argument("Int", "Which page to show")
                            ->setDefaultValue(1))
                    ->setResolver("MageQL_Catalog_Model_Product::resolveBestsellingProducts"),
                "productAttributes" => $this->field("ProductAttributes!", "Product attributes and thier data")
                    ->setResolver("MageQL\\forwardResolver"),
                "productBySku" => $this->field("ProductDetail", "Detailed product information about a specific SKU")
                    ->addArgument("sku", $this->argument("String!", "Product SKU"))
                    ->setResolver("MageQL_Catalog_Model_Product::resolveBySku"),
                "productsBy" => $this->field("ProductsBy!", "Filter products by a specific attribute")
                    ->setResolver("MageQL\\forwardResolver")
                    ->setDeprecated("Use producs query with attributeFilter instead"),
                "products" => $this->field("PaginatedProducts!", "All available products")
                    ->addArgument(
                        "attributeFilter",
                        $this->argument("[ProductAttributeFilter!]", "Filter products by specified attribute and value(s)"))
                    ->addArgument(
                        "priceFilter",
                        $this->argument("ProductPriceFilter", "Filter products by min/max price"))
                    ->addArgument(
                        "pageSize",
                        $this->argument("Int", "Maximum number of products to list")
                            ->setDefaultValue(self::DEFAULT_PAGE_SIZE))
                    ->addArgument(
                        "page",
                        $this->argument("Int", "Which page to show")
                            ->setDefaultValue(1))
                    ->setResolver("MageQL_Catalog_Model_Product::resolveProducts"),
                "productsBySearch" => $this->field("PaginatedProducts", "Filter products by a specified search term, null means the term is too short")
                    ->addArgument("term", $this->argument("String!", "Search term/phrase"))
                    ->addArgument(
                        "attributeFilter",
                        $this->argument("[ProductAttributeFilter!]", "Filter products by specified attribute and value(s)"))
                    ->addArgument(
                        "priceFilter",
                        $this->argument("ProductPriceFilter", "Filter products by min/max price"))
                    ->addArgument(
                        "pageSize",
                        $this->argument("Int", "Maximum number of products to list")
                            ->setDefaultValue(self::DEFAULT_PAGE_SIZE))
                    ->addArgument(
                        "page",
                        $this->argument("Int", "Which page to show")
                            ->setDefaultValue(1))
                    ->setResolver("MageQL_Catalog_Model_Product::resolveProductsBySearch"),
                "recentlyViewedProducts" => $this->field("PaginatedProducts", "Get list of recently viewed products.")
                    ->addArgument(
                        "attributeFilter",
                        $this->argument("[ProductAttributeFilter!]", "Filter products by specified attribute and value(s)"))
                    ->addArgument(
                        "priceFilter",
                        $this->argument("ProductPriceFilter", "Filter products by min/max price"))
                    ->addArgument(
                        "pageSize",
                        $this->argument("Int", "Maximum number of products to list")
                            ->setDefaultValue(self::DEFAULT_PAGE_SIZE))
                    ->addArgument(
                        "page",
                        $this->argument("Int", "Which page to show")
                            ->setDefaultValue(1))
                    ->setResolver("MageQL_Catalog_Model_Product::resolveRecentlyViewedProducts"),
            ];

        case "RouteProduct":
            return [
                "type" => $this->field("RouteType!", "Type of route")
                    ->setResolver(function() {
                        return "product";
                    }),
                // In the case of a category
                "category" => $this->field("Category", "The parent category of the product, if any. This will depend on the route")
                    ->setResolver("MageQL_Catalog_Model_Category::resolveByRoute"),
                "product" => $this->field("ProductDetail!", "The product")
                    ->setResolver("MageQL_Catalog_Model_Product::resolveByRoute"),
            ];

        case "ProductAttributeFilter":
            return [
                "code" => $this->inputField("ID!", "Product attribute code id"),
                "value" => $this->inputField("String", "Product attribute value for bucket attribute"),
                "minValue" => $this->inputField("Float", "Minimum attribute value for range attribute, leave out to allow minimum value"),
                "maxValue" => $this->inputField("Float", "Maximum attribute value for range attribute, leave out to allow maximum value")
            ];

        case "ProductPriceFilter":
            return [
                "minValue" => $this->inputField("Float", "Minimum attribute value for range attribute, leave out to allow minimum value"),
                "maxValue" => $this->inputField("Float", "Maximum attribute value for range attribute, leave out to allow maximum value"),
                "incVat" => $this->inputField("Boolean", "Add vat before comparing minValue and/or maxValue on price")
                    ->setDefaultValue(false),
            ];
        }

        if(strpos($typeName, "ProductDetail") === 0) {
            return $this->createProductAttributeFieldBuilder(
                MageQL_Catalog_Model_Attributes_Abstract::AREA_DETAIL,
                substr($typeName, strlen("ProductDetail"))
            );
        }

        if(strpos($typeName, "ListProduct") === 0) {
            return $this->createProductAttributeFieldBuilder(
                MageQL_Catalog_Model_Attributes_Abstract::AREA_LIST,
                substr($typeName, strlen("ListProduct"))
            );
        }

        return [];
    }

    /**
     * @param Area $area
     */
    public function areaToTypeName(string $area): string {
        switch($area) {
        case MageQL_Catalog_Model_Attributes_Abstract::AREA_LIST:
            return "ListProduct";
        case MageQL_Catalog_Model_Attributes_Abstract::AREA_DETAIL:
            return "ProductDetail";
        }
    }

    /**
     * @param Area $area
     */
    public function resolveAttributeType(Mage_Catalog_Model_Product $src, string $area): string {
        $setType = $this->config->getSetById((int)$src->getAttributeSetId())["type"];
        $typeId = ucfirst($src->getTypeId());
        $prefix = $this->areaToTypeName($area);

        return $prefix.$typeId."Attributes".$setType;
    }

    /**
     * @param Area $area
     * @return InterfaceBuilder|ObjectBuilder|null
     */
    public function createProductTypeBuilder(string $area, string $typeRemainder) {
        $parentType = $this->areaToTypeName($area);

        if(strpos($typeRemainder, "Attributes") === false) {
            // It must be a concrete product type for a type id
            if( ! in_array(lcfirst($typeRemainder), MageQL_Catalog_Model_Attributes_Product::SUPPORTED_TYPE_IDS)) {
                return null;
            }

            $description = "A ".
                ($area === MageQL_Catalog_Model_Attributes_Product::AREA_LIST ? "partially" : "fully").
                " populated ".lcfirst($typeRemainder)." product";

            return $this->object($description)
                ->setResolveField("MageQL\\defaultVarienObjectResolver")
                ->setInterfaces([$parentType]);
        }

        [$typeId, $setType] = explode("Attributes", $typeRemainder, 2);

        $typeResolver = function(Mage_Catalog_Model_Product $src) use($area): string {
            return $this->resolveAttributeType($src, $area);
        };

        if(empty($typeId) && empty($setType)) {
            // Base interface
            $description = sprintf("Product %s Attribute Interface, for a ".
                ($area === MageQL_Catalog_Model_Attributes_Product::AREA_LIST ? "partial" : "full").
                " product", $area);

            return $this->interface($description)
                ->setTypeResolver($typeResolver);
        }

        if(empty($setType)) {
            // TODO: Make this interface inherit from the base attribute set
            // when interface-inheritance makes it in
            $description = sprintf("Product %s %s Attribute Interface", $area, $typeId);

            return $this->interface($description)
                ->setTypeResolver($typeResolver);
        }

        $setName = $this->config->getSetByType($setType)["name"];

        if(empty($typeId)) {
            // TODO: Make this interface inherit from the base attribute set
            // when interface-inheritance makes it in
            $desc = sprintf("Product %s Attribute Set %s Interface", $area, $setName);

            return $this->interface($desc)
                ->setTypeResolver($typeResolver);
        }

        // All the interfaces
        $typeDesc = sprintf("Product %s %s Attribute Set %s", $area, $typeId, $setName);
        $baseAttributeInterface = $parentType."Attributes";
        $setAttributeInterface = $parentType."Attributes".$setType;
        $productTypeInterface = $parentType.$typeId."Attributes";

        return $this->object($typeDesc)
            ->setResolveField("MageQL\\defaultVarienObjectResolver")
            ->setInterfaces([$baseAttributeInterface, $productTypeInterface, $setAttributeInterface]);
    }

    /**
     * @param Area $area
     * @return Array<FieldBuilder>
     */
    public function createProductAttributeFieldBuilder(string $area, string $typeRemainder) {
        if(strpos($typeRemainder, "Attributes") === false) {
            throw new Exception($area.$typeRemainder." does not have any defined fields");
        }

        [$typeId, $setType] = explode("Attributes", $typeRemainder, 2);

        $typeId = strtolower($typeId);

        // TODO: Make this interface inherit from the base attribute set, aka
        // call parent type to obtain fields
        $attrs = $setType ?
            $this->config->getSetAttributes($area, $setType, $typeId) :
            $this->config->getSystemAttributes($area, $typeId);

        if(Mage::getIsDeveloperMode()) {
            // For now, just verify that we are actually getting the proper stuff
            // GraphQL does not verify this properly, make sure we do it
            $this->verifySuperset($attrs, $area, $setType, $typeId);
        }

        return $this->config->createFields($attrs);
    }

    /**
     * Validates that the given fields are a superset of the base product.
     *
     * @param Area $area
     */
    public function verifySuperset(array $attrs, string $area, ?string $setType = null, ?string $typeId = null): void {
        if($typeId && $setType) {
            $this->checkSuperset(
                $this->config->getSetAttributes($area, $setType),
                $attrs,
                $this->areaToTypeName($area).$typeId."Attributes".$setType." from ".
                $this->areaToTypeName($area)."Attributes".$setType

            );

            $this->checkSuperset(
                $this->config->getSystemAttributes($area, $typeId),
                $attrs,
                $this->areaToTypeName($area).$typeId."Attributes".$setType." from ".
                $this->areaToTypeName($area).$typeId."Attributes"
            );
        }

        if($typeId || $setType) {
            $this->checkSuperset(
                $this->config->getSystemAttributes($area),
                $attrs,
                $this->areaToTypeName($area).($typeId ?: "")."Attributes".($setType ?: "")." from ".
                $this->areaToTypeName($area)."Attributes"
            );
        }
    }

    protected function checkSuperset(array $parent, array $child, string $msg): void {
        $diff = array_keys(array_diff_key($parent, $child));

        if(count($diff) > 0) {
            throw new Exception(sprintf(
                "Missing attributes [%s] in %s", implode(", ", $diff), $msg
            ));
        }
    }

    /**
     * @return Array<FieldBuilder>
     */
    public function createProductsByFilters(): array {
        $filterable = $this->config->getFilterableAttributes();
        $fields = [];

        foreach($filterable as $key => $attr) {
            $field = $this->field("PaginatedProducts", "List of products filtered by $key")
                    ->addArgument(
                        "pageSize",
                        $this->argument("Int", "Maximum number of products to list")
                            ->setDefaultValue(self::DEFAULT_PAGE_SIZE))
                    ->addArgument(
                        "page",
                        $this->argument("Int", "Which page to show")
                            ->setDefaultValue(1));

            switch($attr["input"]) {
            case "datetime":
                $field->addArgument("from", $this->argument("String", "The start-date, inclusive"));
                $field->addArgument("to", $this->argument("String", "The end-date, inclusive"));

                break;

            case "select":
            case "multiselect":
                $field->addArgument("value", $this->argument("String!", "The value to match against"));

                break;
            default:
                switch($attr["backend_type"]) {
                case "int":
                    $field->addArgument("min", $this->argument("Int", "The minimum value"));
                    $field->addArgument("max", $this->argument("Int", "The maximum value"));

                    break;

                case "decimal":
                    $field->addArgument("min", $this->argument("Float", "The minimum value"));
                    $field->addArgument("max", $this->argument("Float", "The maximum value"));

                    break;

                default:
                    Mage::log(sprintf("%s: Cannot create ProductsByFilter for attribute code %s", __METHOD__, $key));

                    continue 3;
                }
            }

            $field->setResolver("MageQL_Catalog_Model_Product::resolveProductsByFilter");

            $fields[$key] = $field;
        }

        return $fields;
    }

    /**
     * @return Array<FieldBuilder>
     */
    public function createProductAttributesFields(): array {
        $filterable = $this->config->getFilterableAttributes();
        $fields = [];

        foreach($filterable as $key => $attr) {
            // TODO: Is this the proper condition?
            if( ! in_array($attr["input"], ["select", "multiselect"])) {
                continue;
            }

            $field = $this->field("ProductAttribute!", "Attribute data for $key");

            $field->setResolver("MageQL_Catalog_Model_Attributes_Product::resolveAttribute");

            $fields[$key] = $field;
        }

        return $fields;
    }

    public function getUnreachableTypes(): array {
        $objectTypes = [
            "ListProduct",
            "ProductDetail"
        ];
        $typeNames = [
            "ProductOptionSimple",
            "ProductOptionVirtual",
            "RouteProduct",
        ];
        $setTypes = Mage::getSingleton("mageql_catalog/attributes_product")->getSetTypes();

        foreach($objectTypes as $objType) {
            // TODO: The empty one should be an interface
            foreach(array_merge(MageQL_Catalog_Model_Attributes_Product::SUPPORTED_TYPE_IDS, [""]) as $typeId) {
                $productType = ucfirst($typeId);

                foreach($setTypes as $set) {
                    $typeNames[] = $objType.$productType."Attributes".$set;
                }
            }
        }

        foreach($objectTypes as $objType) {
            foreach(MageQL_Catalog_Model_Attributes_Product::SUPPORTED_TYPE_IDS as $prodType) {
                $typeNames[] = $objType.ucfirst($prodType);
            }
        }

        return $typeNames;
    }
}
