<?php

declare(strict_types=1);

use GraphQL\Deferred;
use GraphQL\Type\Definition\ResolveInfo;

use function MageQL\snakeCaseToCamel;

/**
 * @psalm-type ConfigurationOptionItem array{0:Mage_Catalog_Model_Product, 1:Mage_Catalog_Model_Resource_Product_Type_Configurable_Attribute_Collection}
 * @psalm-type ConfigurationOptionItemValue array{attribute:string, value:string}
 * @psalm-type ProductAttributeFilter array{ code:string, value?:?string, minValue?:?float, maxValue?:?float }
 * @psalm-type ProductPriceFilter array{ minValue?:?float, maxValue?:?float, incVat:bool }
 */
class MageQL_Catalog_Model_Product extends Mage_Core_Model_Abstract {
    /**
     * List of fields containing key-maps of attributes based on their attribute names.
     *
     * @var Array<string>
     */
    public static $LIST_ATTRIBUTE_FIELDS = ["attributes"];

    /**
     * Type resolver for ProductDetail.
     */
    public static function typeResolverDetail(Mage_Catalog_Model_Product $product): string {
        return "ProductDetail".ucfirst($product->getTypeId());
    }

    /**
     * Type resolver for ListProduct.
     */
    public static function typeResolverList(Mage_Catalog_Model_Product $product): string {
        return "ListProduct".ucfirst($product->getTypeId());
    }

    /**
     * Type resolver for ProductOption.
     */
    public static function typeResolverOption(Mage_Catalog_Model_Product $product): string {
        switch($product->getTypeId()) {
        case "simple":
            return "ProductOptionSimple";

        case "virtual":
            return "ProductOptionVirtual";

        default:
            throw new RuntimeException(sprintf(
                "%s: Invalid type id for ProductOption product: ''.",
                __METHOD__,
                $product->getTypeId()
            ));
        }
    }

    public static function resolveType(Mage_Catalog_Model_Product $product): string {
        return $product->getTypeId();
    }

    /**
     * @param mixed $unusedSrc
     * @param array{sku:string} $args
     */
    public static function resolveBySku(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx
    ): ?Mage_Catalog_Model_Product {
        $product = Mage::getModel("catalog/product");
        $id = $product->getIdBySku($args["sku"]);

        if($id) {
            $product->load($id);
        }

        if( ! Mage::helper("mageql_catalog")->isProductVisible($product, $ctx->getStore())) {
            return null;
        }

        // Magento event to log product visit
        Mage::dispatchEvent("catalog_controller_product_view", ["product" => $product]);

        return $product;
    }

    /**
     * Obtains the category already populated by loadRouteCategory.
     *
     * @see MageQL_Core_Model_Route
     */
    public static function resolveByRoute(
        Mage_Core_Model_Url_Rewrite $rewrite
    ): ?Mage_Catalog_Model_Product {
        // Set in loadRouteProduct
        return $rewrite->getData("product");
    }

    /**
     * @return Array<Mage_Catalog_Model_Product>
     */
    public static function resolvePaginatedItems(
        Mage_Catalog_Model_Resource_Product_Collection $collection,
        array $unusedArgs,
        MageQL_Core_Model_Context $unusedCtx,
        ResolveInfo $info
    ): array {
        return self::applyAttributeSelects($collection, $info, [])->getItems();
    }

    public static function resolveConfigurationOptionAttributes(
        Mage_Catalog_Model_Product $src
    ): Mage_Catalog_Model_Resource_Product_Type_Configurable_Attribute_Collection {
        /**
         * @var Mage_Catalog_Model_Product_Type_Configurable
         */
        $instance = $src->getTypeInstance(true);

        // This will be cached on the product
        return $instance->getConfigurableAttributes($src);
    }

    /**
     * @return Array<ConfigurationOptionItem>
     */
    public static function resolveConfigurationOptionItems(
        Mage_Catalog_Model_Product $product,
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx,
        ResolveInfo $info
    ): array {
        /**
         * @var Mage_Catalog_Model_Product_Type_Configurable
         */
        $instance = $product->getTypeInstance(true);
        $prodAttrs = $instance->getConfigurableAttributes($product);

        // Local implementation of Mage_Catalog_Model_Product_Type_Configurable::getUsedProducts
        $collection = $instance->getUsedProductCollection($product);

        $collection->addFilterByRequiredOptions();
        $collection->addStoreFilter($ctx->getStore());
        $collection->addMinimalPrice();
        $collection->addUrlRewrite();
        // TODO: Can we sort for this?
        $collection->addAttributeToSort("position", "asc");

        // No limit

        // Add the attributes we use for the configurating
        foreach($prodAttrs as $attr) {
            $collection->addAttributeToSelect($attr->getProductAttribute()->getAttributeCode());
        }

        // We use all attribute sets since we cannot be certain about which
        // sets we will get, even though they should be the same as parent
        $collection = self::applyAttributeSelects($collection, $info, ["product"]);

        return array_map(function(Mage_Catalog_Model_Product $item) use($prodAttrs) {
            return [$item, $prodAttrs];
        }, $collection->getItems());
    }

    /**
     * @param ConfigurationOptionItem $src
     * @return Array<array{attribute:string, value:string}>
     */
    public static function resolveConfigurationOptionItemValues($src) {
        $items = [];
        $prod = $src[0];
        $attrs = $src[1];

        foreach($attrs as $attr) {
            $prodAttr = $attr->getProductAttribute();
            $model = $prodAttr->getSource();

            $items[] = [
                "attribute" => snakeCaseToCamel($prodAttr->getAttributeCode()),
                "value" => (string)$model->getOptionText($prod->getData($prodAttr->getAttributeCode())),
            ];
        }

        return $items;
    }

    /**
     * @param array{page:int, pageSize:int} $args
     */
    public static function resolveRelatedProducts(
        Mage_Catalog_Model_Product $src,
        array $args,
        MageQL_Core_Model_Context $ctx
    ): Mage_Catalog_Model_Resource_Product_Link_Product_Collection {
        return self::prepareProductLinkCollection(
            $src->getRelatedProductCollection(),
            $args,
            $ctx
        );
    }

    /**
     * @param array{page:int, pageSize:int} $args
     */
    public static function resolveUpSellProducts(
        Mage_Catalog_Model_Product $src,
        array $args,
        MageQL_Core_Model_Context $ctx
    ): Mage_Catalog_Model_Resource_Product_Link_Product_Collection {
        return self::prepareProductLinkCollection(
            $src->getUpSellProductCollection(),
            $args,
            $ctx
        );
    }

    /**
     * @param array{page:int, pageSize:int} $args
     */
    public static function resolveCrossSellProducts(
        Mage_Catalog_Model_Product $src,
        array $args,
        MageQL_Core_Model_Context $ctx
    ): Mage_Catalog_Model_Resource_Product_Link_Product_Collection {
        return self::prepareProductLinkCollection(
            $src->getCrossSellProductCollection(),
            $args,
            $ctx
        );
    }

    /**
     * @param mixed $unusedSrc
     * @param array{
     *   page:int,
     *   pageSize:int,
     *   priceFilter?:?ProductPriceFilter,
     *   attributeFilter?:?Array<ProductAttributeFilter>
     * } $args
     */
    public static function resolveBestsellingProducts(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx
    ): Mage_Catalog_Model_Resource_Product_Collection {
        $storeId = (int)$ctx->getStore()->getId();
        $collection = Mage::getModel("catalog/product")->getCollection();
        $fromDate = date("Y-m-01");

        $collection->getSelect()
            ->join(
                ["aggregation" => $collection->getResource()->getTable("sales/bestsellers_aggregated_monthly")],
                sprintf("e.entity_id = aggregation.product_id AND aggregation.store_id = %d AND aggregation.period = '%s'", $storeId, $fromDate),
                ["SUM(aggregation.qty_ordered) AS sold_quantity"]
            )
            ->group("e.entity_id")
            ->order(array("sold_quantity DESC", "e.created_at"));

        if( ! empty($args["attributeFilter"])) {
            $collection = self::prepareProductAttributeFilters($collection, $args["attributeFilter"]);
        }

        if( ! empty($args["priceFilter"])) {
            $collection = self::prepareProductPriceFilters($collection, $args["priceFilter"]);
        }

        $collection = self::preparePaginatedProductCollection($collection, $args, $ctx);

        return $collection;
    }

    public static function resolveCategories(
        Mage_Catalog_Model_Product $src,
        array $unusedArgs,
        MageQL_Core_Model_Context $unusedCtx,
        ResolveInfo $info
    ): Deferred {
        $catAttrs = Mage::getSingleton("mageql_catalog/attributes_category");
        $toSelect = $catAttrs->getUsedAttributes(
            $catAttrs->getAttributesByArea(MageQL_Catalog_Model_Attributes_Abstract::AREA_ANY),
            $info->getFieldSelection(1)
        );

        $categories = MageQL_Catalog_Model_Product_Categories::instance();

        $categories->add($src->getId(), $toSelect);

        return new Deferred(function() use($categories, $src) {
            return $categories->load()->get($src->getId());
        });
    }

    /**
     * @param mixed $unusedSrc
     * @param array{page:int, pageSize:int, value:string}|array{page:int, pageSize:int, to:string, from:string} $args
     */
    public static function resolveProductsByFilter(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx,
        ResolveInfo $info
    ): Mage_Catalog_Model_Resource_Product_Collection {
        $attributeConfig = Mage::getSingleton("mageql_catalog/attributes_product");
        $attrs = $attributeConfig->getAttributes();

        if( ! array_key_exists($info->fieldName, $attrs)) {
            throw new Exception(sprintf(
                "%s: Unknown attribute field %s.",
                __METHOD__,
                $info->fieldName
            ));
        }

        $attr = $attrs[$info->fieldName];
        $collection = Mage::getModel("catalog/product")->getCollection();
        $attribute = Mage::getModel("catalog/product")->getResource()->getAttribute($attr["code"]);

        if( ! $attribute) {
            throw new Exception(sprintf(
                "%s: Product attribute %s could not be found",
                __METHOD__,
                $attr["code"]
            ));
        }

        if(array_key_exists("value", $args)) {
            $value = $attribute->usesSource() ?
                $attribute->getSource()->getOptionId($args["value"]) :
                $args["value"];

            $collection->addAttributeToFilter($attr["code"], $value);
        }
        else if($attr["input"] === "datetime") {
            if( ! empty($attr["from"])) {
                $collection->addAttributeToFilter([
                    "attribute" => $attr["code"],
                    "gteq" => date("Y-m-d H:i:s", strtotime($attr["from"])),
                ]);
            }

            if( ! empty($attr["to"])) {
                $collection->addAttributeToFilter([
                    "attribute" => $attr["code"],
                    "lteq" => date("Y-m-d H:i:s", strtotime($attr["to"])),
                ]);
            }
        }
        else {
            if( ! empty($attr["from"])) {
                $collection->addAttributeToFilter([
                    "attribute" => $attr["code"],
                    "gteq" => (float)$attr["from"],
                ]);
            }

            if( ! empty($attr["to"])) {
                $collection->addAttributeToFilter([
                    "attribute" => $attr["code"],
                    "lteq" => (float)$attr["to"],
                ]);
            }
        }

        // Just plain URLs here since we do not have a category
        $collection->addUrlRewrite();
        $collection->addAttributeToSort("name", Varien_Data_Collection::SORT_ORDER_ASC);

        return self::preparePaginatedProductCollection($collection, $args, $ctx);
    }

    /**
     * @param mixed $unusedSrc
     * @param array{
     *   term:string,
     *   page:int,
     *   pageSize:int,
     *   priceFilter?:?ProductPriceFilter,
     *   attributeFilter?:?Array<ProductAttributeFilter>
     * } $args
     */
    public static function resolveProductsBySearch(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx
    ): ?Mage_Catalog_Model_Resource_Product_Collection {
        $query = Mage::getModel("catalogsearch/query");
        $fulltext = Mage::getModel("catalogsearch/fulltext");
        $resource = $fulltext->getResource();
        $minLength = $ctx->getStore()->getConfig(Mage_CatalogSearch_Model_Query::XML_PATH_MIN_QUERY_LENGTH);

        if(strlen($args["term"]) < $minLength) {
            return null;
        }

        $query->setStoreId($ctx->getStore()->getId());
        $query->loadByQuery($args["term"]);

        if( ! $query->getId()) {
            $query->setQueryText($args["term"]);
            $query->setStoreId($ctx->getStore()->getId());
            $query->setPopularity(1);
        }
        else {
            $query->setPopularity($query->getPopularity() + 1);
        }

        $query->prepare();

        $resource->prepareResult($fulltext, $args["term"], $query);

        $searchHelper = Mage::getResourceHelper("catalogsearch");
        $collection = Mage::getModel("catalog/product")->getCollection();
        $foundData = $resource->getFoundData();

        ksort($foundData);
        natsort($foundData);

        $foundIds = array_keys($foundData);

        if(empty($foundIds)) {
            // We do not have any found ids, make sure the magento collection is empty.
            $foundIds = [-1];
        }

        $collection->addIdFilter($foundIds);
        $collection->addUrlRewrite();

        // TODO: Will this make it too slow?
        // Sort by relevance
        $order = new Zend_Db_Expr(
            $searchHelper->getFieldOrderExpression("e.entity_id", $foundIds).
            " ASC, e.entity_id ASC"
        );
        $select = $collection->getSelect();
        $select->order($order);

        if( ! empty($args["attributeFilter"])) {
            $collection = self::prepareProductAttributeFilters($collection, $args["attributeFilter"]);
        }

        if( ! empty($args["priceFilter"])) {
            $collection = self::prepareProductPriceFilters($collection, $args["priceFilter"]);
        }

        $collection = self::preparePaginatedProductCollection($collection, $args, $ctx);

        $query->setNumResults($collection->getSize());
        $query->save();

        return $collection;
    }

    /**
     * @param mixed $unusedSrc
     * @param array{
     *   page:int,
     *   pageSize:int,
     *   priceFilter?:?ProductPriceFilter,
     *   attributeFilter?:?Array<ProductAttributeFilter>
     * } $args
     */
    public static function resolveRecentlyViewedProducts(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx
    ): ?Mage_Catalog_Model_Resource_Product_Collection {

        if( ! Mage::getSingleton('customer/session')->isLoggedIn()) {
            return null;
        }

        $customerId = Mage::getSingleton('customer/session')->getCustomer()->getId();

        if(empty($customerId)) {
            return null;
        }

        $collection = Mage::getModel('catalog/product')->getCollection()
            ->addUrlRewrite();

        $collection->getSelect()
                ->join(
                    [ "rv" => "report_viewed_product_index" ],
                    "rv.product_id = e.entity_id",
                     ["rv.added_at as added_at"])
                ->where(sprintf("rv.customer_id = %d AND rv.store_id = %d", $customerId, $ctx->getStore()->getId()))
                ->order("rv.added_at " . Varien_Data_Collection::SORT_ORDER_DESC);

        if( ! empty($args["attributeFilter"])) {
            $collection = self::prepareProductAttributeFilters($collection, $args["attributeFilter"]);
        }

        if( ! empty($args["priceFilter"])) {
            $collection = self::prepareProductPriceFilters($collection, $args["priceFilter"]);
        }

        $collection = self::preparePaginatedProductCollection($collection, $args, $ctx);

        return $collection;
    }

    /**
     * Prepares a product-link collection like related, upsell and crossell for
     * use in a PaginatedProducts type.
     *
     * @param array{page:int, pageSize:int} $args
     */
    public static function prepareProductLinkCollection(
        Mage_Catalog_Model_Resource_Product_Link_Product_Collection $collection,
        array $args,
        MageQL_Core_Model_Context $ctx
    ): Mage_Catalog_Model_Resource_Product_Link_Product_Collection {
        // Just plain URLs here since we do not have a category
        $collection->addUrlRewrite();
        $collection->setPositionOrder();

        return self::preparePaginatedProductCollection($collection, $args, $ctx);
    }

    /**
     * Prepares a basic product collection for use in a PaginatedProducts type,
     * does not add sort, url-rewrite, or any kind of filter to the collection
     * besides store and visibility.
     *
     * @template T of Mage_Catalog_Model_Resource_Product_Collection
     * @param T $collection
     * @param array{page:int, pageSize:int} $args
     * @return T
     */
    public static function preparePaginatedProductCollection(
        Mage_Catalog_Model_Resource_Product_Collection $collection,
        array $args,
        MageQL_Core_Model_Context $ctx
    ): Mage_Catalog_Model_Resource_Product_Collection  {
        $visibility = ["in" => [
            Mage_Catalog_Model_Product_Visibility::VISIBILITY_IN_CATALOG,
            Mage_Catalog_Model_Product_Visibility::VISIBILITY_BOTH
        ]];

        // We do not yet select any attributes, that is done when we
        // fetch the data in the items resolver
        $collection->addAttributeToFilter("visibility", $visibility);
        $collection->addStoreFilter($ctx->getStore());
        $collection->addMinimalPrice();

        // We call this directly on the select to prevent magento from stopping us from
        // going off the end of the list. (setCurPage() prevents this, which is not correct
        // from an API PoV)
        $collection->getSelect()
                   ->limitPage(max($args["page"], 1), $args["pageSize"]);

        return $collection;
    }

    /**
     * Prepares attribute filter for product collection.
     *
     * @template T of Mage_Catalog_Model_Resource_Product_Collection
     * @param T $collection
     * @param Array<ProductAttributeFilter> $filters
     * @return T
     */
    public static function prepareProductAttributeFilters (
        Mage_Catalog_Model_Resource_Product_Collection $collection,
        array $filters
    ): Mage_Catalog_Model_Resource_Product_Collection {
        $config = Mage::getSingleton("eav/config");

        foreach($filters as $filter) {
            // Sanity-check to avoid loading by id:
            if(is_numeric($filter["code"])) {
                throw new RuntimeException("Attribute code was expected in filter code");
            }

            $attr = $config->getAttribute(Mage_Catalog_Model_Product::ENTITY, $filter["code"]);

            if( ! $attr || ! $attr->getIsFilterable()) {
                throw new RuntimeException(sprintf(
                    "Attribute '%s' is not filterable",
                    $filter["code"]
                ));
            }

            // Fetch real attribute code if input was something strange
            $attributeCode = $attr->getAttributeCode();
            $minValue = array_key_exists("minValue", $filter) ? ( $filter["minValue"] !== null ? floatval($filter["minValue"]) : null) : null;
            $maxValue = array_key_exists("maxValue", $filter) ? ( $filter["maxValue"] !== null ? floatval($filter["maxValue"]) : null) : null;

            // Check input type of attribute
            switch($attr->getFrontendInput()) {
                case "select":
                    $attributeCode .= "_value";
                    break;

                case "price":
                    break;

                default:
                    throw new RuntimeException(sprintf(
                        "Attribute '%s' of type '%s' is not filterable",
                        $attr->getAttributeCode(),
                        $attr->getFronendInput()
                    ));
            }

            if(array_key_exists("value", $filter) && $filter["value"] !== null) {
                // Precise value have priority over min and max values, if specified
                $collection->addAttributeToFilter($attributeCode, ["eq" => $filter["value"]]);
            }
            else {
                if($minValue !== null && $minValue > 0) {
                    $collection->addAttributeToFilter($attributeCode, ["gteq" => $minValue]);
                }

                if ($maxValue !== null && $maxValue > 0) {
                    $collection->addAttributeToFilter($attributeCode, ["lteq" => $maxValue]);
                }
            }
        }

        return $collection;
    }

    /**
     * Prepares price filter for product collection.
     *
     * @template T of Mage_Catalog_Model_Resource_Product_Collection
     * @param T $collection
     * @param ProductPriceFilter $filter
     * @return T
     */
    public static function prepareProductPriceFilters (
        Mage_Catalog_Model_Resource_Product_Collection $collection,
        array $filter
    ): Mage_Catalog_Model_Resource_Product_Collection {
        // Minimum and maximum values have already been filtered to be Floats by GraphQL
        $minValue = $filter["minValue"] ?? null;
        $maxValue = $filter["maxValue"] ?? null;

        // Make sure price index table is joined
        $collection->addMinimalPrice();

        $select = $collection->getSelect();

        if($filter["incVat"]) {
            $select->join(["tc" => "tax_class"], "tc.class_id = price_index.tax_class_id", []);
            $select->join(["tcr" => "tax_calculation_rate"], "tcr.code = tc.class_name", []);

            if($minValue !== null) {
                $select->where("min_price * (1 + (tcr.rate / 100)) >= ?", (string)$minValue);
            }

            if($maxValue !== null) {
                $select->where("max_price * (1 + (tcr.rate / 100)) <= ?", (string)$maxValue);
            }
        }
        else {
            if($minValue !== null) {
                $select->where("min_price >= ?", (string)$minValue);
            }

            if($maxValue !== null) {
                $select->where("max_price <= ?", (string)$maxValue);
            }
        }

        return $collection;
    }

    /**
     * @template T of Mage_Catalog_Model_Resource_Product_Collection
     * @param T $collection
     * @param Array<string> $path
     * @return T
     */
    public static function applyAttributeSelects(
        Mage_Catalog_Model_Resource_Product_Collection $collection,
        ResolveInfo $info,
        array $path
    ): Mage_Catalog_Model_Resource_Product_Collection {
        $config = Mage::getSingleton("mageql_catalog/attributes_product");
        // One extra level for attributes
        /**
         * Recursive types do not work properly, use a single level
         *
         * @var array<string, bool | array<string, bool>>
         */
        $fields = $info->getFieldSelection(count($path) + 1);

        foreach($path as $p) {
            /**
             * @var array<string, bool | array<string, bool>>
             */
            $fields = $fields[$p] ?? [];
        }

        // Merge attributes
        foreach(self::$LIST_ATTRIBUTE_FIELDS as $f) {
            if(array_key_exists($f, $fields) && is_array($fields[$f])) {
                $fields = array_merge($fields, $fields[$f]);
            }
        }

        // We use all attribute sets since we cannot be certain about which sets we will get
        $toSelect = $config->getUsedAttributes(
            $config->getAttributesByArea(MageQL_Catalog_Model_Attributes_Abstract::AREA_LIST),
            $fields
        );

        foreach($toSelect as $col) {
            $collection->addAttributeToSelect($col);
        }

        return $collection;
    }

    /**
     * Event observer which loads the specified product for a redirect, if the
     * load fails it will replace the rewrite with null (interpreted as 404).
     *
     * @see MageQL_Core_Model_Route
     */
    public function loadRouteProduct(Varien_Event_Observer $event): void {
        /**
         * @var Mage_Core_Model_Url_Rewrite
         */
        $rewrite = $event->getRewrite();
        /**
         * @var MageQL_Core_Model_Context
         */
        $ctx = $event->getContext();
        $result = $event->getResult();

        if($event->getIsRedirect() || !$rewrite->getProductId()) {
            return;
        }

        $product = Mage::getModel("catalog/product");

        $product->load($rewrite->getProductId());

        if(Mage::helper("mageql_catalog")->isProductVisible($product, $ctx->getStore())) {
            // Save for later in resolveByRoute
            $rewrite->setData("product", $product);

            // Magento event to log product visit
            Mage::dispatchEvent("catalog_controller_product_view", ["product" => $product]);
        }
        else {
            // Not Found
            $result->setRewrite(null);
        }
    }
}
