<?php

declare(strict_types=1);

use GraphQL\Deferred;
use GraphQL\Type\Definition\ResolveInfo;

/**
 * Container for deferred loading of category children, stored in Magento registry.
 */
class MageQL_Catalog_Model_Category_Children {
    public static function registryKey(): string {
        return "mageql_catalog_category_children";
    }

    public static function instance(): self {
        $children = Mage::registry(static::registryKey());

        if( ! $children) {
            $children = new static();

            Mage::register(static::registryKey(), $children);
        }

        return $children;
    }

    /**
     * Clears this deferred container by removing the global instance.
     */
    public static function clear(): void {
        Mage::unregister(static::registryKey());
    }

    private function __construct() {
        // Empty
    }

    /**
     * @var Array<Mage_Catalog_Model_Category>
     */
    protected $data = [];

    /**
     * @var Array<int>
     */
    protected $parentIds = [];

    /**
     * @var Array<string>
     */
    protected $fields = [];

    /**
     * Queues the given category to fetch children with the supplied list of fields.
     *
     * @param int $parentId
     * @param Array<string> $fields
     */
    public function add($parentId, array $fields): self {
        if( ! in_array($parentId, $this->parentIds)) {
            $this->parentIds[] = $parentId;
        }

        $this->fields = array_unique(array_merge($this->fields, $fields));

        return $this;
    }

    /**
     * Loads the children of the currently queued category ids, will skip
     * loading if no category ids have been queued since previous call.
     */
    public function load(): self {
        if(empty($this->parentIds)) {
            // Nothing new to load
            return $this;
        }

        $categories = Mage::getModel("catalog/category")->getCollection();

        foreach($this->fields as $col) {
            $categories->addAttributeToSelect($col);
        }

        $categories->addAttributeToFilter("is_active", "1");
        $categories->addAttributeToFilter("include_in_menu", "1");

        $categories->addAttributeToFilter("parent_id", [ "in" => $this->parentIds ]);

        $categories->addAttributeToSort("position", "asc");
        $categories->addAttributeToSort("name", "asc");

        // Load and reset, merge since we might have loaded more data, ensure
        // that the categories are overwritten if we keep loading new ones since
        // they might have been updated or require different fields
        $this->data = array_merge($this->data, $categories->getItems());
        $this->parentIds = [];
        $this->fields = [];

        return $this;
    }

    /**
     * Fetches all category children belonging to $parent if they have been loaded.
     *
     * @param int $parentId
     * @return Array<Mage_Catalog_Model_Category>
     */
    public function get($parentId): array {
        return array_values(array_filter($this->data, function(Mage_Catalog_Model_Category $category) use($parentId) {
            return $category->getParentId() == $parentId;
        }));
    }
}
