<?php

declare(strict_types=1);

use Spatie\Snapshots\MatchesSnapshots;

use PHPUnit\Framework\TestCase;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;

class ProductTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
    }

    public function onNotSuccessfulTest($e): void {
        MagentoManager::logQueries();

        throw $e;
    }

    public function testMissingParameter() {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku {
                sku
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(400, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testMissingProduct() {
        try {
            // We use the profiler to test the event
            Varien_Profiler::enable();

            $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
                productBySku (sku: "test-non-existing") {
                    sku
                }
            }'));

            $this->assertMatchesJsonSnapshot($res->getBody());
            $this->assertEquals(200, $res->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:catalog_controller_product_view", null));
        }
        catch(Throwable $e) {
            Varien_Profiler::disable();

            throw $e;
        }
    }

    public function testUnassignedProduct() {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-unassigned") {
                sku
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testInvisibleChildProduct() {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-config-child-1") {
                sku
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testSimpleProduct() {
        try {
            // We use the profiler to test the event
            Varien_Profiler::enable();

            $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
                productBySku (sku: "test-simple") {
                    sku
                    type
                    name
                    url
                    attributeSetName
                    attributes {
                        description
                        metaDescription
                        metaKeyword
                        metaTitle
                        minimalPrice
                        msrp
                        msrpEnabled
                        shortDescription
                        smallImage {
                            src
                            smaller: src (width: 50)
                            resized: src (width: 20, height: 20)
                            fill: src (width: 20, height: 20, fill: true)
                            fillSquare: src (width: 20, fill: true)
                        }
                        smallImageLabel
                        thumbnail {
                            src
                        }
                    }
                }
            }'));

            $this->assertMatchesJsonSnapshot($res->getBody());
            $this->assertEquals(200, $res->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
            $this->assertNotFalse(Varien_Profiler::fetch("DISPATCH EVENT:catalog_controller_product_view", null));
        }
        catch(Throwable $e) {
            Varien_Profiler::disable();

            throw $e;
        }
    }

    public function testVirtualProduct() {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-virtual") {
                sku
                type
                name
                url
                attributeSetName
                attributes {
                    description
                    metaDescription
                    metaKeyword
                    metaTitle
                    minimalPrice
                    msrp
                    msrpEnabled
                    shortDescription
                    smallImage {
                        src
                        smaller: src (width: 50)
                        resized: src (width: 20, height: 20)
                        fill: src (width: 20, height: 20, fill: true)
                        fillSquare: src (width: 20, fill: true)
                    }
                    smallImageLabel
                    thumbnail {
                        src
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testConfigurableProduct() {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-config") {
                sku
                type
                name
                url
                attributeSetName
                attributes {
                    description
                    metaDescription
                    metaKeyword
                    metaTitle
                    minimalPrice
                    msrp
                    msrpEnabled
                    shortDescription
                    smallImage {
                        src
                        smaller: src (width: 50)
                        resized: src (width: 20, height: 20)
                        fill: src (width: 20, height: 20, fill: true)
                        fillSquare: src (width: 20, fill: true)
                    }
                    smallImageLabel
                    thumbnail {
                        src
                    }
                }
                ... on ProductDetailConfigurable {
                    configOptions {
                        attributes { attribute label }
                        items {
                            values { attribute value }
                            price { incVat }
                            product {
                                sku
                                name
                                attributes {
                                    smallImage { src }
                                    thumbnail {
                                        src
                                    }
                                    shortDescription
                                    ... on ListProductSimpleAttributesDefault {
                                        color
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testBundleProductDefault() {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-bundle-default") {
                sku
                type
                name
                url
                attributeSetName
                attributes {
                    description
                    metaDescription
                    metaKeyword
                    metaTitle
                    minimalPrice
                    msrp
                    msrpEnabled
                    shortDescription
                    smallImage {
                        src
                        smaller: src (width: 50)
                        resized: src (width: 20, height: 20)
                        fill: src (width: 20, height: 20, fill: true)
                        fillSquare: src (width: 20, fill: true)
                    }
                    smallImageLabel
                    thumbnail {
                        src
                    }
                }
                price { incVat }
                ... on ProductDetailBundle {
                    bundleOptions {
                        required
                        type
                        title
                        selections {
                            isDefault
                            qty
                            isQtyFixed
                            price { incVat }
                            product {
                                sku
                                name
                                attributes {
                                    shortDescription
                                    smallImage { src }
                                }
                            }
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testBundleProductDefaultIds() {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-bundle-default") {
                sku
                ... on ProductDetailBundle {
                    bundleOptions {
                        optionId
                        selections {
                            selectionId
                            product {
                                sku
                            }
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testBundleProductHidden() {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-bundle-hidden") {
                sku
                type
                name
                url
                attributeSetName
                attributes {
                    description
                    metaDescription
                    metaKeyword
                    metaTitle
                    minimalPrice
                    msrp
                    msrpEnabled
                    shortDescription
                    smallImage {
                        src
                        smaller: src (width: 50)
                        resized: src (width: 20, height: 20)
                        fill: src (width: 20, height: 20, fill: true)
                        fillSquare: src (width: 20, fill: true)
                    }
                    smallImageLabel
                    thumbnail {
                        src
                    }
                }
                price { incVat }
                ... on ProductDetailBundle {
                    bundleOptions {
                        required
                        type
                        title
                        selections {
                            isDefault
                            qty
                            isQtyFixed
                            price { incVat }
                            product {
                                sku
                                name
                                attributes {
                                    shortDescription
                                    smallImage { src }
                                }
                            }
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductInterfaceAttributesNotFound() {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-simple") {
                sku
                type
                attributes {
                    ... on ProductDetailSimpleAttributes {
                        color
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(400, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductInterfaceAttributes() {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-simple") {
                sku
                type
                attributes {
                    ... on ProductDetailSimpleAttributes {
                        description
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductMediaGallery() {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-simple") {
                sku
                type
                gallery {
                    label
                    image {
                       src
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductMediaGalleryResize() {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-simple") {
                sku
                type
                gallery {
                    label
                    image {
                        src
                        small: src (width: 200)
                        thumb: src (width: 50, height: 50)
                        odd: src (width: 50, height: 50, fill: true)
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testCrossSell() {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-config") {
                sku
                crossSellProducts {
                    totalCount
                    items {
                        sku
                        attributes {
                            shortDescription
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testUpSell() {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-config") {
                sku
                upSellProducts {
                    totalCount
                    items {
                        sku
                        attributes {
                            shortDescription
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testRelatedProducts() {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-config") {
                sku
                relatedProducts {
                    totalCount
                    items {
                        sku
                        attributes {
                            shortDescription
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testRelatedProductsPaginated() {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-config") {
                sku
                relatedProducts (pageSize: 1, page: 1) {
                    totalCount
                    items {
                        sku
                        attributes {
                            shortDescription
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));

        MagentoManager::reset();

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-config") {
                sku
                relatedProducts (pageSize: 1, page: 2) {
                    totalCount
                    items {
                        sku
                        attributes {
                            shortDescription
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));

        MagentoManager::reset();

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-config") {
                sku
                relatedProducts (pageSize: 1, page: 3) {
                    totalCount
                    items {
                        sku
                        attributes {
                            shortDescription
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductsByManufacturer() {
        $res = MagentoManager::runRequest(new Request("POST /graphql",
            ["Content-Type" => "application/graphql"],
        'query {
            productsBy {
                manufacturer (value: "Manufacturer A") {
                    items {
                        sku
                        attributes {
                            shortDescription
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductAttributeValues() {
        $res = MagentoManager::runRequest(new Request("POST /graphql",
            ["Content-Type" => "application/graphql"],
        'query {
            productAttributes {
                manufacturer {
                    label
                    values
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductsBySearch() {
        $res = MagentoManager::runRequest(new Request("POST /graphql",
            ["Content-Type" => "application/graphql"],
        'query {
            productsBySearch(term: "Test") {
                items {
                    sku
                    attributes {
                        shortDescription
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));

        $query = Mage::getModel("catalogsearch/query");

        $query->loadByQuery("Test");

        $this->assertEquals($query->getNumResults(), 6);
    }

    public function testProductsBySearchTooShort() {
        $res = MagentoManager::runRequest(new Request("POST /graphql",
            ["Content-Type" => "application/graphql"],
        'query {
            productsBySearch(term: "T") {
                items {
                    sku
                    attributes {
                        shortDescription
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }
}
