<?php

declare(strict_types=1);

namespace Fixtures;

use Mage;
use Mage_Catalog_Model_Category;
use Mage_Catalog_Model_Product;
use Mage_Catalog_Model_Product_Status;
use Mage_Catalog_Model_Product_Visibility;
use Mage_Eav_Model_Entity_Setup;
use Crossroads\Magento\Test\Integration\MagentoManager;

/**
 * @psalm-type ProductData array {
 *   sku: string,
 * }
 *
 * @psalm-type BundleOptions Array<array {
 *   title: string,
 *   type: string,
 *   required: int,
 *   position: ?int,
 *   selections: Array<array {
 *     product_id: string,
 *     selection_price_value: float,
 *     selection_price_type: int,
 *     selection_qty: float,
 *     selection_can_change_qty: int,
 *     is_default: int,
 *     position: ?int,
 *   }>
 * }>
 */
class Products {
    public function load(): void {
        MagentoManager::initAdmin();

        $this->createAttributes();

        MagentoManager::reset();
        MagentoManager::initAdmin();

        $this->createSimpleProduct();
        $this->createVirtualProduct();
        $this->createConfigurableProduct();
        $this->createUnassignedProduct();
        $this->createDefaultBundle();
        $this->createHiddenBundle();
        $this->createBundleSelect();

        MagentoManager::reindex();

        MagentoManager::reset();
    }

    /**
     * @param $productData ProductData
     */
    public function loadOrCreate(array $productData): Mage_Catalog_Model_Product {
        $model = Mage::getModel("catalog/product");

        if ($entityId = $model->getIdBySku($productData["sku"])) {
            $model->load($entityId);
        }

        if (empty($productData["attribute_set_id"])) {
            $productData["attribute_set_id"] = $model->getDefaultAttributeSetId();
        }

        if (empty($productData["category_ids"])) {
            $poductData["category_ids"] = [];
        }

        $model->addData($productData);
        $model->save();

        return $model;
    }

    public function assignCategory(
        Mage_Catalog_Model_Product $product,
        Mage_Catalog_Model_Category $category,
        int $position
    ): void {
        $catPositions = $category->getProductsPosition();

        $catPositions[$product->getId()] = $position;

        $category->setPostedProducts($catPositions);
        $category->save();

        $product->setCategoryIds(array_unique(
            array_merge($product->getCategoryIds() ?: [],
            [$category->getId()]
        )));

        $product->save();
    }

    /**
     * @param BundleOptions $bundleData
     */
    public function assignBundleOptions(
        Mage_Catalog_Model_Product $product,
        array $bundleData
    ): void {
        $bundleOptions = [];
        $bundleSelections = [];
        $optionColl = $product->getTypeInstance(true)->getOptionsCollection($product);

        foreach($optionColl as $o) {
            foreach($bundleData as $d) {
                if($o->getDefaultTitle() === $d["title"]) {
                    continue 2;
                }
            }

            $o->delete();
        }

        foreach($bundleData as $k => $d) {
            $option = null;

            foreach($optionColl as $o) {
                if($o->getDefaultTitle() === $d["title"]) {
                    $option = [
                        "title" => $o->getDefaultTitle(),
                        "option_id" => $o->getId(),
                        "type" => $d["type"],
                        "required" => $d["required"],
                        "position" => $d["position"] ?? 0,
                    ];

                    break;
                }
            }

            if( ! $option) {
                $option = $d;
            }

            $option["delete"] = "";
            $option["option_id"] = $option["option_id"] ?? "";

            if($option["option_id"]) {
                $k = $option["option_id"];
            }

            $bundleOptions[$k] = $option;

            $selColl = isset($option["option_id"]) ?
                Mage::getResourceModel("bundle/selection_collection")
                    ->setOptionIdsFilter([$option["option_id"]]) :
                [];

            foreach($selColl as $os) {
                foreach($d["selections"] as $ds) {
                    if($ds["product_id"] === $os->getProductId()) {
                        continue 2;
                    }
                }

                $os->delete();
            }

            foreach($d["selections"] as $ds) {
                $sel = null;

                foreach($selColl as $os) {
                    if($os->getProductId() === $ds["product_id"]) {
                        // She selection collection contains products, not selections
                        $sel = [
                            "selection_id" => $os->getSelectionId(),
                            "product_id" => $os->getProductId(),
                            "selection_price_value" => $ds["selection_price_value"],
                            "selection_price_type" => $ds["selection_price_type"],
                            "selection_qty" => $ds["selection_qty"],
                            "selection_can_change_qty" => $ds["selection_can_change_qty"],
                            "is_default" => $ds["is_default"],
                            "position" => $ds["position"] ?? 0,
                        ];

                        break;
                    }
                }

                if( ! $sel) {
                    $sel = $ds;
                };

                $sel["delete"] = "";

                if( ! array_key_exists($k, $bundleSelections)) {
                    $bundleSelections[$k] = [];
                }

                $bundleSelections[$k][] = $sel;
            }
        }

        // Got to enable flags and register to be able to properly save a bundle product
        $product->setCanSaveCustomOptions(true);
        $product->setCanSaveBundleSelections(true);
        $product->setAffectBundleProductSelections(true);

        Mage::register("product", $product);

        $product->setBundleOptionsData($bundleOptions);
        $product->setBundleSelectionsData($bundleSelections);

        $product->save();

        Mage::unregister("product");
    }

    public function createAttribute(string $code, array $newOptions) {
        $installer = new Mage_Eav_Model_Entity_Setup("core_setup");
        $installer->startSetup();

        $attr = Mage::getModel("eav/entity_attribute")->loadByCode("catalog_product", $code);

        $attr->setIsVisibleOnFront(true);
        $attr->setUsedInProductListing(true);
        $attr->setApplyTo(implode(",", ["bundle", "configurable", "simple", "virtual"]));
        $attr->save();

        $existing = [];
        $options = Mage::getModel("eav/entity_attribute_source_table")
            ->setAttribute($attr)
            ->getAllOptions(false);

        foreach($options as $option) {
            if(in_array($option["label"], $newOptions)) {
                array_push($existing, $option["label"]);
            }
        }

        $insertOptions = array_diff($newOptions, $existing);

        if( ! empty($insertOptions)) {
            $installer->addAttributeOption([
                "attribute_id" => $attr->getId(),
                "values" => $insertOptions,
            ]);
        }

        $attributeSetId = $installer->getAttributeSetId("catalog_product", "Default");

        $installer->addAttributeToSet("catalog_product", $attributeSetId, "General", $code, 10);

        $installer->endSetup();
    }

    public function createAttributes() {
        $this->createAttribute("color", ["Red","Black", "Yellow"]);
        $this->createAttribute("manufacturer", ["Manufacturer A", "Manufacturer B"]);
    }

    public function getAttributeValue($code, $option) {
        $attr = Mage::getSingleton("eav/config")->getAttribute("catalog_product", $code);
        $attrOptions = $attr->getSource()->getAllOptions();

        foreach($attrOptions as $o) {
            if($o["label"] === $option) {
                return $o["value"];
            }
        }

        return null;
    }

    public function createSimpleProduct() {
        $store = Mage::getModel("core/store")->load(MagentoManager::TESTING_STORE);
        $manufacturer = $this->getAttributeValue("manufacturer", "Manufacturer A");

        $model = $this->loadOrCreate([
            "store_id" => $store->getId(),
            "website_ids" => [$store->getWebsiteId()],
            "type_id" => "simple",
            "sku" => "test-simple",
            "name" => "Test for Simple Product",
            "manufacturer" => $manufacturer,
            "status" => Mage_Catalog_Model_Product_Status::STATUS_ENABLED,
            "tax_class" => 1, // Default
            "visibility" => Mage_Catalog_Model_Product_Visibility::VISIBILITY_BOTH,
            "price" => 12.34,
            "meta_title" => "Product meta title",
            "meta_description" => "Product meta description",
            "meta_keyword" => "Product,meta,keywords",
            "description" => "The long product description",
            "short_description" => "The shortest product description",
            "weight" => 3.14,
            "small_image" => "product_small_image.jpg",
            "small_image_label" => "Small Product Image",
            "thumbnail" => "product_thumbnail.jpg",
            "thumbnail_label" => "Thumbnail Product Image",
            "media_gallery" => [
                "images" => [
                    ["file" => "first_gallery.jpg", "position" => 0, "label" => "First Gallery", "disabled" => 0],
                    ["file" => "second_gallery.jpg", "position" => 1, "label" => "Second Gallery", "disabled" => 0],
                    ["file" => "disabled_gallery.jpg", "position" => 2, "label" => "Disabled Gallery", "disabled" => 1],
                ],
            ],
        ]);

        $this->assignCategory(
            $model,
            MagentoManager::loadCategoryByAttribute("varchar", "url_key", "test-category"),
            1
        );

        Mage::getModel("cataloginventory/stock_item")->addData([
            "stock_id" => 1,
            "use_config_manage_stock" => 0,
            "manage_stock" => 1,
            "min_sale_qty" => 1,
            "max_sale_qty" => 2,
            "is_in_stock" => 1,
            "qty" => 999,
        ])->assignProduct($model)->save();
    }

    public function createVirtualProduct() {
        $store = Mage::getModel("core/store")->load(MagentoManager::TESTING_STORE);
        $manufacturer = $this->getAttributeValue("manufacturer", "Manufacturer B");

        $model = $this->loadOrCreate([
            "store_id" => $store->getId(),
            "website_ids" => [$store->getWebsiteId()],
            "type_id" => "virtual",
            "sku" => "test-virtual",
            "name" => "Test for Virtual Product",
            "manufacturer" => $manufacturer,
            "status" => Mage_Catalog_Model_Product_Status::STATUS_ENABLED,
            "tax_class" => 1, // Default
            "visibility" => Mage_Catalog_Model_Product_Visibility::VISIBILITY_BOTH,
            "price" => 9.99,
            "meta_title" => "Product meta title virtual",
            "meta_description" => "Product meta description virtual",
            "meta_keyword" => "Product,meta,keywords,Virtual",
            "description" => "The long product description for virtual",
            "short_description" => "Virtual short",
            "small_image" => "virtual_small.jpg",
            "small_image_label" => "Small Virtual Product Image",
            "thumbnail" => "virtual_thumbnail.jpg",
            "thumbnail_label" => "Thumbnail Virtual Image",
            "media_gallery" => [
                "images" => [
                    ["file" => "virtual_first_gallery.jpg", "position" => 0, "label" => "First Virtual Gallery", "disabled" => 0],
                    ["file" => "virtual_second_gallery.jpg", "position" => 1, "label" => "Second Virtual Gallery", "disabled" => 0],
                    ["file" => "virtual_disabled_gallery.jpg", "position" => 2, "label" => "Disabled Virtual Gallery", "disabled" => 1],
                ],
            ],
        ]);

        $this->assignCategory(
            $model,
            MagentoManager::loadCategoryByAttribute("varchar", "url_key", "test-category"),
            2
        );

        Mage::getModel("cataloginventory/stock_item")->addData([
            "stock_id" => 1,
            "use_config_manage_stock" => 0,
            "manage_stock" => 1,
            "min_sale_qty" => 1,
            "max_sale_qty" => 2,
            "is_in_stock" => 1,
            "qty" => 999,
        ])->assignProduct($model)->save();
    }

    public function createConfigurableProduct() {
        $store = Mage::getModel("core/store")->load(MagentoManager::TESTING_STORE);
        $manufacturer = $this->getAttributeValue("manufacturer", "Manufacturer B");
        $attr = Mage::getSingleton("eav/config")->getAttribute("catalog_product", "color");
        $attrOptions = $attr->getSource()->getAllOptions();
        $attrRed = null;
        $attrBlack = null;

        foreach($attrOptions as $o) {
            switch($o["label"]) {
            case "Red":
                $attrRed = $o["value"];
                break;
            case "Black":
                $attrBlack = $o["value"];
                break;
            }
        }

        $child1 = $this->loadOrCreate([
            "store_id" => $store->getId(),
            "website_ids" => [$store->getWebsiteId()],
            "type_id" => "simple",
            "sku" => "test-config-child-1",
            "name" => "Red Child",
            "manufacturer" => $manufacturer,
            "status" => Mage_Catalog_Model_Product_Status::STATUS_ENABLED,
            "tax_class" => 1, // Default
            "visibility" => Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE,
            "price" => 12.34,
            "color" => $attrRed,
            "meta_title" => "Meta title Child 1 for config product",
            "meta_description" => "Product meta description config child 1",
            "meta_keyword" => "Product,meta,keywords,config,child,1",
            "description" => "The long product description for config child 1",
            "short_description" => "Config short child 1",
            "small_image" => "config_small_child1.jpg",
            "small_image_label" => "Small Config Product Image Child 1",
            "thumbnail" => "config_thumb_child1.jpg",
            "thumbnail_label" => "Thumbnail Thumbnail Image Child 1",
            "media_gallery" => [
                "images" => [
                    ["file" => "config_first_gallery_child1.jpg", "position" => 0, "label" => "First Config Gallery Child 1", "disabled" => 0],
                    ["file" => "config_second_gallery_child1.jpg", "position" => 1, "label" => "Second Config Galler Child 1y", "disabled" => 0],
                    ["file" => "config_disabled_gallery_child1.jpg", "position" => 2, "label" => "Disabled Config Gallery Child 1", "disabled" => 1],
                ],
            ],
        ]);

        Mage::getModel("cataloginventory/stock_item")->addData([
            "stock_id" => 1,
            "use_config_manage_stock" => 0,
            "manage_stock" => 1,
            "min_sale_qty" => 1,
            "max_sale_qty" => 5,
            "is_in_stock" => 1,
            "qty" => 999,
        ])->assignProduct($child1)->save();

        $child2 = $this->loadOrCreate([
            "store_id" => $store->getId(),
            "website_ids" => [$store->getWebsiteId()],
            "type_id" => "simple",
            "sku" => "test-config-child-2",
            "name" => "Black Child",
            "manufacturer" => $manufacturer,
            "status" => Mage_Catalog_Model_Product_Status::STATUS_ENABLED,
            "tax_class" => 1, // Default
            "visibility" => Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE,
            "price" => 22.34,
            "color" => $attrBlack,
            "meta_title" => "Meta title Child 2 for config product",
            "meta_description" => "Product meta description config child 2",
            "meta_keyword" => "Product,meta,keywords,config,child,2",
            "description" => "The long product description for config child 2",
            "short_description" => "Config short child 2",
            "small_image" => "config_small_child2.jpg",
            "small_image_label" => "Small Config Product Image Child 2",
            "thumbnail" => "config_thumb_child2.jpg",
            "thumbnail_label" => "Thumbnail Thumbnail Image Child 2",
            "media_gallery" => [
                "images" => [
                    ["file" => "config_first_gallery_child2.jpg", "position" => 0, "label" => "First Config Gallery Child 2", "disabled" => 0],
                    ["file" => "config_second_gallery_child2.jpg", "position" => 1, "label" => "Second Config Galler Child 2", "disabled" => 0],
                    ["file" => "config_disabled_gallery_child2.jpg", "position" => 2, "label" => "Disabled Config Gallery Child 2", "disabled" => 1],
                ],
            ],
        ]);

        Mage::getModel("cataloginventory/stock_item")->addData([
            "stock_id" => 1,
            "use_config_manage_stock" => 0,
            "manage_stock" => 1,
            "min_sale_qty" => 1,
            "max_sale_qty" => 5,
            "is_in_stock" => 1,
            "qty" => 999,
        ])->assignProduct($child2)->save();

        $model = $this->loadOrCreate([
            "store_id" => $store->getId(),
            "website_ids" => [$store->getWebsiteId()],
            "type_id" => "configurable",
            "sku" => "test-config",
            "name" => "Test for Configurable Product",
            "manufacturer" => $manufacturer,
            "status" => Mage_Catalog_Model_Product_Status::STATUS_ENABLED,
            "tax_class" => 1, // Default
            "visibility" => Mage_Catalog_Model_Product_Visibility::VISIBILITY_BOTH,
            "price" => 12.34,
            "meta_title" => "Product meta title config",
            "meta_description" => "Product meta description config",
            "meta_keyword" => "Product,meta,keywords,config",
            "description" => "The long product description for config",
            "short_description" => "Config short",
            "small_image" => "config_small.jpg",
            "small_image_label" => "Small Config Product Image",
            "thumbnail" => "config_thumb.jpg",
            "thumbnail_label" => "Thumbnail Thumbnail Image",
            "media_gallery" => [
                "images" => [
                    ["file" => "config_first_gallery.jpg", "position" => 0, "label" => "First Config Gallery", "disabled" => 0],
                    ["file" => "config_second_gallery.jpg", "position" => 1, "label" => "Second Config Gallery", "disabled" => 0],
                    ["file" => "config_disabled_gallery.jpg", "position" => 2, "label" => "Disabled Config Gallery", "disabled" => 1],
                ],
            ],
        ]);

        $this->assignCategory(
            $model,
            MagentoManager::loadCategoryByAttribute("varchar", "url_key", "test-category"),
            3
        );

        Mage::getModel("cataloginventory/stock_item")->addData([
            "stock_id" => 1,
            "use_config_manage_stock" => 0,
            "manage_stock" => 1,
            "is_in_stock" => 1,
        ])->assignProduct($model)->save();

        // Reload after we instantiate to avoid duplicates and conflicts
        $model = Mage::getModel("catalog/product")->load($model->getId());

        // Start configuring the product
        $model->setCanSaveConfigurableAttributes(true);

        $attrCfg = $model->getTypeInstance()->getConfigurableAttributesAsArray($model);

        // Create the attribute configuration if it does not exist
        if(empty($attrCfg)) {
            $model->setConfigurableAttributesData([
                [
                    "label" => $attr->getLabel(),
                    "values" => [],
                    "attribute_id" => $attr->getId(),
                    "attribute_code" => $attr->getAttributeCode(),
                    "frontend_label" => $attr->getFrontend()->getLabel(),
                    "store_label" => $attr->getStoreLabel(),
                ]
            ]);
        }
        else {
            $model->setConfigurableAttributesData($attrCfg);
        }

        $childProducts = [
            $child1->getId() => [
                [
                    "label" => "Red",
                    "attribute_id" => $attr->getId(),
                    "value_index" => $attrRed,
                    "is_percent" => "0",
                    // Value for price, should be ignored with simple-configurable-products:
                    "pricing_value" => "21",
                ]
            ],
            $child2->getId() => [
                [
                    "label" => "Black",
                    "attribute_id" => $attr->getId(),
                    "value_index" => $attrBlack,
                    "is_percent" => "0",
                    // Value for price, should be ignored with simple-configurable-products:
                    "pricing_value" => "22",
                ]
            ],
        ];

        $model->setConfigurableProductsData($childProducts);
        $model->save();

        $this->assignConfigurableProductLinks($model->getId());
    }

    public function assignConfigurableProductLinks(string $entityId) {
        // We need to load it again since the child-product links will cause issues
        $model = Mage::getModel("catalog/product")->load($entityId);

        $simpleId = $model->getIdBySku("test-simple");
        $virtualId = $model->getIdBySku("test-virtual");

        $upSell = [
            $simpleId => [ "position" => "" ],
        ];
        $crossSell = [
            $virtualId => [ "position" => "" ],
        ];
        $related = [
            $virtualId => [ "position" => 1 ],
            $simpleId => [ "position" => 2 ],
        ];

        $model->setUpSellLinkData($upSell);
        $model->setCrossSellLinkData($crossSell);
        $model->setRelatedLinkData($related);

        $model->save();
    }

    public function createUnassignedProduct() {
        $store = Mage::getModel("core/store")->load(MagentoManager::TESTING_STORE);
        $manufacturer = $this->getAttributeValue("manufacturer", "Manufacturer A");

        $model = $this->loadOrCreate([
            // Missing website ids
            "website_ids" => [],
            "type_id" => "simple",
            "sku" => "test-unassigned",
            "name" => "Test for Unassigned Simple Product",
            "manufacturer" => $manufacturer,
            "status" => Mage_Catalog_Model_Product_Status::STATUS_ENABLED,
            "tax_class" => 1, // Default
            "visibility" => Mage_Catalog_Model_Product_Visibility::VISIBILITY_BOTH,
            "price" => 12.34,
            "meta_title" => "Unassigned meta title",
            "meta_description" => "Unassigned meta description",
            "meta_keyword" => "Unassigned,meta,keywords",
            "description" => "The long Unassigned description",
            "short_description" => "The shortest Unassigned description",
            "weight" => 3.14,
            "small_image" => "product_small_image.jpg",
            "small_image_label" => "Small Unassigned Image",
            "thumbnail" => "product_thumbnail.jpg",
            "thumbnail_label" => "Thumbnail Unassigned Image",
            "media_gallery" => [
                "images" => [
                    ["file" => "first_gallery.jpg", "position" => 0, "label" => "First Gallery", "disabled" => 0],
                    ["file" => "second_gallery.jpg", "position" => 1, "label" => "Second Gallery", "disabled" => 0],
                    ["file" => "disabled_gallery.jpg", "position" => 2, "label" => "Disabled Gallery", "disabled" => 1],
                ],
            ],
        ]);

        // We still assign this
        $this->assignCategory(
            $model,
            MagentoManager::loadCategoryByAttribute("varchar", "url_key", "test-category"),
            0
        );

        Mage::getModel("cataloginventory/stock_item")->addData([
            "stock_id" => 1,
            "use_config_manage_stock" => 0,
            "manage_stock" => 1,
            "min_sale_qty" => 1,
            "max_sale_qty" => 2,
            "is_in_stock" => 1,
            "qty" => 999,
        ])->assignProduct($model)->save();
    }

    /**
     * A product with a single optional option which is enabled by default.
     */
    public function createDefaultBundle() {
        $store = Mage::getModel("core/store")->load(MagentoManager::TESTING_STORE);
        $manufacturer = $this->getAttributeValue("manufacturer", "Manufacturer A");

        $optionModel = $this->loadOrCreate([
            "store_id" => $store->getId(),
            "website_ids" => [$store->getWebsiteId()],
            "type_id" => "simple",
            "sku" => "test-bundle-default-option",
            "name" => "Bundle Default Option Product",
            "manufacturer" => $manufacturer,
            "status" => Mage_Catalog_Model_Product_Status::STATUS_ENABLED,
            "tax_class" => 1, // Default
            "visibility" => Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE,
            "price" => 12.34,
            "meta_title" => "Bundle Default Option title",
            "meta_description" => "Bundle Default Option description",
            "meta_keyword" => "Product,meta,keywords",
            "description" => "The long Bundle Default Option",
            "short_description" => "The shortest Bundle Default Option",
            "weight" => 3.14,
            "small_image" => "product_small_image.jpg",
            "small_image_label" => "Small Product Image",
            "thumbnail" => "product_thumbnail.jpg",
            "thumbnail_label" => "Thumbnail Product Image",
            "media_gallery" => [
                "images" => [
                    ["file" => "first_gallery.jpg", "position" => 0, "label" => "First Gallery", "disabled" => 0],
                    ["file" => "second_gallery.jpg", "position" => 1, "label" => "Second Gallery", "disabled" => 0],
                    ["file" => "disabled_gallery.jpg", "position" => 2, "label" => "Disabled Gallery", "disabled" => 1],
                ],
            ],
        ]);

        Mage::getModel("cataloginventory/stock_item")->addData([
            "stock_id" => 1,
            "use_config_manage_stock" => 0,
            "manage_stock" => 1,
            "min_sale_qty" => 1,
            "max_sale_qty" => 2,
            "is_in_stock" => 1,
            "qty" => 999,
        ])->assignProduct($optionModel)->save();

        $model = $this->loadOrCreate([
            "website_ids" => [$store->getWebsiteId()],
            "type_id" => "bundle",
            "sku" => "test-bundle-default",
            "name" => "Test Bundle Default Product",
            "manufacturer" => $manufacturer,
            "status" => Mage_Catalog_Model_Product_Status::STATUS_ENABLED,
            "tax_class" => 1, // Default
            "visibility" => Mage_Catalog_Model_Product_Visibility::VISIBILITY_BOTH,
            "price" => 12.34,
            "price_type" => 1, // Dynamic (0), Fixed (1)
            "price_view" => 0, // Price range (0), Price as low as (1)
            "weight_type" => 0, // Dynamic (0), Fixed (1)
            "shipment_type" => 0, // Together (0), Separately (1)
            "meta_title" => "Test Bundle Default meta title",
            "meta_description" => "Test Bundle meta description",
            "meta_keyword" => "bundle,meta,keywords",
            "description" => "The long Test Bundle Default description",
            "short_description" => "The shortest Test Bundle Default description",
            "small_image" => "product_bundle_small_image.jpg",
            "small_image_label" => "Small Bundle Image",
            "thumbnail" => "product_bundle_thumbnail.jpg",
            "thumbnail_label" => "Thumbnail Bundle Image",
            "media_gallery" => [
                "images" => [
                    ["file" => "first_gallery.jpg", "position" => 0, "label" => "First Gallery", "disabled" => 0],
                    ["file" => "second_gallery.jpg", "position" => 1, "label" => "Second Gallery", "disabled" => 0],
                    ["file" => "disabled_gallery.jpg", "position" => 2, "label" => "Disabled Gallery", "disabled" => 1],
                ],
            ],
        ]);

        $this->assignCategory(
            $model,
            MagentoManager::loadCategoryByAttribute("varchar", "url_key", "test-category"),
            4
        );

        Mage::getModel("cataloginventory/stock_item")->addData([
            "stock_id" => 1,
            "use_config_manage_stock" => 1,
            "manage_stock" => 1,
            "is_in_stock" => 1,
        ])->assignProduct($model)->save();

        $this->assignBundleOptions($model, [
            [
                "title" => "Include extra",
                "type" => "checkbox",
                "required" => 0, // Not required
                "position" => 1,
                "selections" => [
                    [
                        "product_id" => $optionModel->getId(),
                        "selection_price_value" => 10,
                        "selection_price_type" => 0,
                        "selection_qty" => 1,
                        "selection_can_change_qty" => 0,
                        "is_default" => 1, // But is set to default
                        "position" => 1,
                    ],
                ],
            ],
        ]);
    }

    /**
     * A bundle with a hidden product default = 3, not possible to modify,
     * reusing a normal simple product, using fixed pricing.
     */
    public function createHiddenBundle() {
        $store = Mage::getModel("core/store")->load(MagentoManager::TESTING_STORE);
        $manufacturer = $this->getAttributeValue("manufacturer", "Manufacturer A");

        $optionModel = Mage::getModel("catalog/product");
        $optionModel->load($optionModel->getIdBySku("test-simple"));

        $model = $this->loadOrCreate([
            "website_ids" => [$store->getWebsiteId()],
            "type_id" => "bundle",
            "sku" => "test-bundle-hidden",
            "name" => "Test Bundle Hidden Product",
            "manufacturer" => $manufacturer,
            "status" => Mage_Catalog_Model_Product_Status::STATUS_ENABLED,
            "tax_class" => 1, // Default
            "visibility" => Mage_Catalog_Model_Product_Visibility::VISIBILITY_BOTH,
            "price" => 12.34, // Actual price
            "price_type" => 1, // Dynamic (0), Fixed (1)
            "price_view" => 1, // Price range (0), Price as low as (1)
            "weight_type" => 1, // Dynamic (0), Fixed (1)
            "shipment_type" => 0, // Together (0), Separately (1)
            "meta_title" => "Test Bundle Hidden meta title",
            "meta_description" => "Test Bundle meta description",
            "meta_keyword" => "bundle,meta,keywords",
            "description" => "The long Test Bundle Hidden description",
            "short_description" => "The shortest Test Bundle Hidden description",
            "small_image" => "product_bundle_small_image.jpg",
            "small_image_label" => "Small Bundle Image",
            "thumbnail" => "product_bundle_thumbnail.jpg",
            "thumbnail_label" => "Thumbnail Bundle Image",
            "media_gallery" => [
                "images" => [
                    ["file" => "first_gallery.jpg", "position" => 0, "label" => "First Gallery", "disabled" => 0],
                    ["file" => "second_gallery.jpg", "position" => 1, "label" => "Second Gallery", "disabled" => 0],
                    ["file" => "disabled_gallery.jpg", "position" => 2, "label" => "Disabled Gallery", "disabled" => 1],
                ],
            ],
        ]);

        $this->assignCategory(
            $model,
            MagentoManager::loadCategoryByAttribute("varchar", "url_key", "test-category"),
            5
        );

        Mage::getModel("cataloginventory/stock_item")->addData([
            "stock_id" => 1,
            "use_config_manage_stock" => 1,
            "manage_stock" => 1,
            "is_in_stock" => 1,
        ])->assignProduct($model)->save();

        $bundleOptions = [
            [
                "title" => "The Hidden option",
                "type" => "hidden",
                "required" => 1, // Not required
                "position" => 1,
                "selections" => [
                    [
                        "product_id" => $optionModel->getId(),
                        "delete" => "",
                        "selection_price_value" => 0,
                        "selection_price_type" => 0,
                        "selection_qty" => 3,
                        "selection_can_change_qty" => 0,
                        "is_default" => 1,
                    ],
                ],
            ],
        ];

        $this->assignBundleOptions($model, $bundleOptions);
    }

    /**
     * A product with a single optional option which is enabled by default.
     */
    public function createBundleSelect() {
        $store = Mage::getModel("core/store")->load(MagentoManager::TESTING_STORE);
        $manufacturerA = $this->getAttributeValue("manufacturer", "Manufacturer A");
        $manufacturerB = $this->getAttributeValue("manufacturer", "Manufacturer B");

        $optionModel1 = $this->loadOrCreate([
            "store_id" => $store->getId(),
            "website_ids" => [$store->getWebsiteId()],
            "type_id" => "simple",
            "sku" => "test-bundle-select-1",
            "name" => "Bundle Select 1",
            "manufacturer" => $manufacturerA,
            "status" => Mage_Catalog_Model_Product_Status::STATUS_ENABLED,
            "tax_class" => 1, // Default
            "visibility" => Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE,
            "price" => 12.34,
            "meta_title" => "Bundle Select 1",
            "meta_description" => "Bundle Select 1 description",
            "meta_keyword" => "Product,select,meta,keywords",
            "description" => "The long Bundle Select 1",
            "short_description" => "The shortest Select 1",
            "weight" => 3.14,
            "small_image" => "product_small_select1.jpg",
            "small_image_label" => "Small Product Image",
            "thumbnail" => "product_thumbnail.jpg",
            "thumbnail_label" => "Thumbnail Product Image",
            "media_gallery" => [
                "images" => [
                    ["file" => "first_gallery.jpg", "position" => 0, "label" => "First Gallery", "disabled" => 0],
                    ["file" => "second_gallery.jpg", "position" => 1, "label" => "Second Gallery", "disabled" => 0],
                    ["file" => "disabled_gallery.jpg", "position" => 2, "label" => "Disabled Gallery", "disabled" => 1],
                ],
            ],
        ]);

        Mage::getModel("cataloginventory/stock_item")->addData([
            "stock_id" => 1,
            "use_config_manage_stock" => 0,
            "manage_stock" => 1,
            "min_sale_qty" => 1,
            "max_sale_qty" => 2,
            "is_in_stock" => 1,
            "qty" => 999,
        ])->assignProduct($optionModel1)->save();

        $optionModel2 = $this->loadOrCreate([
            "store_id" => $store->getId(),
            "website_ids" => [$store->getWebsiteId()],
            "type_id" => "simple",
            "sku" => "test-bundle-select-2",
            "name" => "Bundle Select 2",
            "manufacturer" => $manufacturerB,
            "status" => Mage_Catalog_Model_Product_Status::STATUS_ENABLED,
            "tax_class" => 1, // Default
            "visibility" => Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE,
            "price" => 12.34,
            "meta_title" => "Bundle Select 2",
            "meta_description" => "Bundle Select 2 description",
            "meta_keyword" => "Product,select,meta,keywords",
            "description" => "The long Bundle Select 2",
            "short_description" => "The shortest Select 2",
            "weight" => 3.14,
            "small_image" => "product_small_select2.jpg",
            "small_image_label" => "Small Product Image",
            "thumbnail" => "product_thumbnail.jpg",
            "thumbnail_label" => "Thumbnail Product Image",
            "media_gallery" => [
                "images" => [
                    ["file" => "first_gallery.jpg", "position" => 0, "label" => "First Gallery", "disabled" => 0],
                    ["file" => "second_gallery.jpg", "position" => 1, "label" => "Second Gallery", "disabled" => 0],
                    ["file" => "disabled_gallery.jpg", "position" => 2, "label" => "Disabled Gallery", "disabled" => 1],
                ],
            ],
        ]);

        Mage::getModel("cataloginventory/stock_item")->addData([
            "stock_id" => 1,
            "use_config_manage_stock" => 0,
            "manage_stock" => 1,
            "min_sale_qty" => 1,
            "max_sale_qty" => 2,
            "is_in_stock" => 1,
            "qty" => 999,
        ])->assignProduct($optionModel2)->save();

        $model = $this->loadOrCreate([
            "website_ids" => [$store->getWebsiteId()],
            "type_id" => "bundle",
            "sku" => "test-bundle-select",
            "name" => "Test Bundle Select Product",
            "manufacturer" => $manufacturerA,
            "status" => Mage_Catalog_Model_Product_Status::STATUS_ENABLED,
            "tax_class" => 1, // Default
            "visibility" => Mage_Catalog_Model_Product_Visibility::VISIBILITY_BOTH,
            "price" => 12.34,
            "price_type" => 1, // Dynamic (0), Fixed (1)
            "price_view" => 0, // Price range (0), Price as low as (1)
            "weight_type" => 0, // Dynamic (0), Fixed (1)
            "shipment_type" => 0, // Together (0), Separately (1)
            "meta_title" => "Test Bundle Select meta title",
            "meta_description" => "Test Bundle meta description",
            "meta_keyword" => "bundle,meta,keywords",
            "description" => "The long Test Bundle Select description",
            "short_description" => "The shortest Test Bundle Select description",
            "small_image" => "product_bundle_small_image.jpg",
            "small_image_label" => "Small Bundle Image",
            "thumbnail" => "product_bundle_thumbnail.jpg",
            "thumbnail_label" => "Thumbnail Bundle Image",
            "media_gallery" => [
                "images" => [
                    ["file" => "first_gallery.jpg", "position" => 0, "label" => "First Gallery", "disabled" => 0],
                    ["file" => "second_gallery.jpg", "position" => 1, "label" => "Second Gallery", "disabled" => 0],
                    ["file" => "disabled_gallery.jpg", "position" => 2, "label" => "Disabled Gallery", "disabled" => 1],
                ],
            ],
        ]);

        $this->assignCategory(
            $model,
            MagentoManager::loadCategoryByAttribute("varchar", "url_key", "test-category"),
            6
        );

        Mage::getModel("cataloginventory/stock_item")->addData([
            "stock_id" => 1,
            "use_config_manage_stock" => 1,
            "manage_stock" => 1,
            "is_in_stock" => 1,
        ])->assignProduct($model)->save();

        $this->assignBundleOptions($model, [
            [
                "title" => "Pick extras",
                "type" => "select",
                "required" => 1,
                "position" => 1,
                "selections" => [
                    [
                        "product_id" => $optionModel1->getId(),
                        "selection_price_value" => 10,
                        "selection_price_type" => 1,
                        "selection_qty" => 1,
                        "selection_can_change_qty" => 0,
                        "is_default" => 0,
                        "position" => 2,
                    ],
                    [
                        "product_id" => $optionModel2->getId(),
                        "selection_price_value" => 20,
                        "selection_price_type" => 1,
                        "selection_qty" => 1,
                        "selection_can_change_qty" => 1,
                        "is_default" => 0,
                        "position" => 1,
                    ],
                ],
            ],
        ]);
    }
}
