<?php

declare(strict_types=1);

use GraphQL\Deferred;
use GraphQL\Type\Definition\ResolveInfo;

class MageQL_Catalog_Model_Category extends Mage_Core_Model_Abstract {
    /**
     * @param mixed $unusedSrc
     */
    public static function resolveRoot(
        $unusedSrc,
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx
    ): ?Mage_Catalog_Model_Category {
        $category = Mage::getModel("catalog/category");

        $category->load($ctx->getStore()->getRootCategoryId());

        if( ! $category->getId() || ! $category->getIsActive()) {
            throw new Exception("Could not find root category");
        }

        return $category;
    }

    /**
     * @param mixed $unusedSrc
     * @return Array<Mage_Catalog_Model_Category>|Deferred
     */
    public static function resolveNavItems(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx,
        ResolveInfo $info
    ) {
        $fakeRoot = new Varien_Object([
            "id" => $ctx->getStore()->getRootCategoryId(),
            // We are faking this, make sure we do not use it as the actual count
            // in resolveChildren
            "children_count" => 1,
        ]);

        return self::resolveChildren($fakeRoot, $args, $ctx, $info);
    }

    /**
     * Obtains the category already populated by loadRouteCategory.
     *
     * @see MageQL_Core_Model_Route
     */
    public static function resolveByRoute(
        Mage_Core_Model_Url_Rewrite $rewrite
    ): ?Mage_Catalog_Model_Category {
        return $rewrite->getCategory();
    }

    /**
     * @param Mage_Catalog_Model_Category|Varien_Object $src
     * @return Array<Mage_Catalog_Model_Category>|Deferred
     */
    public static function resolveChildren(
        $src,
        array $unusedArgs,
        MageQL_Core_Model_Context $unusedContext,
        ResolveInfo $info
    ) {
        if($src->getChildrenCount() < 1) {
            // Optimization, no need to query if we do not have any children
            return [];
        }

        $catAttrs = Mage::getSingleton("mageql_catalog/attributes_category");
        $toSelect = $catAttrs->getUsedAttributes(
            $catAttrs->getAttributesByArea(MageQL_Catalog_Model_Attributes_Abstract::AREA_ANY),
            $info->getFieldSelection(1)
        );

        $children = MageQL_Catalog_Model_Category_Children::instance();

        $children->add($src->getId(), $toSelect);

        return new Deferred(function() use($children, $src) {
            return $children->load()->get($src->getId());
        });
    }

    public static function resolveParent(
        Mage_Catalog_Model_Category $src
    ): ?Mage_Catalog_Model_Category {
        // Skip the root category
        if($src->getLevel() <= 2) {
            return null;
        }

        // Loads the full object
        return $src->getParentCategory();
    }

    /**
     * @param array{page:int, pageSize:int} $args
     */
    public static function resolveProducts(
        Mage_Catalog_Model_Category $category,
        array $args,
        MageQL_Core_Model_Context $ctx
    ): Mage_Catalog_Model_Resource_Product_Collection {
        $collection = Mage::getModel("catalog/product")->getCollection();

        // We do not yet select any attributes, that is done when we
        // fetch the data in the items resolver
        $collection->addCategoryFilter($category);
        $collection->addUrlRewrite($category->getId());
        $collection->addAttributeToSort("position", "asc");

        return MageQL_Catalog_Model_Product::preparePaginatedProductCollection(
            $collection,
            $args,
            $ctx
        );
    }

    /**
     * Event observer which loads the specified category for a redirect, if the
     * load fails it will replace the rewrite with null (interpreted as 404).
     *
     * @see MageQL_Core_Model_Route
     */
    public function loadRouteCategory(Varien_Event_Observer $event): void {
        /**
         * @var Mage_Core_Model_Url_Rewrite
         */
        $rewrite = $event->getRewrite();
        $result = $event->getResult();

        if($event->getIsRedirect() || !$rewrite->getCategoryId()) {
            return;
        }

        $category = Mage::getModel("catalog/category");

        $category->load($rewrite->getCategoryId());

        if($category->getId() && $category->getIsActive()) {
            // Save for later
            $rewrite->setCategory($category);
        }
        else {
            // Not found
            $result->setRewrite(null);
        }
    }
}
