<?php

declare(strict_types=1);

use Spatie\Snapshots\MatchesSnapshots;

use PHPUnit\Framework\TestCase;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;

class RouteTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
    }

    public function onNotSuccessfulTest($e): void {
        MagentoManager::logQueries();

        throw $e;
    }

    public function testNotFound() {
        try {
            // We use the profiler to test the event
            Varien_Profiler::enable();

            $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
                route(path: "this/is-not-found") {
                    type
                }
            }'));

            $this->assertMatchesJsonSnapshot($res->getBody());
            $this->assertEquals(200, $res->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:catalog_controller_product_view", null));
        }
        catch(Throwable $e) {
            Varien_Profiler::disable();

            throw $e;
        }
    }

    public function testUnassignedProduct() {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            route(path: "test-unassigned") {
                type
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProduct() {
        try {
            // We use the profiler to test the event
            Varien_Profiler::enable();

            $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
                route(path: "test-for-simple-product.html") {
                    type
                    ... on RouteProduct {
                        product {
                            sku
                            type
                            name
                            url
                        }
                    }
                }
            }'));

            $this->assertMatchesJsonSnapshot($res->getBody());
            $this->assertEquals(200, $res->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
            $this->assertNotFalse(Varien_Profiler::fetch("DISPATCH EVENT:catalog_controller_product_view", null));
        }
        catch(Throwable $e) {
            Varien_Profiler::disable();

            throw $e;
        }
    }

    public function testCategory() {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            route(path: "test-category.html") {
                type
                ... on RouteCategory {
                    category {
                        name
                        url
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductWCategory() {
        try {
            // We use the profiler to test the event
            Varien_Profiler::enable();

            $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
                route(path: "test-category/test-for-simple-product.html") {
                    type
                    ... on RouteProduct {
                        category {
                            name
                            url
                        }
                        product {
                            sku
                            type
                            name
                            url
                        }
                    }
                }
            }'));

            $this->assertMatchesJsonSnapshot($res->getBody());
            $this->assertEquals(200, $res->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
            $this->assertNotFalse(Varien_Profiler::fetch("DISPATCH EVENT:catalog_controller_product_view", null));
        }
        catch(Throwable $e) {
            Varien_Profiler::disable();

            throw $e;
        }
    }

    public function testCategoryRedirect() {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            route(path: "test-category") {
                type
                ... on RouteRedirect {
                    url
                    isPermanent
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testBestsellingProducts() {
        MagentoManager::initAdmin();

        // Clear first
        $resource = Mage::getModel("core/resource");
        $table = $resource->getTableName("sales/bestsellers_aggregated_monthly");
        $resource->getConnection("core_write")->delete($table);

        MagentoManager::reset();

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            bestsellingProducts {
                totalCount
                items {
                    sku
                    attributes {
                        shortDescription
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));

        MagentoManager::init();

        $resource = Mage::getModel("core/resource");
        $storeId = Mage::app()->getStore()->getId();
        $resource->getConnection("core_write")->insert($table,[
            "period" => date("Y-m-01"),
            "store_id" => $storeId,
            "product_id" => Mage::getModel("catalog/product")->getIdBySku("test-simple"),
            "product_type_id" => "simple",
            "product_price" => 1337,
            "qty_ordered" => 23,
        ]);
        $resource->getConnection("core_write")->insert($table, [
            "period" => date("Y-m-01"),
            "store_id" => $storeId,
            "product_id" => Mage::getModel("catalog/product")->getIdBySku("test-virtual"),
            "product_type_id" => "virtual",
            "product_price" => 1337,
            "qty_ordered" => 355,
        ]);
        $resource->getConnection("core_write")->insert($table, [
            "period" => date("Y-m-01"),
            "store_id" => $storeId,
            "product_id" => Mage::getModel("catalog/product")->getIdBySku("test-config"),
            "product_type_id" => "configurable",
            "product_price" => 1337,
            "qty_ordered" => 14,
        ]);

        MagentoManager::reset();

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            bestsellingProducts {
                totalCount
                items {
                    sku
                    attributes {
                        shortDescription
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));

        MagentoManager::reset();

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            bestsellingProducts (pageSize: 2) {
                totalCount
                items {
                    sku
                    attributes {
                        shortDescription
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));

        MagentoManager::reset();

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            bestsellingProducts (pageSize: 2, page: 2) {
                totalCount
                items {
                    sku
                    attributes {
                        shortDescription
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));

        MagentoManager::reset();

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            bestsellingProducts (pageSize: 2, page: 3) {
                totalCount
                items {
                    sku
                    attributes {
                        shortDescription
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }
}
