<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

use function MageQL\snakeCaseToCamel;

/**
 * @psalm-type ConfigurationOptionItem array{0:Mage_Catalog_Model_Product, 1:Array<Mage_Catalog_Model_Product_Type_Configurable_Attribute>}
 * @psalm-type ConfigurationOptionItemValue array{attribute:string, value:string}
 */
class MageQL_Catalog_Model_Product extends Mage_Core_Model_Abstract {
    /**
     * @param mixed $unusedSrc
     * @param array{sku:string} $args
     */
    public static function resolveBySku(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx
    ): ?Mage_Catalog_Model_Product {
        $product = Mage::getModel("catalog/product");
        $id = $product->getIdBySku($args["sku"]);

        if($id) {
            $product->load($id);
        }

        if( ! Mage::helper("mageql_catalog")->isProductVisible($product, $ctx->getStore())) {
            return null;
        }

        // Magento event to log product visit
        Mage::dispatchEvent("catalog_controller_product_view", ["product" => $product]);

        return $product;
    }

    public static function resolveByRoute(
        Mage_Core_Model_Url_Rewrite $rewrite,
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx
    ): ?Mage_Catalog_Model_Product {
        $product = Mage::getModel("catalog/product");

        $product->load($rewrite->getProductId());

        if( ! Mage::helper("mageql_catalog")->isProductVisible($product, $ctx->getStore())) {
            return null;
        }

        // Magento event to log product visit
        Mage::dispatchEvent("catalog_controller_product_view", ["product" => $product]);

        return $product;
    }

    /**
     * @return Array<Mage_Catalog_Model_Product>
     */
    public static function resolvePaginatedItems(
        Mage_Catalog_Model_Resource_Product_Collection $collection,
        array $unusedArgs,
        MageQL_Core_Model_Context $unusedCtx,
        ResolveInfo $info
    ): array {
        $config = Mage::getSingleton("mageql_catalog/attributes_product");
        // 2 levels deep to get attributes (1 product, 2 attributes)
        $fields = $info->getFieldSelection(2);
        // Merge attributes
        $fields = array_merge($fields, (array)($fields["attributes"] ?? []));

        // We use all attribute sets since we cannot be certain about which sets we will get
        $toSelect = $config->getUsedAttributes(
            $config->getAttributesByArea(MageQL_Catalog_Model_Attributes_Abstract::AREA_LIST),
            $fields
        );

        foreach($toSelect as $col) {
            $collection->addAttributeToSelect($col);
        }

        return $collection->getItems();
    }

    public static function resolveConfigurationOptionAttributes(
        Mage_Catalog_Model_Product $src
    ): Mage_Catalog_Model_Resource_Product_Type_Configurable_Attribute_Collection {
        /**
         * @var Mage_Catalog_Model_Product_Type_Configurable
         */
        $instance = $src->getTypeInstance(true);

        // This will be cached on the product
        return $instance->getConfigurableAttributes($src);
    }

    /**
     * @return ConfigurationOptionItem
     */
    public static function resolveConfigurationOptionItems(
        Mage_Catalog_Model_Product $product,
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx,
        ResolveInfo $info
    ): array {
        /**
         * @var Mage_Catalog_Model_Product_Type_Configurable
         */
        $instance = $product->getTypeInstance(true);
        $config = Mage::getSingleton("mageql_catalog/attributes_product");
        $prodAttrs = $instance->getConfigurableAttributes($product);
        // Get 3 levels deep, since that is the attribute nesting
        $fields = (array)($info->getFieldSelection(3)["product"] ?? []);
        $fields = array_merge($fields, (array)($fields["attributes"] ?? []));

        // Local implementation of Mage_Catalog_Model_Product_Type_Configurable::getUsedProducts

        $collection = $instance->getUsedProductCollection($product);

        $collection->addFilterByRequiredOptions();
        $collection->addStoreFilter($ctx->getStore());
        $collection->addMinimalPrice();
        $collection->addUrlRewrite();
        // TODO: Can we sort for this?
        $collection->addAttributeToSort("position", "asc");

        // No limit

        // We use all attribute sets since we cannot be certain about which
        // sets we will get, even though they should be the same as parent
        $toSelect = $config->getUsedAttributes(
            $config->getAttributesByArea(MageQL_Catalog_Model_Attributes_Abstract::AREA_LIST),
            $fields
        );

        // Add the attributes we use for the configurating
        foreach($prodAttrs as $attr) {
            $collection->addAttributeToSelect($attr->getProductAttribute()->getAttributeCode());
        }

        foreach($toSelect as $col) {
            $collection->addAttributeToSelect($col);
        }

        return array_map(function(Mage_Catalog_Model_Product $item) use($prodAttrs) {
            return [$item, $prodAttrs];
        }, $collection->getItems());
    }

    /**
     * @param ConfigurationOptionItem $src
     * @return Array<array{attribute:string, value:string}>
     */
    public static function resolveConfigurationOptionItemValues($src) {
        $items = [];
        $prod = $src[0];
        $attrs = $src[1];

        foreach($attrs as $attr) {
            $prodAttr = $attr->getProductAttribute();
            $model = $prodAttr->getSource();

            $items[] = [
                "attribute" => snakeCaseToCamel($prodAttr->getAttributeCode()),
                "value" => (string)$model->getOptionText($prod->getData($prodAttr->getAttributeCode())),
            ];
        }

        return $items;
    }

    /**
     * @param array{page:int, pageSize:int} $args
     */
    public static function resolveRelatedProducts(
        Mage_Catalog_Model_Product $src,
        array $args,
        MageQL_Core_Model_Context $ctx
    ): Mage_Catalog_Model_Resource_Product_Link_Product_Collection {
        return self::prepareProductLinkCollection(
            $src->getRelatedProductCollection(),
            $args,
            $ctx
        );
    }

    /**
     * @param array{page:int, pageSize:int} $args
     */
    public static function resolveUpSellProducts(
        Mage_Catalog_Model_Product $src,
        array $args,
        MageQL_Core_Model_Context $ctx
    ): Mage_Catalog_Model_Resource_Product_Link_Product_Collection {
        return self::prepareProductLinkCollection(
            $src->getUpSellProductCollection(),
            $args,
            $ctx
        );
    }

    /**
     * @param array{page:int, pageSize:int} $args
     */
    public static function resolveCrossSellProducts(
        Mage_Catalog_Model_Product $src,
        array $args,
        MageQL_Core_Model_Context $ctx
    ): Mage_Catalog_Model_Resource_Product_Link_Product_Collection {
        return self::prepareProductLinkCollection(
            $src->getCrossSellProductCollection(),
            $args,
            $ctx
        );
    }

    /**
     * @param mixed $unusedSrc
     * @param array{page:int, pageSize:int} $args
     */
    public static function resolveBestsellingProducts(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx
    ): Mage_Catalog_Model_Resource_Product_Collection {
        $storeId = (int)$ctx->getStore()->getId();
        $collection = Mage::getModel("catalog/product")->getCollection();
        $fromDate = date("Y-m-01");

        $collection->getSelect()
            ->join(
                ["aggregation" => $collection->getResource()->getTable("sales/bestsellers_aggregated_monthly")],
                sprintf("e.entity_id = aggregation.product_id AND aggregation.store_id = %d AND aggregation.period = '%s'", $storeId, $fromDate),
                ["SUM(aggregation.qty_ordered) AS sold_quantity"]
            )
            ->group("e.entity_id")
            ->order(array("sold_quantity DESC", "e.created_at"));

        return self::preparePaginatedProductCollection(
            $collection,
            $args,
            $ctx
        );
    }

    /**
     * @param mixed $unusedSrc
     * @param array{page:int, pageSize:int, value:string}|array{page:int, pageSize:int, to:string, from:string} $args
     */
    public static function resolveProductsByFilter(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx,
        ResolveInfo $info
    ): Mage_Catalog_Model_Resource_Product_Collection {
        $attributeConfig = Mage::getSingleton("mageql_catalog/attributes_product");
        $attrs = $attributeConfig->getAttributes();

        if( ! array_key_exists($info->fieldName, $attrs)) {
            throw new Exception(sprintf(
                "%s: Unknown attribute field %s.",
                __METHOD__,
                $info->fieldName
            ));
        }

        $attr = $attrs[$info->fieldName];
        $collection = Mage::getModel("catalog/product")->getCollection();
        $attribute = Mage::getModel("catalog/product")->getResource()->getAttribute($attr["code"]);

        if( ! $attribute) {
            throw new Exception(sprintf(
                "%s: Product attribute %s could not be found",
                __METHOD__,
                $attr["code"]
            ));
        }

        if(array_key_exists("value", $args)) {
            $value = $attribute->usesSource() ?
                $attribute->getSource()->getOptionId($args["value"]) :
                $args["value"];

            $collection->addAttributeToFilter($attr["code"], $value);
        }
        else if($attr["input"] === "datetime") {
            if( ! empty($attr["from"])) {
                $collection->addAttributeToFilter([
                    "attribute" => $attr["code"],
                    "gteq" => date("Y-m-d H:i:s", strtotime($attr["from"])),
                ]);
            }

            if( ! empty($attr["to"])) {
                $collection->addAttributeToFilter([
                    "attribute" => $attr["code"],
                    "lteq" => date("Y-m-d H:i:s", strtotime($attr["to"])),
                ]);
            }
        }
        else {
            if( ! empty($attr["from"])) {
                $collection->addAttributeToFilter([
                    "attribute" => $attr["code"],
                    "gteq" => (float)$attr["from"],
                ]);
            }

            if( ! empty($attr["to"])) {
                $collection->addAttributeToFilter([
                    "attribute" => $attr["code"],
                    "lteq" => (float)$attr["to"],
                ]);
            }
        }

        // Just plain URLs here since we do not have a category
        $collection->addUrlRewrite();
        $collection->addAttributeToSort("name", Varien_Data_Collection::SORT_ORDER_ASC);

        return self::preparePaginatedProductCollection($collection, $args, $ctx);
    }

    /**
     * @param mixed $unusedSrc
     */
    public static function resolveProductsBySearch(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx
    ): ?Mage_Catalog_Model_Resource_Product_Collection {
        $query = Mage::getModel("catalogsearch/query");
        $fulltext = Mage::getModel("catalogsearch/fulltext");
        $resource = $fulltext->getResource();
        $minLength = $ctx->getStore()->getConfig(Mage_CatalogSearch_Model_Query::XML_PATH_MIN_QUERY_LENGTH);

        if(strlen($args["term"]) < $minLength) {
            return null;
        }

        $query->setStoreId($ctx->getStore()->getId());
        $query->loadByQuery($args["term"]);

        if( ! $query->getId()) {
            $query->setQueryText($args["term"]);
            $query->setStoreId($ctx->getStore()->getId());
            $query->setPopularity(1);
        }
        else {
            $query->setPopularity($query->getPopularity() + 1);
        }

        $query->prepare();

        $resource->prepareResult($fulltext, $args["term"], $query);

        $searchHelper = Mage::getResourceHelper("catalogsearch");
        $collection = Mage::getModel("catalog/product")->getCollection();
        $foundData = $resource->getFoundData();

        ksort($foundData);
        natsort($foundData);

        $foundIds = array_keys($foundData);

        $collection->addIdFilter($foundIds);
        $collection->addUrlRewrite();

        // TODO: Will this make it too slow?
        // Sort by relevance
        $order = new Zend_Db_Expr(
            $searchHelper->getFieldOrderExpression("e.entity_id", $foundIds).
            " ASC, e.entity_id ASC"
        );
        $select = $collection->getSelect();
        $select->order($order);

        $query->save();

        return self::preparePaginatedProductCollection($collection, $args, $ctx);
    }

    /**
     * Prepares a product-link collection like related, upsell and crossell for
     * use in a PaginatedProducts type.
     *
     * @param array{page:int, pageSize:int} $args
     */
    public static function prepareProductLinkCollection(
        Mage_Catalog_Model_Resource_Product_Link_Product_Collection $collection,
        array $args,
        MageQL_Core_Model_Context $ctx
    ): Mage_Catalog_Model_Resource_Product_Link_Product_Collection {
        // Just plain URLs here since we do not have a category
        $collection->addUrlRewrite();
        $collection->setPositionOrder();

        return self::preparePaginatedProductCollection($collection, $args, $ctx);
    }

    /**
     * Prepares a basic product collection for use in a PaginatedProducts type,
     * does not add sort, url-rewrite, or any kind of filter to the collection
     * besides store and visibility.
     *
     * @template T of Mage_Catalog_Model_Resource_Product_Collection
     * @param T $collection
     * @param array{page:int, pageSize:int} $args
     * @return T
     */
    public static function preparePaginatedProductCollection(
        $collection,
        array $args,
        MageQL_Core_Model_Context $ctx
    ) {
        $visibility = ["in" => [
            Mage_Catalog_Model_Product_Visibility::VISIBILITY_IN_CATALOG,
            Mage_Catalog_Model_Product_Visibility::VISIBILITY_BOTH
        ]];

        // We do not yet select any attributes, that is done when we
        // fetch the data in the items resolver
        $collection->addAttributeToFilter("visibility", $visibility);
        $collection->addStoreFilter($ctx->getStore());
        $collection->addMinimalPrice();

        // We call this directly on the select to prevent magento from stopping us from
        // going off the end of the list. (setCurPage() prevents this, which is not correct
        // from an API PoV)
        $collection->getSelect()
                   ->limitPage(max($args["page"], 1), $args["pageSize"]);

        return $collection;
    }
}
