<?php

declare(strict_types=1);

use Spatie\Snapshots\MatchesSnapshots;

use PHPUnit\Framework\TestCase;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;

class CategoryTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
    }

    public function onNotSuccessfulTest($e): void {
        MagentoManager::logQueries();

        throw $e;
    }

    public function testRootCategory() {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            rootCategory {
                url
                name
                image
                smaller: image(width: 50)
                resized: image(width: 20, height: 20)
                fill: image(width: 20, height: 20, fill: true)
                fillSquare: image(width: 20, fill: true)
                displayMode
                description
                metaDescription
                metaTitle
                metaKeywords
                parent {
                    url
                    name
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testNav() {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            categoryNav {
                url
                name
                image
                smaller: image(width: 50)
                resized: image(width: 20, height: 20)
                displayMode
                description
                metaDescription
                metaTitle
                metaKeywords
                parent {
                    url
                    name
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testChildren() {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            rootCategory {
                name
                children {
                    url
                    name
                    children {
                        url
                        name
                        children {
                            url
                            name
                            parent {
                                url
                                name
                            }
                        }
                        parent {
                            url
                            name
                        }
                    }
                }
                parent {
                    url
                    name
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProducts() {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            categoryNav {
                name
                products {
                    totalCount
                    items {
                        sku
                        name
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductsEmpty() {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            rootCategory {
                name
                products {
                    totalCount
                    items {
                        sku
                        name
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductsWConfigurable() {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            categoryNav {
                name
                products {
                    totalCount
                    items {
                        sku
                        name
                        ... on ListProductConfigurable {
                            configOptions {
                                attributes { attribute label }
                                items {
                                    values { attribute value }
                                    product {
                                        sku
                                        name
                                        price { incVat }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testNavProductUrls() {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            categoryNav {
                url
                name
                products {
                    totalCount
                    items {
                        sku
                        name
                        url
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testDeferredChildren() {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            categoryNav {
                url
                name
            }
            route(path: "test-category.html") {
                type
                ... on RouteCategory {
                    category {
                        url
                        name
                        children {
                            url
                            name
                            image
                            description
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }
}
