<?php

declare(strict_types=1);

/**
 * Class extending the default attribute model with the extra fields Catalog
 * provides for categories and products.
 *
 * Fields:
 *
 *  * appy_to: array of string, product types
 *  * show_in_detail: bool
 *  * show_in_list: bool
 */
abstract class MageQL_Catalog_Model_Attributes_Abstract extends MageQL_Core_Model_Attributes_Abstract {
    const AREA_ANY = "BOTH";
    const AREA_DETAIL = "Detail";
    const AREA_LIST = "List";

    // Join more data on this since categories and products have different representations
    protected function loadAttributes() {
        $db = Mage::getSingleton("core/resource")->getConnection("core_read");

        return $db->query(
"SELECT a.attribute_code code,
    a.backend_type,
    a.frontend_input input,
    a.frontend_label label,
    a.is_required required,
    COALESCE(c.apply_to, '') apply_to,
    a.is_user_defined not_system,
    COALESCE(GROUP_CONCAT(DISTINCT s.attribute_set_name SEPARATOR ','), '') attribute_set,
    c.is_visible_on_front show_in_detail,
    c.used_in_product_listing show_in_list,
    c.is_filterable filterable
FROM eav_attribute a
JOIN eav_entity_type t ON t.entity_type_id = a.entity_type_id
JOIN catalog_eav_attribute c ON c.attribute_id = a.attribute_id
JOIN eav_entity_attribute e ON e.attribute_id = a.attribute_id
JOIN eav_attribute_set s ON s.attribute_set_id = e.attribute_set_id
WHERE t.entity_type_code = ?
  AND a.backend_type <> 'static'
  AND (c.is_visible_on_front = 1 OR c.used_in_product_listing = 1)
GROUP BY a.attribute_id
ORDER BY a.attribute_code ASC", [$this->getEntityType()])->fetchAll();
    }

    protected function filterAttributeData($a) {
        $a["show_in_detail"] = (bool)$a["show_in_detail"];
        $a["show_in_list"] = (bool)$a["show_in_list"];
        $a["filterable"] = (bool)$a["filterable"];
        $a["apply_to"] = array_filter(array_map("trim", explode(",", $a["apply_to"])));

        return parent::filterAttributeData($a);
    }

    /**
     * Returns all system attributes for a given area and type, if type is omitted
     * only attributes applying to all types are returned.
     */
    public function getSystemAttributes(string $area, ?string $typeId = null): array {
        $attrs = [];

        foreach($this->getAttributes() as $field => $a) {
            $isVisible = $area === self::AREA_ANY ||
                $area === self::AREA_DETAIL && $a["show_in_detail"] ||
                $area === self::AREA_LIST && $a["show_in_list"];
            $canApply = (empty($a["apply_to"]) && empty($a["apply_to"])) ||
                $typeId && in_array($typeId, $a["apply_to"]);

            if( ! $a["not_system"] && $isVisible && $canApply) {
                $attrs[$field] = $a;
            }
        }

        return $attrs;
    }

    /**
     * Returns all attributes for a given area, set, and type, if type is omitted
     * only attributes applying to all types are returned.
     */
    public function getSetAttributes(string $area, string $setType, ?string $appliesTo = null): array {
        $attrs = [];

        foreach($this->getAttributes() as $field => $a) {
            $isVisible = $area === self::AREA_ANY ||
                $area === self::AREA_DETAIL && $a["show_in_detail"] ||
                $area === self::AREA_LIST && $a["show_in_list"];
            $inCurrentSet = ( ! $a["not_system"]) ||
                in_array($setType, $a["attribute_set_type"]);
            $canApply = (empty($a["apply_to"]) && empty($a["apply_to"])) ||
                $appliesTo && in_array($appliesTo, $a["apply_to"]);

            if($inCurrentSet && $isVisible && $canApply) {
                $attrs[$field] = $a;
            }
        }

        return $attrs;
    }

    /**
     * Returns a map of field-name -> attribute data for all attributes visible
     * in the supplied area type.
     */
    public function getAttributesByArea(string $area): array {
        $attrs = [];

        foreach($this->getAttributes() as $field => $a) {
            $isVisible = $area === self::AREA_ANY ||
                $area === self::AREA_DETAIL && $a["show_in_detail"] ||
                $area === self::AREA_LIST && $a["show_in_list"];

            if($isVisible) {
                $attrs[$field] = $a;
            }
        }

        return $attrs;
    }
}
