<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

use function MageQL\snakeCaseToCamel;

use MageQL\Registry;
use MageQL\Type\AbstractBuilder;

class MageQL_Catalog_Model_Schema_Default_Product extends MageQL_Core_Model_Schema_Abstract {
    const DEFAULT_LINK_SIZE = 4;
    const DEFAULT_PAGE_SIZE = 20;

    protected $config;

    public function __construct() {
        $this->config = Mage::getSingleton("mageql_catalog/attributes_product");
    }

    public function getTypeBuilder(string $typeName, Registry $registry): ?AbstractBuilder {
        switch($typeName) {
        case "ConfigurationAttribute":
            return $this->object("An attribute which can be configured on configurable products");

        case "ConfigurationOptionItem":
            return $this->object("A child-product/configuration-item available to pick for a configurable product");

        case "ConfigurationOptionItemValue":
            return $this->object("A value for the configurable attribute on configurable products")
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "ConfigurationOptions":
            return $this->object("Available options and their attributes for a configurable product");

        case "GalleryItem":
            return $this->object("Media Gallery Image");

        case "ListProduct":
            return $this->interface("A partially populated product of unknown type")
                ->setResolveType(function($item) use($registry) {
                    return $registry->getType("ListProduct".ucfirst($item->getTypeId()));
                });

        case "ProductAttribute":
            return $this->object("Information about a filterable product attribute");

        case "ProductAttributes":
            return $this->object("Filterable product attributes and their values");

        case "ProductType":
            return $this->enum("Type indicating variant of product", [
                "bundle" => [
                    "description" => "Complex product containing multiple variants",
                ],
                "configurable" => [
                    "description" => "Complex product containing variants",
                ],
                "simple" => [
                    "description" => "Simple single product",
                ],
                "virtual" => [
                    "description" => "Simple product without physical representation",
                ],
            ]);

        case "PaginatedProducts":
            return $this->object("Object containing a list of partially populated products");

        case "ProductDetail":
            return $this->interface("A fully populated product of unknown type")
                ->setResolveType(function($item) use($registry) {
                    return $registry->getType("ProductDetail".ucfirst($item->getTypeId()));
                });

        case "ProductPrice":
            return $this->object("Price information for a product in base store currency")
                ->setInterfaces(["Price"]);

        case "ProductsBy":
            return $this->object("Available filters for products");

        case "RouteProduct":
            return $this->object("A response containing a detailed product")
                ->setInterfaces(["Route"]);
        }

        if(strpos($typeName, "ProductDetail") === 0) {
            return $this->createProductTypeBuilder(
                MageQL_Catalog_Model_Attributes_Abstract::AREA_DETAIL,
                substr($typeName, strlen("ProductDetail")),
                $registry
            );
        }

        if(strpos($typeName, "ListProduct") === 0) {
            return $this->createProductTypeBuilder(
                MageQL_Catalog_Model_Attributes_Abstract::AREA_LIST,
                substr($typeName, strlen("ListProduct")),
                $registry
            );
        }

        return null;
    }

    public function getTypeFields(string $typeName, Registry $registry): array {
        switch($typeName) {
        case "ConfigurationAttribute":
            return [
                "attribute" => $this->field("String!", "Attribute property")
                    ->setResolver(function($src) {
                        return snakeCaseToCamel($src->getProductAttribute()->getAttributeCode());
                    }),
                "label" => $this->field("String!", "Attribute label")
                    ->setResolver(function($src) {
                        $prodAttr = $src->getProductAttribute();

                        return $prodAttr->getStoreLabel() ?: $prodAttr->getFrontend()->getLabel() ?: $src->getLabel();
                    }),
            ];

        case "ConfigurationOptionItem":
            return [
                "product" => $this->field("ListProduct!", "The product")
                    ->setResolver(function($src) {
                        return $src[0];
                    }),
                "values" => $this->field("[ConfigurationOptionItemValue!]!", "List of values this item fulfills")
                    ->setResolver("MageQL_Catalog_Model_Product::resolveConfigurationOptionItemValues"),
            ];

        case "ConfigurationOptionItemValue":
            return [
                "attribute" => $this->field("String!", "Attribute property this value belongs to")
                    ->setResolver(function($src) {
                        return $src["attribute"];
                    }),
                "value" => $this->field("String!", "Attribute value")
                    ->setResolver(function($src) {
                        return $src["value"];
                    }),
            ];

        case "ConfigurationOptions":
            return [
                "attributes" => $this->field("[ConfigurationAttribute!]!", "List of configurable attributes")
                    ->setResolver("MageQL_Catalog_Model_Product::resolveConfigurationOptionAttributes"),
                "items" => $this->field("[ConfigurationOptionItem!]!", "List of items to pick from")
                    ->setResolver("MageQL_Catalog_Model_Product::resolveConfigurationOptionItems"),
            ];

        case "GalleryItem":
            return [
                "image" => $this->field("String!", "Image url")
                    ->addArgument("width", $this->argument("Int", "Maximum width of the image"))
                    ->addArgument("height", $this->argument("Int", "Minimum width of the image"))
                    ->addArgument(
                        "fill",
                        $this->argument("Boolean", "If to fill the image to the given size")
                            ->setDefaultValue(false))
                    ->setResolver(function($src, array $args) {
                        [$product, $image] = $src;

                        $value = $image->getFile();

                        if( ! $value || $value === "/no_selection") {
                            return null;
                        }

                        return $this->config->resizeImage($product, "gallery", $value, $args);
                    }),
                "label" => $this->field("String", "Image label")
                    ->setResolver(function($src) {
                        [$product, $image] = $src;

                        return $image->getLabel();
                    }),
            ];

        case "ListProduct":
            return [
                // TODO: Some should be reusable here
                "sku" => $this->field("String!", "SKU"),
                "type" => $this->field("ProductType!", "Product type")
                    ->setResolver(function($src) { return $src->getTypeId(); }),
                "name" => $this->field("String!", "Product name"),
                "attributes" => $this->field("ListProductAttributes!", "Product attributes for list")
                    ->setResolver("MageQL\\forwardResolver"),
                "attributeSetName" => $this->field("String!", "Attribute set name")
                    ->setResolver(function($src) {
                        return $this->config->getSetById($src->getAttributeSetId())["name"];
                    }),
                "originalPrice" => $this->field("ProductPrice!", "Product original price")
                    ->setResolver(function($src) {
                        return new MageQL_Catalog_Model_Product_Price($src, $src->getPrice());
                    }),
                "price" => $this->field("ProductPrice!", "Product final price")
                    ->setResolver(function($src) {
                        return new MageQL_Catalog_Model_Product_Price($src, $src->getFinalPrice());
                    }),
                // TODO: Fields
                // stock
                // options (configurable/bundle)
                "url" => $this->field("String!", "URL")->setResolver(function($src) {
                    return $this->context->stripBaseUrl($src->getProductUrl());
                }),
            ];

        case "ListProductAttributes":
            return $this->config->createFields(
                $this->config->getSystemAttributes(MageQL_Catalog_Model_Attributes_Abstract::AREA_LIST)
            );

        case "ListProductBundle":
            // TODO: Specifc fields which are bundle only?
            // TODO: Bundle options
            return $registry->getFieldBuilders("ListProduct");

        case "ListProductConfigurable":
            // TODO: Specifc fields which are configurable only?
            return array_merge($registry->getFieldBuilders("ListProduct"), [
                "configOptions" => $this->field("ConfigurationOptions!", "Options for a configurable product")
                    ->setResolver("MageQL\\forwardResolver"),
            ]);

        case "ListProductSimple":
            // TODO: Specifc fields which are simple only?
            return $registry->getFieldBuilders("ListProduct");

        case "ListProductVirtual":
            // TODO: Specifc fields which are virtual only?
            return $registry->getFieldBuilders("ListProduct");

        case "PaginatedProducts":
            return [
                "items" => $this->field("[ListProduct!]!", "List of products")
                    ->setResolver("MageQL_Catalog_Model_Product::resolvePaginatedItems"),
                "totalCount" => $this->field("Int!", "Total number of products in this paginated collection")
                    ->setResolver(function($collection) {
                        return $collection->getSize();
                    }),
            ];

        case "ProductAttribute":
            return [
                "label" => $this->field("String!", "The attribute label")
                    ->setResolver("MageQL_Catalog_Model_Attributes_Product::resolveLabel"),
                "values" => $this->field("[String!]", "If the attribute has different values which can be selected they will be listed here")
                    ->setResolver("MageQL_Catalog_Model_Attributes_Product::resolveValues"),
            ];

        case "ProductAttributes":
            return $this->createProductAttributesFileds($registry);

        case "ProductDetail":
            return [
                "sku" => $this->field("String!", "SKU"),
                "type" => $this->field("ProductType!", "Product type")
                    ->setResolver(function($src) { return $src->getTypeId(); }),
                "name" => $this->field("String!", "Product name"),
                "attributes" => $this->field("ProductDetailAttributes!", "Product attributes for detail page")
                    ->setResolver("MageQL\\forwardResolver"),
                "attributeSetName" => $this->field("String!", "Attribute set name")
                    ->setResolver(function($src) {
                        return $this->config->getSetById($src->getAttributeSetId())["name"];
                    }),
                "gallery" => $this->field("[GalleryItem]!", "Media gallery")
                    ->setResolver(function($product) {
                        return array_map(function($image) use($product) {
                            // The product instance is required to be able to generate a link
                            return [$product, $image];
                        }, $product->getMediaGalleryImages()->getItems());
                    }),
                "originalPrice" => $this->field("ProductPrice!", "Product original price")
                    ->setResolver(function($src) {
                        return new MageQL_Catalog_Model_Product_Price($src, $src->getPrice());
                    }),
                "price" => $this->field("ProductPrice!", "Product final price")
                    ->setResolver(function($src) {
                        return new MageQL_Catalog_Model_Product_Price($src, $src->getFinalPrice());
                    }),
                "url" => $this->field("String!", "URL")->setResolver(function($src) {
                    return $this->context->stripBaseUrl($src->getProductUrl());
                }),
                "crossSellProducts" => $this->field("PaginatedProducts!", "Cross-sell products")
                    ->addArgument(
                        "pageSize",
                        $this->argument("Int", "Maximum number of products to list")
                            ->setDefaultValue(self::DEFAULT_LINK_SIZE))
                    ->addArgument(
                        "page",
                        $this->argument("Int", "Which page to show")
                            ->setDefaultValue(1))
                    ->setResolver("MageQL_Catalog_Model_Product::resolveCrossSellProducts"),
                "relatedProducts" => $this->field("PaginatedProducts!", "Related products")
                    ->addArgument(
                        "pageSize",
                        $this->argument("Int", "Maximum number of products to list")
                            ->setDefaultValue(self::DEFAULT_LINK_SIZE))
                    ->addArgument(
                        "page",
                        $this->argument("Int", "Which page to show")
                            ->setDefaultValue(1))
                    ->setResolver("MageQL_Catalog_Model_Product::resolveRelatedProducts"),
                "upSellProducts" => $this->field("PaginatedProducts!", "Up-sell products")
                    ->addArgument(
                        "pageSize",
                        $this->argument("Int", "Maximum number of products to list")
                            ->setDefaultValue(self::DEFAULT_LINK_SIZE))
                    ->addArgument(
                        "page",
                        $this->argument("Int", "Which page to show")
                            ->setDefaultValue(1))
                    ->setResolver("MageQL_Catalog_Model_Product::resolveUpSellProducts"),
                // TODO: Fields
                // group prices
                // tier price
                // special price
                // available
                // stock
                // options (bundle)
                // custom options
            ];

        case "ProductDetailAttributes":
            return $this->config->createFields(
                $this->config->getSystemAttributes(MageQL_Catalog_Model_Attributes_Abstract::AREA_DETAIL)
            );

        case "ProductDetailBundle":
            // TODO: Specifc fields which are bundle only?
            // TODO: Bundle variants
            return $registry->getFieldBuilders("ProductDetail");

        case "ProductDetailConfigurable":
            // TODO: Specifc fields which are configurable only?
            return array_merge($registry->getFieldBuilders("ProductDetail"), [
                "configOptions" => $this->field("ConfigurationOptions!", "Options for a configurable product")
                    ->setResolver("MageQL\\forwardResolver"),
            ]);

        case "ProductDetailSimple":
            // TODO: Specifc fields which are simple only?
            return $registry->getFieldBuilders("ProductDetail");

        case "ProductDetailVirtual":
            // TODO: Specifc fields which are virtual only?
            return $registry->getFieldBuilders("ProductDetail");

        case "ProductPrice":
            return [
                "exVat" => $this->field("Float!", "Price excluding VAT")
                    ->setResolver(function($src) { return $src->getExVat(); }),
                "incVat" => $this->field("Float!", "Price including VAT")
                    ->setResolver(function($src) { return $src->getIncVat(); }),
                "vat" => $this->field("Float!", "VAT amount")
                    ->setResolver(function($src, $args, $ctx) { return $src->getVat($ctx->getStore()); }),
            ];


        case "ProductsBy":
            return $this->createProductsByFilters($registry);

        case "Query":
            return [
                "bestsellingProducts" => $this->field("PaginatedProducts!", "All purchased products ordered by number of ordered products, from all users in the store")
                    ->addArgument(
                        "pageSize",
                        $this->argument("Int", "Maximum number of products to list")
                            ->setDefaultValue(self::DEFAULT_PAGE_SIZE))
                    ->addArgument(
                        "page",
                        $this->argument("Int", "Which page to show")
                            ->setDefaultValue(1))
                    ->setResolver("MageQL_Catalog_Model_Product::resolveBestsellingProducts"),
                "productAttributes" => $this->field("ProductAttributes!", "Product attributes and thier data")
                    ->setResolver("MageQL\\forwardResolver"),
                "productBySku" => $this->field("ProductDetail", "Detailed product information about a specific SKU")
                    ->addArgument("sku", $this->argument("String!", "Product SKU"))
                    ->setResolver("MageQL_Catalog_Model_Product::resolveBySku"),
                "productsBy" => $this->field("ProductsBy!", "Filter products by a specific attribute")
                    ->setResolver("MageQL\\forwardResolver"),
                "productsBySearch" => $this->field("PaginatedProducts", "Filter products by a specified search term, null means the term is too short")
                    ->addArgument("term", $this->argument("String!", "Search term/phrase"))
                    ->addArgument(
                        "pageSize",
                        $this->argument("Int", "Maximum number of products to list")
                            ->setDefaultValue(self::DEFAULT_PAGE_SIZE))
                    ->addArgument(
                        "page",
                        $this->argument("Int", "Which page to show")
                            ->setDefaultValue(1))
                    ->setResolver("MageQL_Catalog_Model_Product::resolveProductsBySearch"),
            ];

        case "RouteProduct":
            return [
                "type" => $this->field("RouteType!", "Type of route")
                    ->setResolver(function() {
                        return "product";
                    }),
                // In the case of a category
                "category" => $this->field("Category", "The parent category of the product, if any. This will depend on the route")
                    ->setResolver("MageQL_Catalog_Model_Category::resolveByRoute"),
                "product" => $this->field("ProductDetail!", "The product")
                    ->setResolver("MageQL_Catalog_Model_Product::resolveByRoute"),
            ];
        }

        if(strpos($typeName, "ProductDetail") === 0) {
            return $this->createProductAttributeFieldBuilder(
                MageQL_Catalog_Model_Attributes_Abstract::AREA_DETAIL,
                substr($typeName, strlen("ProductDetail"))
            );
        }

        if(strpos($typeName, "ListProduct") === 0) {
            return $this->createProductAttributeFieldBuilder(
                MageQL_Catalog_Model_Attributes_Abstract::AREA_LIST,
                substr($typeName, strlen("ListProduct"))
            );
        }

        return [];
    }

    public function areaToTypeName($area) {
        switch($area) {
        case MageQL_Catalog_Model_Attributes_Abstract::AREA_LIST:
            return "ListProduct";
        case MageQL_Catalog_Model_Attributes_Abstract::AREA_DETAIL:
            return "ProductDetail";
        default:
            throw new Exception(sprintf("Unknown attribute area '%s'.", $area));
        }
    }

    public function resolveAttributeType($src, $area, $registry) {
        $setType = $this->config->getSetById((int)$src->getAttributeSetId())["type"];
        $typeId = ucfirst($src->getTypeId());
        $prefix = $this->areaToTypeName($area);

        return $registry->getType($prefix.$typeId."Attributes".$setType);
    }

    public function createProductTypeBuilder($area, $typeRemainder, $registry) {
        $parentType = $this->areaToTypeName($area);

        if(strpos($typeRemainder, "Attributes") === false) {
            // It must be a concrete product type for a type id
            if( ! in_array(lcfirst($typeRemainder), MageQL_Catalog_Model_Attributes_Product::SUPPORTED_TYPE_IDS)) {
                return null;
            }

            $description = "A ".
                ($area === MageQL_Catalog_Model_Attributes_Product::AREA_LIST ? "partially" : "fully").
                " populated ".lcfirst($typeRemainder)." product";

            return $this->object($description)
                ->setResolveField("MageQL\\defaultVarienObjectResolver")
                ->setInterfaces([$parentType]);
        }

        [$typeId, $setType] = explode("Attributes", $typeRemainder, 2);

        if(empty($typeId) && empty($setType)) {
            // Base interface
            $description = sprintf("Product %s Attribute Interface, for a ".
                ($area === MageQL_Catalog_Model_Attributes_Product::AREA_LIST ? "partial" : "full").
                " product", $area);

            return $this->interface($description)
                ->setResolveType(function($src) use($registry, $area) {
                    return $this->resolveAttributeType($src, $area, $registry);
                });
        }

        if(empty($setType)) {
            // TODO: Make this interface inherit from the base attribute set
            // when interface-inheritance makes it in
            $description = sprintf("Product %s %s Attribute Interface", $area, $typeId);

            return $this->interface($description)
                ->setResolveType(function($src) use($area, $registry) {
                    return $this->resolveAttributeType($src, $area, $registry);
                });
        }

        $setName = $this->config->getSetByType($setType)["name"];

        if(empty($typeId)) {
            // TODO: Make this interface inherit from the base attribute set
            // when interface-inheritance makes it in
            $desc = sprintf("Product %s Attribute Set %s Interface", $area, $setName);

            return $this->interface($desc)
                ->setResolveType(function($src) use($area, $registry) {
                    return $this->resolveAttributeType($src, $area, $registry);
                });
        }

        // All the interfaces
        $typeDesc = sprintf("Product %s %s Attribute Set %s", $area, $typeId, $setName);
        $baseAttributeInterface = $parentType."Attributes";
        $setAttributeInterface = $parentType."Attributes".$setType;
        $productTypeInterface = $parentType.$typeId."Attributes";

        return $this->object($typeDesc)
            ->setResolveField("MageQL\\defaultVarienObjectResolver")
            ->setInterfaces([$baseAttributeInterface, $productTypeInterface, $setAttributeInterface]);
    }

    public function createProductAttributeFieldBuilder($area, $typeRemainder) {
        if(strpos($typeRemainder, "Attributes") === false) {
            throw new Exception($area.$typeRemainder." does not have any defined fields");
        }

        [$typeId, $setType] = explode("Attributes", $typeRemainder, 2);

        $typeId = strtolower($typeId);

        // TODO: Make this interface inherit from the base attribute set, aka
        // call parent type to obtain fields
        $attrs = $setType ?
            $this->config->getSetAttributes($area, $setType, $typeId) :
            $this->config->getSystemAttributes($area, $typeId);

        if(Mage::getIsDeveloperMode()) {
            // For now, just verify that we are actually getting the proper stuff
            // GraphQL does not verify this properly, make sure we do it
            $this->verifySuperset($attrs, $area, $setType, $typeId);
        }

        return $this->config->createFields($attrs);
    }

    /**
     * Validates that the given fields are a superset of the base product.
     */
    public function verifySuperset(array $attrs, string $area, ?string $setType = null, ?string $typeId = null) {
        if($typeId && $setType) {
            $this->checkSuperset(
                $this->config->getSetAttributes($area, $setType),
                $attrs,
                $this->areaToTypeName($area).$typeId."Attributes".$setType." from ".
                $this->areaToTypeName($area)."Attributes".$setType

            );

            $this->checkSuperset(
                $this->config->getSystemAttributes($area, $typeId),
                $attrs,
                $this->areaToTypeName($area).$typeId."Attributes".$setType." from ".
                $this->areaToTypeName($area).$typeId."Attributes"
            );
        }

        if($typeId || $setType) {
            $this->checkSuperset(
                $this->config->getSystemAttributes($area),
                $attrs,
                $this->areaToTypeName($area).$typeId."Attributes".$setType." from ".
                $this->areaToTypeName($area)."Attributes"
            );
        }
    }

    protected function checkSuperset(array $parent, array $child, string $msg) {
        $diff = array_keys(array_diff_key($parent, $child));

        if(count($diff) > 0) {
            throw new Exception("Missing attributes [%s] in %s", implode(", ", $diff), $msg);
        }
    }

    public function createProductsByFilters(Registry $registry): array {
        $filterable = $this->config->getFilterableAttributes();
        $fields = [];

        foreach($filterable as $key => $attr) {
            $field = $this->field("PaginatedProducts", "List of products filtered by $key")
                    ->addArgument(
                        "pageSize",
                        $this->argument("Int", "Maximum number of products to list")
                            ->setDefaultValue(self::DEFAULT_PAGE_SIZE))
                    ->addArgument(
                        "page",
                        $this->argument("Int", "Which page to show")
                            ->setDefaultValue(1));

            switch($attr["input"]) {
            case "datetime":
                $field->addArgument("from", $this->argument("String", "The start-date, inclusive"));
                $field->addArgument("to", $this->argument("String", "The end-date, inclusive"));

                break;

            case "select":
            case "multiselect":
                $field->addArgument("value", $this->argument("String!", "The value to match against"));

                break;
            default:
                switch($attr["backend_type"]) {
                case "int":
                    $field->addArgument("min", $this->argument("Int", "The minimum value"));
                    $field->addArgument("max", $this->argument("Int", "The maximum value"));

                    break;

                case "decimal":
                    $field->addArgument("min", $this->argument("Float", "The minimum value"));
                    $field->addArgument("max", $this->argument("Float", "The maximum value"));

                    break;

                default:
                    Mage::log(sprintf("%s: Cannot create ProductsByFilter for attribute code %s", __METHOD__, $key));

                    continue 3;
                }
            }

            $field->setResolver("MageQL_Catalog_Model_Product::resolveProductsByFilter");

            $fields[$key] = $field;
        }

        return $fields;
    }

    public function createProductAttributesFileds(Registry $registry): array {
        $filterable = $this->config->getFilterableAttributes();
        $fields = [];

        foreach($filterable as $key => $attr) {
            // TODO: Is this the proper condition?
            if( ! in_array($attr["input"], ["select", "multiselect"])) {
                continue;
            }

            $field = $this->field("ProductAttribute!", "Attribute data for $key");

            $field->setResolver("MageQL_Catalog_Model_Attributes_Product::resolveAttribute");

            $fields[$key] = $field;
        }

        return $fields;
    }

    public function getUnreachableTypes(): array {
        $objectTypes = [
            "ListProduct",
            "ProductDetail"
        ];
        $typeNames = [
            "RouteProduct",
        ];
        $setTypes = Mage::getSingleton("mageql_catalog/attributes_product")->getSetTypes();

        foreach($objectTypes as $objType) {
            // TODO: The empty one should be an interface
            foreach(array_merge(MageQL_Catalog_Model_Attributes_Product::SUPPORTED_TYPE_IDS, [""]) as $typeId) {
                $productType = ucfirst($typeId);

                foreach($setTypes as $set) {
                    $typeNames[] = $objType.$productType."Attributes".$set;
                }
            }
        }

        foreach($objectTypes as $objType) {
            foreach(MageQL_Catalog_Model_Attributes_Product::SUPPORTED_TYPE_IDS as $prodType) {
                $typeNames[] = $objType.ucfirst($prodType);
            }
        }

        return $typeNames;
    }
}
